/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "ccc.h"
#include "ccc_settings.h"
#include <iostream>
#include <algorithm>


// ----------------------------------------------------------------------------
// Defines
// ----------------------------------------------------------------------------
#define MAJOR_VERSION		"0"
#define MINOR_VERSION		"2"
#define REVIS_VERSION		"0"


// ----------------------------------------------------------------------------
// Action implementations
// ----------------------------------------------------------------------------
void PrintVersion()
{
	std::cout << CCC_ConstructorSettings::GetProgramName() << " " << MAJOR_VERSION << "." << MINOR_VERSION << "." << REVIS_VERSION << std::endl;
	std::cout << std::endl;
}

void PrintUsage()
{
	std::cout << "Usage: " << CCC_ConstructorSettings::GetProgramName() << " [OPTION]... FILE..." << std::endl;
	std::cout << "Construct a call graph from the specified FILEs." << std::endl;
	std::cout << std::endl;
	std::cout << "Mandatory arguments to long options are mandatory for short options too.       " << std::endl;
	std::cout << "  -t, --input-type TYPE     Set the type of the input files. Can be 'binary' or" << std::endl;
	std::cout << "                            'xml'. Default is 'binary'.                        " << std::endl;
	std::cout << "  -e, --entry-point NAME    Set the name of the method that should be regarded " << std::endl;
	std::cout << "                            as the entry point. You can specify 'main' as a    " << std::endl;
	std::cout << "                            shorthand for the 'main' function. By default, no  " << std::endl;
	std::cout << "                            entry point is specified and a call graph for the  " << std::endl;
	std::cout << "                            entire system is generated (which may be a forest  " << std::endl;
	std::cout << "                            of multiple unconnected call graphs).              " << std::endl;
	std::cout << "  -f, --output-format FMT   Specifies the format of the generated graph file.  " << std::endl;
	std::cout << "                            Can be 'sqlite', 'tulip' or 'dot'. Default is      " << std::endl;
	std::cout << "                            'sqlite'.                                          " << std::endl;
	std::cout << "  -o, --output-file FILE    Specifies that output should be written to FILE    " << std::endl;
	std::cout << "                            instead of stdout. FILE will be overwritten.       " << std::endl;
	std::cout << "  -d, --validate-input      Check that all called functions are decl'ed/def'ed." << std::endl;
	std::cout << "  -Wl, --Wcnsv-list-cands   Print all call candidates when the set of call     " << std::endl;
	std::cout << "                            candidates is not guaranteed to be conservative.   " << std::endl;
	std::cout << "  -Wo, --Wcnsv-disable-one  Do not print a warning about the set of call       " << std::endl;
	std::cout << "                            candidates not being conservative when there is    " << std::endl;
	std::cout << "                            exactly one call candidate.                        " << std::endl;
	std::cout << "  -Wd, --Wcnsv-disable      Never print warnings about the set of call         " << std::endl;
	std::cout << "                            candidates not being conservative.                 " << std::endl;
	std::cout << "  -i, --include-init-finit  Include calls made by initializers and finalizers  " << std::endl;
	std::cout << "                            in the call graph. These are the calls made before " << std::endl;
	std::cout << "                            and after the call to the 'main' entry point.      " << std::endl;
	std::cout << "  -p, --print-signatures    Print all signatures found in input files and exit." << std::endl;
	std::cout << "  -b, --verbose             Print what is being done.                          " << std::endl;
	std::cout << "  -v, --version             Print version information and exit.                " << std::endl;
	std::cout << "  -h, --help                Print this help message and exit.                  " << std::endl;
	std::cout << std::endl;
}

void PrintSignatures()
{
	StringVector* pInputFileNames = CCC_ConstructorSettings::GetInputFileNames();
	StringVector::iterator itInputFileName = pInputFileNames->begin();

	StringMap signatures;
	while(itInputFileName != pInputFileNames->end())
	{
		LocalFunctionMap localFunctions;
		FunctionCallVector initializingCalls;
		FunctionCallVector finalizingCalls;
		switch(CCC_ConstructorSettings::GetInputType())
		{
			case CCC_IT_Binary:
			{
				CCI_DeserializerBin deserializer(*itInputFileName);
				deserializer.Deserialize(localFunctions, initializingCalls, finalizingCalls);
				break;
			}
			case CCC_IT_Xml:
			{
				CCI_DeserializerXml deserializer(*itInputFileName);
				deserializer.Deserialize(localFunctions, initializingCalls, finalizingCalls);
				break;
			}
		}
		
		// Store all read signatures in the map.
		LocalFunctionMap::iterator itLocalFunction = localFunctions.begin();
		while(itLocalFunction != localFunctions.end())
		{
			std::string szSignature = itLocalFunction->second->GetSignature();
			if(signatures.find(szSignature) == signatures.end())
			{
				// Add the location of the function to the string. If a defintion
				// has been found, add its location. Otherwise, add the location
				// of the declaration.
				if(itLocalFunction->second->HasFunctionDefinition())
				{
					szSignature += " " + itLocalFunction->second->GetFunctionDefinition()->GetSourceLoc()->ToString("");
				}
				else
				{
					szSignature += " " + itLocalFunction->second->GetFunctionDeclaration()->GetSourceLoc()->ToString("");
				}
				signatures[szSignature] = true;
			}
			delete itLocalFunction->second;
			itLocalFunction++;
		}

		itInputFileName++;
	}

	// Print all signatures in the map.
	StringMap::iterator itSignature = signatures.begin();
	while(itSignature != signatures.end())
	{
		std::cout << itSignature->first << std::endl;
		itSignature++;
	}
}

void ConstructCallGraph()
{
	//CCC_ConstructorSettings::Print("");
	CCC_Constructor::ConstructCallGraph();
}


// ----------------------------------------------------------------------------
// Main
// ----------------------------------------------------------------------------
int main(int argc, char** argv)
{
	if(!CCC_ConstructorSettings::Parse(argc, argv))
		return 1;

	switch(CCC_ConstructorSettings::GetAction())
	{
		case CCC_A_PrintVersion:
			PrintVersion();
			break;
		case CCC_A_PrintUsage:
			PrintUsage();
			break;
		case CCC_A_PrintSignatures:
			PrintSignatures();
			break;
		case CCC_A_ConstructCallGraph:
			ConstructCallGraph();
			break;
	}

	return 0;
}
