/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef CCI_CALL_CANDIDATES_H
#define CCI_CALL_CANDIDATES_H


// -----------------------------------------------------------------------------
// Includes
// -----------------------------------------------------------------------------
#include "libcci/cci_typedefs.h"
#include "libcci/cci_printable.h"


// -----------------------------------------------------------------------------
// Class CCI_CallCandidates
// -----------------------------------------------------------------------------
// CCI_CallCandidates assists in determining the call candidates of a function-
// or pointer-signature.
//
// CCI_CallCandidates contains and constructs maps that translate:
// * a supplied function signature to a list of function signatures of functions
//   that override the supplied function.
// * a supplied pointer signature to a list of function signatures of functions
//   that have a pointer signature that matches the supplied pointer signature.
//
// When Resolve() is called, CCI_CallCandidates constructs the mentioned maps
// from the elements currently contained in the LocalFunctionMap that is
// supplied during construction.
//
// You should call Resolve() after you have extracted all your function
// -declarations, -definitions and -calls, since then you only need to call
// Resolve() once.
//
// After Resolve() has been called, you can supply the CCI_CallCandidates object
// in a call to CCI_FunctionCall::GetCallCandidates() to retrieve the actual
// collection of call candidates for that function call.
//
class CCI_CallCandidates : public CCI_Printable
{
	// -------------------------------------------------------------------------
	// Construction & destruction
	// -------------------------------------------------------------------------
public:
	// Constructs and initializes a newe CCI_CallCandidates object. The supplied
	// pointer to the LocalFunctionMap will be stored and its elements will be
	// used by Resolve().
	CCI_CallCandidates();
	~CCI_CallCandidates();


	// -------------------------------------------------------------------------
	// Resolution
	// -------------------------------------------------------------------------
public:
	void Resolve(LocalFunctionMap* pLocalFunctions);

private:
	void ResolveOverriders();
	void ResolvePointerMatches();
	void AddToStringVectorMap(StringVectorMap & stringVectorMap, std::string szKey, std::string szValue);


	// -------------------------------------------------------------------------
	// Cleanup
	// -------------------------------------------------------------------------
public:
	void Cleanup();


	// -------------------------------------------------------------------------
	// Properties
	// -------------------------------------------------------------------------
public:
	StringVectorMap* GetOverridersMap();
	StringVectorMap* GetPointerMatchesMap();
	StringVector* GetOverriders(std::string szSignature);
	StringVector* GetPointerMatches(std::string szPointerSignature);


	// -------------------------------------------------------------------------
	// Call candidate deletion
	// -------------------------------------------------------------------------
public:
	void RemoveFunction(std::string szSignature);

private:
	void RemoveFunctionFromOverriders(std::string szSignature);
	void RemoveFunctionFromPointerMatches(std::string szSignature);
	void RemoveFunctionFromLocalFunctions(std::string szSignature);


	// -------------------------------------------------------------------------
	// Printable implementation
	// -------------------------------------------------------------------------
public:
	virtual std::string ToString(std::string szPrefix);


	// -------------------------------------------------------------------------
	// Attributes
	// -------------------------------------------------------------------------
private:
	// The collection of CCI_LocalFunctions. Each CCI_LocationFunction is
	// accessible via its signature.
	LocalFunctionMap* m_pLocalFunctions;
	// The collection of StringVectors that contain the function signatures that
	// override the function represented by the key. I.e., this maps a
	// function's signature to the signatures of its overriders.
	StringVectorMap m_overriders;
	// The collection of StringVectors that contain the function sigantures that
	// have a pointer signature that matches the key's pointer signature. I.e.,
	// this maps a pointer signature to its possible call candidates.
	StringVectorMap m_pointerMatches;
};


#endif
