/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#ifndef CCI_FUNCTION_CALL_H
#define CCI_FUNCTION_CALL_H


// -----------------------------------------------------------------------------
// Includes
// -----------------------------------------------------------------------------
#include "libcci/cci_locatable.h"
#include "libcci/cci_call_candidates.h"
#include "libcci/cci_typedefs.h"
#include <vector>
#include <map>


// -----------------------------------------------------------------------------
// Enums
// -----------------------------------------------------------------------------
enum CCI_FunctionCallType
{
	// This is a call to a c-style function, e.g.:
	// {
	//     f();
	// }
	CCI_FCT_DirectFunction,
	
	// This is a call to a c++ method, on an object instance, e.g.:
	// {
	//     Object o;
	//     o.m();
	// }
	CCI_FCT_DirectMethod,
	
	// This is a call to a c++ method, on an object pointer, e.g.:
	// {
	//    Object* o;
	//    o->m();
	// }
	CCI_FCT_ObjectPointer,
	
	// This is a call to a c++ method, on an object reference, e.g.:
	// {
	//     Object & o;
	//     o.m();
	// }
	CCI_FCT_ObjectReference,
	
	// This is a call to a c++ constructor, e.g.:
	// {
	//     Object  o1;
	//     Object* o2 = new Object();
	// }
	CCI_FCT_Constructor,
	
	// This is a call to a c++ destructor, e.g.:
	// {
	//     Object* o1;
	//     delete o1;
	//
	//     {
	//         Object o;
	//         // dtor call because o goes out of scope.
	//     }
	// }
	CCI_FCT_Destructor,
	
	// This is a call to a c-style function, through a pointer-to-function.
	// {
	//     void (*f)(int);
	//     f();
	// }
	CCI_FCT_PointerToFunction,
	
	// This is a call to a c++ method, through a pointer-to-method.
	// {
	//     Object o;
	//     void (Object::*m)(int);
	//     (o).*(m)(1);
	// }
	CCI_FCT_PointerToMember
};


// -----------------------------------------------------------------------------
// Class CCI_FunctionCall
// -----------------------------------------------------------------------------
class CCI_FunctionCall : public CCI_Locatable
{
	// -------------------------------------------------------------------------
	// Construction & destruction
	// -------------------------------------------------------------------------
public:
	// Constructs a CCI_FunctionCall with the specified properties.
	CCI_FunctionCall(CCI_FunctionCallType type, CCI_SourceLoc* pSourceLoc, std::string szSignature);


	// -------------------------------------------------------------------------
	// Properties
	// -------------------------------------------------------------------------
public:
	// Returns what type of function call this is.
	CCI_FunctionCallType GetFunctionCallType();

	// Convenience methods for the function call type 'm_type'.
	bool IsType(CCI_FunctionCallType type);
	bool IsTypeDirectFunction();
	bool IsTypeDirectMethod();
	bool IsTypeObjectPointer();
	bool IsTypeObjectReference();
	bool IsTypeConstructor();
	bool IsTypeDestructor();
	bool IsTypePointerToFunction();
	bool IsTypePointerToMember();

	// Returns true if this is a 'direct' call. I.e., either a call to a c-style
	// function, or a call to a method on an (non-pointer and non-reference)
	// object. Therefore, there can be exactly zero or one call candidates.
	// This returns true if either of the following returns true:
	// * IsTypeDirectFunction()
	// * IsTypeDirectMethod()
	// * IsTypeConstructor()
	// * IsTypeDestructor()
	bool IsTypeDirect();
	
	// Returns true if this is a call on an object pointer or an object
	// reference. If this call is to a virtual method, then this call
	// potentially has more than one call candidate. Therefore, this call will
	// have zero or more call candidates.
	// This returns true if either of the following returns true:
	// * IsTypeObjectPointer()
	// * IsTypeObjectReference()
	bool IsTypeObject();
	
	// Returns true if this is a call through a pointer-to-function or a
	// pointer-to-member. Since many functions can match the "pointer signature"
	// of this call, this call potentially has more than one call candidate.
	// Therefore, this call will have zero or more call candidates.
	// This returns true if either of the following returns true:
	// * IsTypePointerToFunction()
	// * IsTypePointerToMember()
	bool IsTypePointer();
	
	// Returns true if this is a call to a C-style function. This is the inverse
	// of IsTypeCpp().
	bool IsTypeC();
	// Returns true if this is a call to a C++ method. This is the inverse of
	// IsTypeC().
	bool IsTypeCpp();

	// Returns the signature of this function call. If either IsTypeDirect() or
	// IsTypeObject() returns true, then the signature will be a plain function
	// signature (i.e., the signature will explicitly contain the name of the
	// function that it refers to, next to its return and parameters types).
	// Otherwise, if IsTypePointer() returns true, then the signature will be a
	// pointer signature (i.e., the signature will not explicitly contain the
	// name of the function that it refers to, just its return and parameter
	// types. This causes all functions that match this pointer signature to be
	// a call candidate).
	std::string GetSignature();

	// Sets the signature of this call. This method is used when mangling the
	// names of functions that have static linkage (in that case, all references
	// to those functions need to be mangled as well, hence this method).
	// DO NOT USE THIS METHOD IN YOUR OWN CODE!
	void SetSignature(std::string szSignature);


	// -------------------------------------------------------------------------
	// Call candidates
	// -------------------------------------------------------------------------
public:
	void GetCallCandidates(CCI_CallCandidates & callCandidates, StringVector & callCandidateSignatures);


	// -------------------------------------------------------------------------
	// Equality
	// -------------------------------------------------------------------------
public:
	// Returns whether this function call is equal to the other function call.
	// Two function calls are considered equal iff the following are equal
	// according to the == operator:
	// * m_type, and
	// * The CCI_Locatable bases, and
	// * m_szSignature
	bool operator== (CCI_FunctionCall & other);
	bool operator!= (CCI_FunctionCall & other);


	// -------------------------------------------------------------------------
	// Printable implementation
	// -------------------------------------------------------------------------
public:
	virtual std::string ToString(std::string szPrefix);
	std::string GetFunctionTypeString();


	// -------------------------------------------------------------------------
	// State
	// -------------------------------------------------------------------------
protected:
	// Contains what type of function call this is.
	CCI_FunctionCallType m_type;

	// Contains the (plain or pointer) signature of the called function or
	// method.
	std::string m_szSignature;
};


#endif
