/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#ifndef CCI_LOCAL_FUNCTION_H
#define CCI_LOCAL_FUNCTION_H


// -----------------------------------------------------------------------------
// Includes
// -----------------------------------------------------------------------------
#include "libcci/cci_printable.h"
#include "libcci/cci_function_declaration.h"
#include "libcci/cci_function_definition.h"
#include "libcci/cci_typedefs.h"


// -----------------------------------------------------------------------------
// Enums
// -----------------------------------------------------------------------------
enum CCI_LocalFunctionFlag
{
	// No flags. A C-style function that does not have static linkage.
	CCI_LFF_NONE           = 0x00,

	// Indicates that this function has static linkage.
	CCI_LFF_STATIC_LINKAGE = 0x01,

	// Indicates that this function is a method (i.e., not a c-style function).
	CCI_LFF_METHOD         = 0x02,

	// Indicates that this function is a virtual method.
	CCI_LFF_VIRTUAL        = 0x04,

	// Indicate whether this is a static method.
	CCI_LFF_STATIC         = 0x08,

	// Indicates that this function has the public access modifier.
	CCI_LFF_PUBLIC         = 0x10,

	// Indicates that this function has the protected access modifier.
	CCI_LFF_PROTECTED      = 0x20,

	// Indicates that this function has the private access modifier.
	CCI_LFF_PRIVATE        = 0x40,

	// Indicates that this function is declared inline.
	CCI_LFF_INLINE         = 0x80
};


// -----------------------------------------------------------------------------
// class CCI_LocalFunction
// -----------------------------------------------------------------------------
class CCI_LocalFunction : public CCI_Printable
{
	// -------------------------------------------------------------------------
	// Construction & destruction
	// -------------------------------------------------------------------------
public:
	// Constructs a CCI_LocalFunction with the specified properties.
	CCI_LocalFunction(std::string szName,
					  std::string szSignature,
	                  std::string szPointerSignature,
					  std::string szClassName,
	                  CCI_FunctionDeclaration* pFunctionDeclaration,
					  CCI_FunctionDefinition* pFunctionDefinition,
					  int iFlags,
					  /*bool bIsFlagStaticLinkage,
					  bool bIsFlagMethod,
					  bool bIsFlagVirtual,*/
					  StringVector* pOverriddenMethodSignatures);
	~CCI_LocalFunction();


	// -------------------------------------------------------------------------
	// Properties
	// -------------------------------------------------------------------------
public:
	
	// Returns the name of this function. The name only contains the bare name
	// of the function. It does not contain the return and parameter types.
	// Therefore, the returned name does not uniquely identify the function.
	std::string GetName();

	// Returns the signature of this function. The signature contains the
	// explicit name of this function and its return and parameter types.
	std::string GetSignature();

	// Sets the signature of this function. This method is used when mangling
	// the names of functions that have static linkage.
	// DO NOT USE THIS METHOD IN YOUR OWN CODE!
	void SetSignature(std::string szSignature);

	// Returns the pointer signature of this function. The pointer signature
	// does not contain the explicit name of this function, but does contain
	// its return and parameter types. This causes the pointer signature to
	// match any function with the same return and parameter types (and, in the
	// case of a pointer to a method, any method of the same class with the same
	// return and parameters types).
	std::string GetPointerSignature();

	// Returns whether this function has a class name.
	bool HasClassName();

	// If this function is a class-method, the fully qualified name of the class
	// is returned. Otherwise, an empty string is returned.
	std::string GetClassName();

	// Returns whether this function has a definition. If false, then
	// GetFunctionDefinition will return NULL.
	bool HasFunctionDefinition();

	// Returns the function declaration of this function. Every function is
	// guaranteed to have a valid CCI_FunctionDeclaration.
	CCI_FunctionDeclaration* GetFunctionDeclaration();

	// Return the function definition of this function, if it exists. If
	// HasFunctionDefinition() returns false, this returns NULL.
	CCI_FunctionDefinition* GetFunctionDefinition();

	// Returns the flags of this function.
	int GetFlags();

	// Convenience methods for the function's flags 'm_iFlags'.
	bool IsFlag(CCI_LocalFunctionFlag flag);
	bool IsFlagStaticLinkage();
	bool IsFlagMethod();
	bool IsFlagVirtual();
	bool IsFlagStatic();
	bool IsFlagPublic();
	bool IsFlagProtected();
	bool IsFlagPrivate();
	bool IsFlagInline();

	// Static convenience methods for the function's flags 'm_iFlags'.
	static bool IsFlag(int iFlags, CCI_LocalFunctionFlag flag);
	static bool IsFlagStaticLinkage(int iFlags);
	static bool IsFlagMethod(int iFlags);
	static bool IsFlagVirtual(int iFlags);
	static bool IsFlagStatic(int iFlags);
	static bool IsFlagPublic(int iFlags);
	static bool IsFlagProtected(int iFlags);
	static bool IsFlagPrivate(int iFlags);
	static bool IsFlagInline(int iFlags);

	// Returns a pointer to the vector containing the signatures of the methods
	// that this method overrides. If this method does not override any methods,
	// then this returns a pointer to an empty vector.
	StringVector* GetOverriddenMethodSignatures();


	// -------------------------------------------------------------------------
	// Equality
	// -------------------------------------------------------------------------
public:
	// Returns whether this function is equal to the other function. Two
	// functions are considered equal iff the following are equal according to
	// the == operator:
	// * m_szSignature, and
	// * The dereferenced value of m_pFunctionDeclaration, and
	// * The dereferenced value of m_pFunctionDefinition (if available).
	bool operator== (CCI_LocalFunction & other);
	bool operator!= (CCI_LocalFunction & other);


	// -------------------------------------------------------------------------
	// Printable implementation
	// -------------------------------------------------------------------------
public:
	virtual std::string ToString(std::string szPrefix);


	// -------------------------------------------------------------------------
	// Attributes
	// -------------------------------------------------------------------------
protected:
	// The name of the function, mainly for display purposes.
	std::string m_szName;
	// The signature of the function, uniquely identifying it program-wide.
	std::string m_szSignature;
	// The signature of the function, identifying what it needs to be called via
	// a pointer-to-function or pointer-to-member.
	std::string m_szPointerSignature;
	// If IsFlagMethod is true, then this is set to the fully qualified name of
	// the method's class.
	std::string m_szClassName;
	
	// Pointer to either the first declaration, or the declaration corresponding
	// to the definition, of this function.
	CCI_FunctionDeclaration* m_pFunctionDeclaration;
	// Pointer to the definition of this function. If no definition exists,
	// this is NULL.
	CCI_FunctionDefinition* m_pFunctionDefinition;

	// Integer containing the CCI_LocalFunctionFlags.
 	int m_iFlags;

	// Holds the signatures of all base-methods this method overrides.
	StringVector m_overriddenMethodSignatures;
};


#endif
