/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// -----------------------------------------------------------------------------
// Includes
// -----------------------------------------------------------------------------
#include "libcci/cci_call_candidates.h"
#include "libcci/cci_local_function.h"
#include <algorithm>
#include <sstream>
#include <assert.h>


// -----------------------------------------------------------------------------
// Construction & destruction
// -----------------------------------------------------------------------------
CCI_CallCandidates::CCI_CallCandidates()
{
	m_pLocalFunctions = NULL;
}

CCI_CallCandidates::~CCI_CallCandidates()
{
	Cleanup();
}


// -----------------------------------------------------------------------------
// Resolution
// -----------------------------------------------------------------------------
void CCI_CallCandidates::Resolve(LocalFunctionMap* pLocalFunctions)
{
	assert(pLocalFunctions != NULL);
	m_pLocalFunctions = pLocalFunctions;
	
	Cleanup();
	ResolveOverriders();
	ResolvePointerMatches();
}

void CCI_CallCandidates::ResolveOverriders()
{
	//
	// This maps all signatures of overridden methods to the signatures of their
	// overriders.
	//
	LocalFunctionMap::iterator itLocalFunction = m_pLocalFunctions->begin();
	while(itLocalFunction != m_pLocalFunctions->end())
	{
		StringVector* pOverriddenMethodSignatures = itLocalFunction->second->GetOverriddenMethodSignatures();
		StringVector::iterator itOverriddenMethodSignature = pOverriddenMethodSignatures->begin();
		while(itOverriddenMethodSignature != pOverriddenMethodSignatures->end())
		{
			std::string szOverriddenMethodSignature = *itOverriddenMethodSignature;
			std::string szOverridingMethodSignature = itLocalFunction->second->GetSignature();

			AddToStringVectorMap(m_overriders, szOverriddenMethodSignature, szOverridingMethodSignature);

			itOverriddenMethodSignature++;
		}
		itLocalFunction++;
	}
}

void CCI_CallCandidates::ResolvePointerMatches()
{
	//
	// This maps all pointer signatures to the signatures of the functions that
	// are call candidates of that pointer signature.
	//
	LocalFunctionMap::iterator itLocalFunction = m_pLocalFunctions->begin();
	while(itLocalFunction != m_pLocalFunctions->end())
	{
		std::string szPointerSignature = itLocalFunction->second->GetPointerSignature();
		std::string szSignature        = itLocalFunction->second->GetSignature();

		AddToStringVectorMap(m_pointerMatches, szPointerSignature, szSignature);

		itLocalFunction++;
	}
}

void CCI_CallCandidates::AddToStringVectorMap(StringVectorMap & stringVectorMap, std::string szKey, std::string szValue)
{
	if(stringVectorMap.find(szKey) == stringVectorMap.end())
		stringVectorMap[szKey] = new StringVector;

	stringVectorMap[szKey]->push_back(szValue);
}


// -----------------------------------------------------------------------------
// Cleanup
// -----------------------------------------------------------------------------
void CCI_CallCandidates::Cleanup()
{
	// Delete the StringVectors in the m_overriders map.
	StringVectorMap::iterator itOverriders = m_overriders.begin();
	while(itOverriders != m_overriders.end())
	{
		delete itOverriders->second;
		itOverriders++;
	}
	m_overriders.clear();

	// Delete the StringVectors in the m_pointerMatches map.
	StringVectorMap::iterator itPointerMatches = m_pointerMatches.begin();
	while(itPointerMatches != m_pointerMatches.end())
	{
		delete itPointerMatches->second;
		itPointerMatches++;
	}
	m_pointerMatches.clear();
}


// -----------------------------------------------------------------------------
// Properties
// -----------------------------------------------------------------------------
StringVectorMap* CCI_CallCandidates::GetOverridersMap()
{
	return &m_overriders;
}

StringVectorMap* CCI_CallCandidates::GetPointerMatchesMap()
{
	return &m_pointerMatches;
}

StringVector* CCI_CallCandidates::GetOverriders(std::string szSignature)
{
	StringVectorMap::iterator itOverriders = m_overriders.find(szSignature);
	if(itOverriders == m_overriders.end())
		return NULL;

	return itOverriders->second;
}

StringVector* CCI_CallCandidates::GetPointerMatches(std::string szPointerSignature)
{
	StringVectorMap::iterator itPointerMatches = m_pointerMatches.find(szPointerSignature);
	if(itPointerMatches == m_pointerMatches.end())
		return NULL;

	return itPointerMatches->second;
}


// -----------------------------------------------------------------------------
// Call candidate deletion
// -----------------------------------------------------------------------------
void CCI_CallCandidates::RemoveFunction(std::string szSignature)
{
	RemoveFunctionFromOverriders(szSignature);
	RemoveFunctionFromPointerMatches(szSignature);
	RemoveFunctionFromLocalFunctions(szSignature);
}

void CCI_CallCandidates::RemoveFunctionFromOverriders(std::string szSignature)
{
	// The function identified by the specified signature needs to be removed
	// from the system. This means it needs to be removed from the following:
	//
	// 1. This function overrides a number of functions and it maintains a list
	//    of which functions that are. We are going to use that list to find the
	//    entries in the "m_overriders" mapping that map the overridden
	//    functions to a list of functions that override it. The supplied
	//    function is an element of that list and it needs to be removed from
	//    it.
	//
	// 2. The "m_overriders" mapping maps this function to a list of functions
	//    that all override this function. Each of these functions maintains a
	//    list of functions that they override. This function needs to be
	//    removed from those lists.

	// -------------------------------------------------------------------------
	// 1. Remove this function from the "overriders" list of each function that
	//    this function overrides.
	// -------------------------------------------------------------------------
	// Find the function in the local function map.
	LocalFunctionMap::iterator itLocalFunction = m_pLocalFunctions->find(szSignature);
	if(itLocalFunction != m_pLocalFunctions->end())
	{
		// Retrieve the CCI_LocalFunction that belongs to this function.
		CCI_LocalFunction* pLocalFunction = itLocalFunction->second;

		// Retrieve the list of the signatures of the functions that this function overrides.
		StringVector* pOverriddenMethodSignatures = pLocalFunction->GetOverriddenMethodSignatures();

		// Iterate over each overridden function's signature to remove this function
		// from the entry in the overriders map corresponding to the overridden function.
		StringVector::iterator itOverriddenMethodSignature = pOverriddenMethodSignatures->begin();
		while(itOverriddenMethodSignature != pOverriddenMethodSignatures->end())
		{
			// Use the signature of the overridden method to retrieve its list of
			// methods that override it. Remove this function from that list.
			StringVectorMap::iterator itOverriders = m_overriders.find(*itOverriddenMethodSignature);
			if(itOverriders != m_overriders.end())
			{
				StringVector* pOverriders = itOverriders->second;
				StringVector::iterator itThisOverrider = std::find(pOverriders->begin(), pOverriders->end(), szSignature);
				if(itThisOverrider != pOverriders->end())
				{
					// Remove this function from the overridden function's list of overriders.
					pOverriders->erase(itThisOverrider);
				}
			}
			itOverriddenMethodSignature++;
		}
	}

	// -------------------------------------------------------------------------
	// 2. Remove this function from the "overrides" list of each function that
	//    overrides this function. This function can no longer be overridden by
	//    any function.
	// -------------------------------------------------------------------------
	// Retrieve the list of signatures of functions that override this function.
	StringVectorMap::iterator itOverriders = m_overriders.find(szSignature);
	if(itOverriders != m_overriders.end())
	{
		// Iterate over each signature of the functions that override this function,
		// retrieve their corresponding CCI_LocalFunction and remove this function
		// from its list of functions that it overrides.
		StringVector* pOverriders = itOverriders->second;
		StringVector::iterator itOverrider = pOverriders->begin();
		while(itOverrider != pOverriders->end())
		{
			LocalFunctionMap::iterator itOverriderLocalFunction = m_pLocalFunctions->find(*itOverrider);
			if(itOverriderLocalFunction != m_pLocalFunctions->end())
			{
				CCI_LocalFunction* pOverriderLocalFunction = itOverriderLocalFunction->second;
				
				// Now retrieve the overrider's list of functions that it overrides
				// and remove this function from that list.
				StringVector* pOverriddenMethodSignatures = pOverriderLocalFunction->GetOverriddenMethodSignatures();
				StringVector::iterator itThisOverriddenMethod = std::find(pOverriddenMethodSignatures->begin(), pOverriddenMethodSignatures->end(), szSignature);
				if(itThisOverriddenMethod != pOverriddenMethodSignatures->end())
				{
					pOverriddenMethodSignatures->erase(itThisOverriddenMethod);
				}
			}
			itOverrider++;
		}

		// Lastly, delete the StringVector which contained the signatures of the
		// functions that override this function and then remove the entry in
		// the overriders map for this function the altogether.
		delete pOverriders; // pOverriders == itOverriders->second
		m_overriders.erase(itOverriders);
	}
}

void CCI_CallCandidates::RemoveFunctionFromPointerMatches(std::string szSignature)
{
	// Find the function in the local function map.
	LocalFunctionMap::iterator itLocalFunction = m_pLocalFunctions->find(szSignature);
	if(itLocalFunction != m_pLocalFunctions->end())
	{
		// Retrieve the CCI_LocalFunction that belongs to this function.
		CCI_LocalFunction* pLocalFunction = itLocalFunction->second;

		// Retrieve the pointer signature of this function.
		std::string szPointerSignature = pLocalFunction->GetPointerSignature();
		
		// Retrieve the list of the signatures of the functions that are
		// callable through the pointer signature of this function. Then, remove
		// this function from that list.
		StringVectorMap::iterator itPointerMatches = m_pointerMatches.find(szPointerSignature);
		if(itPointerMatches != m_pointerMatches.end())
		{
			// Find this function in the list of matching functions.
			StringVector* pPointerMatches = itPointerMatches->second;
			StringVector::iterator itThisPointerMatch = std::find(pPointerMatches->begin(), pPointerMatches->end(), szSignature);
			if(itThisPointerMatch != pPointerMatches->end())
			{
				// Delete this function from the list of matching functions.
				pPointerMatches->erase(itThisPointerMatch);
			}
			
			// If this was the only signature in the list then the entire list
			// can be deleted.
			if(pPointerMatches->size() == 0)
			{
				delete pPointerMatches; // pPointerMatches == itPointerMatches->second
				m_pointerMatches.erase(itPointerMatches);
			}
		}
	}
}

void CCI_CallCandidates::RemoveFunctionFromLocalFunctions(std::string szSignature)
{
	// Delete the CCI_LocalFunction corresponding to this function's signature
	// and remove this function's entry from the LocalFunctionMap.
	LocalFunctionMap::iterator itLocalFunction = m_pLocalFunctions->find(szSignature);
	if(itLocalFunction != m_pLocalFunctions->end())
	{
		delete itLocalFunction->second;
		m_pLocalFunctions->erase(itLocalFunction);
	}
}


// -----------------------------------------------------------------------------
// Printable implementation
// -----------------------------------------------------------------------------
std::string CCI_CallCandidates::ToString(std::string szPrefix)
{
	std::stringstream ss;
	
	ss << szPrefix << "CallCandidates" << std::endl;
	ss << szPrefix << "{" << std::endl;


	ss << szPrefix << "\tOverriders" << std::endl;
	ss << szPrefix << "\t{" << std::endl;
	StringVectorMap::iterator itOverriders = m_overriders.begin();
	while(itOverriders != m_overriders.end())
	{
		ss << szPrefix << "\t\t" << itOverriders->first << std::endl;
		ss << szPrefix << "\t\t{" << std::endl;
		StringVector::iterator itOverrider = itOverriders->second->begin();
		while(itOverrider != itOverriders->second->end())
		{
			ss << szPrefix << "\t\t\t" << *itOverrider << std::endl;
			itOverrider++;
		}
		ss << szPrefix << "\t\t}" << std::endl;
		itOverriders++;
	}
	ss << szPrefix << "\t}" << std::endl;


	ss << szPrefix << "\tPointerMatches" << std::endl;
	ss << szPrefix << "\t{" << std::endl;
	StringVectorMap::iterator itPointerMatches = m_pointerMatches.begin();
	while(itPointerMatches != m_pointerMatches.end())
	{
		ss << szPrefix << "\t\t" << itPointerMatches->first << std::endl;
		ss << szPrefix << "\t\t{" << std::endl;
		StringVector::iterator itPointerMatch = itPointerMatches->second->begin();
		while(itPointerMatch != itPointerMatches->second->end())
		{
			ss << szPrefix << "\t\t\t" << *itPointerMatch << std::endl;
			itPointerMatch++;
		}
		ss << szPrefix << "\t\t}" << std::endl;
		itPointerMatches++;
	}
	ss << szPrefix << "\t}" << std::endl;


	ss << szPrefix << "}" << std::endl;

	return ss.str();
}
