/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// -----------------------------------------------------------------------------
// Includes
// -----------------------------------------------------------------------------
#include "libcci/cci_deserializer_bin.h"
#include "libcci/cci_defines.h"


// -----------------------------------------------------------------------------
// Construction & destruction
// -----------------------------------------------------------------------------
CCI_DeserializerBin::CCI_DeserializerBin() : CCI_Deserializer()
{
}

CCI_DeserializerBin::CCI_DeserializerBin(std::string szFileName) : CCI_Deserializer(szFileName)
{
}


// -----------------------------------------------------------------------------
// Output stream
// -----------------------------------------------------------------------------
std::ios::openmode CCI_DeserializerBin::GetStreamOpenMode()
{
	return CCI_Deserializer::GetStreamOpenMode() | std::ios::binary;
}


// -----------------------------------------------------------------------------
// Deserialization
// -----------------------------------------------------------------------------
bool CCI_DeserializerBin::ReadCallInfo(LocalFunctionMap & localFunctions, FunctionCallVector & initializingCalls, FunctionCallVector & finalizingCalls)
{
	CCI_TRY(ReadLocalFunctions(localFunctions));
	CCI_TRY(ReadInitializingCalls(initializingCalls));
	CCI_TRY(ReadFinalizingCalls(finalizingCalls));

	return true;
}

bool CCI_DeserializerBin::ReadLocalFunctions(LocalFunctionMap & localFunctions)
{
	// Read the number of local functions.
	int iLocalFunctionCount;
	CCI_TRY(ReadInt(iLocalFunctionCount));

	// Read the local functions.
	for(int i = 0; i < iLocalFunctionCount; i++)
	{
		// Read a local function.
		CCI_LocalFunction* pLocalFunction;
		CCI_CLEANUP_MAP_ON_FAIL(ReadLocalFunction(pLocalFunction), localFunctions, LocalFunctionMap);

		// Retrieve the signature of the local function and validate that a
		// local function with that signature does not yet exist in the map.
		std::string szSignature = pLocalFunction->GetSignature();
		if(localFunctions.find(szSignature) != localFunctions.end())
		{
			delete pLocalFunction;
			CCI_CLEANUP_MAP(localFunctions, LocalFunctionMap);
			return false;
		}

		// Add the local function to the map.
		localFunctions[szSignature] = pLocalFunction;
	}

	return true;
}

bool CCI_DeserializerBin::ReadInitializingCalls(FunctionCallVector & initializingCalls)
{
	// Read the number of initializing calls.
	int iCallCount;
	CCI_TRY(ReadInt(iCallCount));

	// Read the initializing calls.
	for(int i = 0; i < iCallCount; i++)
	{
		CCI_FunctionCall* pFunctionCall;
		CCI_TRY(ReadFunctionCall(pFunctionCall));

		initializingCalls.push_back(pFunctionCall);
	}

	return true;
}

bool CCI_DeserializerBin::ReadFinalizingCalls(FunctionCallVector & finalizingCalls)
{
	// Read the number of finalizing calls.
	int iCallCount;
	CCI_TRY(ReadInt(iCallCount));

	// Read the finalizing calls.
	for(int i = 0; i < iCallCount; i++)
	{
		CCI_FunctionCall* pFunctionCall;
		CCI_TRY(ReadFunctionCall(pFunctionCall));

		finalizingCalls.push_back(pFunctionCall);
	}

	return true;
}

bool CCI_DeserializerBin::ReadLocalFunction(CCI_LocalFunction* & pLocalFunction)
{
	// Read the function name.
	std::string szName;
	CCI_TRY(ReadString(szName));
	
	// Read the function signature.
	std::string szSignature;
	CCI_TRY(ReadString(szSignature));

	// Read the function pointer signature.
	std::string szPointerSignature;
	CCI_TRY(ReadString(szPointerSignature));
	
	// Read the class name of the function.
	std::string szClassName;
	CCI_TRY(ReadString(szClassName));

	// Read the function declaration.
	CCI_FunctionDeclaration* pFunctionDeclaration;
	CCI_TRY(ReadFunctionDeclaration(pFunctionDeclaration));

	// Read the function definition.
	CCI_FunctionDefinition* pFunctionDefinition;
	CCI_DELETE_ON_FAIL(ReadFunctionDefinition(pFunctionDefinition), pFunctionDeclaration);

	// Read the function flags.
	int iFlags;
	CCI_DELETE_ON_FAIL2(ReadInt(iFlags), pFunctionDeclaration, pFunctionDefinition);

	// If this is a method and it is virtual, then read the methods it overrides.
	StringVector overriddenMethodSignatures;
	if(CCI_LocalFunction::IsFlagMethod(iFlags) && CCI_LocalFunction::IsFlagVirtual(iFlags))
	{
		// Read the number of methods this method overrides.
		int iOverriddenMethodSignatureCount;
		CCI_DELETE_ON_FAIL2(ReadInt(iOverriddenMethodSignatureCount), pFunctionDeclaration, pFunctionDefinition);

		// Read the signatures of the methods this methods overrides.
		for(int i = 0; i < iOverriddenMethodSignatureCount; i++)
		{
			std::string szOverriddenMethodSignature;
			CCI_DELETE_ON_FAIL2(ReadString(szOverriddenMethodSignature), pFunctionDeclaration, pFunctionDefinition);

			// Read the signature and add it to the list of overridden methods.
			overriddenMethodSignatures.push_back(szOverriddenMethodSignature);
		}
	}

	// All reading succeeded, so create a new CCI_LocalFunction.
	pLocalFunction = new CCI_LocalFunction
	(
		szName,
		szSignature,
		szPointerSignature,
		szClassName,
		pFunctionDeclaration,
		pFunctionDefinition,
		iFlags,
  		/*CCI_LocalFunction::IsFlagStaticLinkage(iFlags),
		CCI_LocalFunction::IsFlagMethod(iFlags),
		CCI_LocalFunction::IsFlagVirtual(iFlags),*/
		&overriddenMethodSignatures
	);
	CCI_DELETE_ON_FAIL2(pLocalFunction != NULL, pFunctionDeclaration, pFunctionDefinition);

	return true;
}

bool CCI_DeserializerBin::ReadFunctionDeclaration(CCI_FunctionDeclaration* & pFunctionDeclaration)
{
	// Read the CCI_SourceLoc of the function declaration.
	CCI_SourceLoc* pSourceLoc;
	CCI_TRY(ReadSourceLoc(pSourceLoc));

	// All reading succeeded, so create a new CCI_FunctionDeclaration.
	pFunctionDeclaration = new CCI_FunctionDeclaration(pSourceLoc);
	CCI_DELETE_ON_FAIL(pFunctionDeclaration != NULL, pSourceLoc);

	return true;
}

bool CCI_DeserializerBin::ReadFunctionDefinition(CCI_FunctionDefinition* & pFunctionDefinition)
{
	// Assume failure or no function definition.
	pFunctionDefinition = NULL;

	// Read whether the function definition exists.
	bool bIsFunctionDefinition;
	CCI_TRY(ReadBool(bIsFunctionDefinition));

	// If there is a function definition, read it.
	if(bIsFunctionDefinition)
	{
		// Read the CCI_SourceLoc of the function definition
		CCI_SourceLoc* pSourceLoc;
		CCI_TRY(ReadSourceLoc(pSourceLoc));

		// Read the number of function calls that are made in this function definition.
		int iFunctionCallCount;
		CCI_TRY(ReadInt(iFunctionCallCount));

		// Read the function calls that are made in this function definition.
		FunctionCallVector functionCalls;
		for(int i = 0; i < iFunctionCallCount; i++)
		{
			CCI_FunctionCall* pFunctionCall;
			CCI_CLEANUP_VECTOR_AND_DELETE_ON_FAIL(ReadFunctionCall(pFunctionCall), functionCalls, FunctionCallVector, pSourceLoc);
			functionCalls.push_back(pFunctionCall);
		}

		// All reading succeeded, so create a new CCI_FunctionDefinition.
		pFunctionDefinition = new CCI_FunctionDefinition(pSourceLoc, &functionCalls);
		CCI_CLEANUP_VECTOR_AND_DELETE_ON_FAIL(pFunctionDefinition != NULL, functionCalls, FunctionCallVector, pSourceLoc);
	}

	return true;
}

bool CCI_DeserializerBin::ReadFunctionCall(CCI_FunctionCall* & pFunctionCall)
{
	// Read the function type.
	int iType;
	CCI_TRY(ReadInt(iType));

	// Read the FunctionCall's CCI_SourceLoc.
	CCI_SourceLoc* pSourceLoc;
	CCI_TRY(ReadSourceLoc(pSourceLoc));

	// Read the signature of the called function.
	std::string szSignature;
	CCI_DELETE_ON_FAIL(ReadString(szSignature), pSourceLoc);

	// All reading succeeded, so create a new CCI_FunctionCall.
	pFunctionCall = new CCI_FunctionCall((CCI_FunctionCallType)iType, pSourceLoc, szSignature);
	CCI_DELETE_ON_FAIL(pFunctionCall != NULL, pSourceLoc);

	return true;
}

bool CCI_DeserializerBin::ReadSourceLoc(CCI_SourceLoc* & pSourceLoc)
{
	// Read the line number.
	int iLine;
	CCI_TRY(ReadInt(iLine));

	// Read the column number.
	int iColumn;
	CCI_TRY(ReadInt(iColumn));

	// Read the filename.
	std::string szFileName;
	CCI_TRY(ReadString(szFileName));

	// Read the translation unit filename.
	std::string szTranslationUnit;
	CCI_TRY(ReadString(szTranslationUnit));

	// All reading succeeded, so create and return a new CCI_LocalFunction.
	pSourceLoc = new CCI_SourceLoc(szFileName, iLine, iColumn, szTranslationUnit);
	CCI_TRY(pSourceLoc != NULL);

	return true;
}
