/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// -----------------------------------------------------------------------------
// Includes
// -----------------------------------------------------------------------------
#include "libcci/cci_function_call.h"
#include "libcci/cci_function_definition.h"
#include "libcci/cci_sourceloc.h"
#include <iostream>
#include <sstream>
#include <assert.h>


// -----------------------------------------------------------------------------
// Construction
// -----------------------------------------------------------------------------
CCI_FunctionCall::CCI_FunctionCall(CCI_FunctionCallType type,
                                   CCI_SourceLoc* pSourceLoc,
                                   std::string szSignature) :
	m_type(type),
	CCI_Locatable(pSourceLoc),
	m_szSignature(szSignature)
{
}


// -----------------------------------------------------------------------------
// Properties
// -----------------------------------------------------------------------------
CCI_FunctionCallType CCI_FunctionCall::GetFunctionCallType()
{
	return m_type;
}

bool CCI_FunctionCall::IsType(CCI_FunctionCallType type)
{
	return m_type == type;
}

bool CCI_FunctionCall::IsTypeDirectFunction()
{
	return IsType(CCI_FCT_DirectFunction);
}

bool CCI_FunctionCall::IsTypeDirectMethod()
{
	return IsType(CCI_FCT_DirectMethod);
}

bool CCI_FunctionCall::IsTypeObjectPointer()
{
	return IsType(CCI_FCT_ObjectPointer);
}

bool CCI_FunctionCall::IsTypeObjectReference()
{
	return IsType(CCI_FCT_ObjectReference);
}

bool CCI_FunctionCall::IsTypeConstructor()
{
	return IsType(CCI_FCT_Constructor);
}

bool CCI_FunctionCall::IsTypeDestructor()
{
	return IsType(CCI_FCT_Constructor);
}

bool CCI_FunctionCall::IsTypePointerToFunction()
{
	return IsType(CCI_FCT_PointerToFunction);
}

bool CCI_FunctionCall::IsTypePointerToMember()
{
	return IsType(CCI_FCT_PointerToMember);
}

bool CCI_FunctionCall::IsTypeDirect()
{
	return IsTypeDirectFunction() || IsTypeDirectMethod() || IsTypeConstructor() || IsTypeDestructor();
}

bool CCI_FunctionCall::IsTypeObject()
{
	return IsTypeObjectPointer() || IsTypeObjectReference();
}

bool CCI_FunctionCall::IsTypePointer()
{
	return IsTypePointerToFunction() || IsTypePointerToMember();
}

bool CCI_FunctionCall::IsTypeC()
{
	return IsTypeDirectFunction() || IsTypePointerToFunction();
}

bool CCI_FunctionCall::IsTypeCpp()
{
	return !IsTypeC();
}

std::string CCI_FunctionCall::GetSignature()
{
	return m_szSignature;
}

void CCI_FunctionCall::SetSignature(std::string szSignature)
{
	m_szSignature = szSignature;
}


// -----------------------------------------------------------------------------
// Call candidates
// -----------------------------------------------------------------------------
void CCI_FunctionCall::GetCallCandidates(CCI_CallCandidates & callCandidates, StringVector & callCandidateSignatures)
{
	// Make sure the vector of call candidate signatures is empty.
	callCandidateSignatures.clear();

	// Compose the set of call candidates, depending on what type of function
	// call this is.
	if(IsTypeDirect())
	{
		// There is only one call candidate, since this is a 'direct' call.
		callCandidateSignatures.push_back(m_szSignature);
	}
	else if(IsTypeObject())
	{
		// The function with the signature of this call is a call candidate.
		callCandidateSignatures.push_back(m_szSignature);

		// Also, all functions that override the function with the signature of
		// this call are a call candidate.
		StringVector* pOverriders = callCandidates.GetOverriders(m_szSignature);
		if(pOverriders != NULL)
			callCandidateSignatures.insert(callCandidateSignatures.end(), pOverriders->begin(), pOverriders->end());
	}
	else if(IsTypePointer())
	{
		// All functions that have a matching pointer signature are a call
		// candidate. If a matching function is a method and virtual, then all
		// methods that override it are also a call candidate.
		StringVector* pPointerMatches = callCandidates.GetPointerMatches(m_szSignature);

		if (IsTypePointerToMember())
		{
			// All functions that have a matching pointer signature are a call
			// candidate, but also all functions that override these functions
			// are call candidates.
			StringVector::iterator itPointerMatch = pPointerMatches->begin();
			while (itPointerMatch != pPointerMatches->end())
			{
				// The function with this signature is a call candidate.
				callCandidateSignatures.push_back(*itPointerMatch);

				// Also, all functions that override the function with this signature
				// are a call candidate.
				StringVector* pOverriders = callCandidates.GetOverriders(*itPointerMatch);
				if(pOverriders != NULL)
					callCandidateSignatures.insert(callCandidateSignatures.end(), pOverriders->begin(), pOverriders->end());

				// Advance to the next signature.
				itPointerMatch++;
			}
		}
		else
		{
			// All functions that have a matching pointer signature are a call
			// candidate.
			StringVector* pPointerMatches = callCandidates.GetPointerMatches(m_szSignature);
			if(pPointerMatches != NULL)
				callCandidateSignatures.insert(callCandidateSignatures.end(), pPointerMatches->begin(), pPointerMatches->end());
		}
	}
}


// -----------------------------------------------------------------------------
// Equality
// -----------------------------------------------------------------------------
bool CCI_FunctionCall::operator== (CCI_FunctionCall & other)
{
	return (m_type                  == other.m_type       ) &&
	       (m_szSignature           == other.m_szSignature) &&
	       (*((CCI_Locatable*)this) == other              );
}

bool CCI_FunctionCall::operator!= (CCI_FunctionCall & other)
{
	return (m_type                  != other.m_type       ) ||
	       (m_szSignature           != other.m_szSignature) ||
	       (*((CCI_Locatable*)this) != other              );
}


// -----------------------------------------------------------------------------
// Printable implementation
// -----------------------------------------------------------------------------
std::string CCI_FunctionCall::ToString(std::string szPrefix)
{
	std::stringstream ss;

	ss << szPrefix << "Function call"                               << std::endl;
	ss << szPrefix << "{"                                           << std::endl;
	ss << szPrefix << "\tLocation: " << CCI_Locatable::ToString("") << std::endl;
	ss << szPrefix << "\tCallee  : " << m_szSignature               << std::endl;
	ss << szPrefix << "\tType    : " << GetFunctionTypeString()     << std::endl;
	ss << szPrefix << "}"                                           << std::endl;

	return ss.str();
}

std::string CCI_FunctionCall::GetFunctionTypeString()
{
	switch(m_type)
	{
		case CCI_FCT_DirectFunction:
			return "DirectFunction";
		case CCI_FCT_DirectMethod:
			return "DirectMethod";
		case CCI_FCT_ObjectPointer:
			return "ObjectPointer";
		case CCI_FCT_ObjectReference:
			return "ObjectReference";
		case CCI_FCT_Constructor:
			return "Constructor";
		case CCI_FCT_Destructor:
			return "Destructor";
		case CCI_FCT_PointerToFunction:
			return "PointerToFunction";
		case CCI_FCT_PointerToMember:
			return "PointerToMember";
	}

	return "Unknown";
}
