/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// -----------------------------------------------------------------------------
// Includes
// -----------------------------------------------------------------------------
#include "libcci/cci_local_function.h"
#include <sstream>
#include <assert.h>


// -----------------------------------------------------------------------------
// Construction
// -----------------------------------------------------------------------------
CCI_LocalFunction::CCI_LocalFunction(std::string szName,
                                     std::string szSignature,
                                     std::string szPointerSignature,
									 std::string szClassName,
                                     CCI_FunctionDeclaration* pFunctionDeclaration,
                                     CCI_FunctionDefinition* pFunctionDefinition,
									 int iFlags,
								     /*bool bIsFlagStaticLinkage,
								     bool bIsFlagMethod,
								     bool bIsFlagVirtual,*/
								     StringVector* pOverriddenMethodSignatures) :
	m_szName(szName),
	m_szSignature(szSignature),
	m_szPointerSignature(szPointerSignature),
	m_szClassName(szClassName),
	m_pFunctionDeclaration(pFunctionDeclaration),
	m_pFunctionDefinition(pFunctionDefinition),
	m_iFlags(iFlags),
	/*(
	 	(bIsFlagStaticLinkage ? CCI_LFF_STATIC_LINKAGE : 0) |
		(bIsFlagMethod        ? CCI_LFF_METHOD         : 0) |
		(bIsFlagVirtual       ? CCI_LFF_VIRTUAL        : 0)
	),*/
	m_overriddenMethodSignatures(*pOverriddenMethodSignatures)
{
	assert(pFunctionDeclaration != NULL);
}

CCI_LocalFunction::~CCI_LocalFunction()
{
	// Delete the function declaration.
	if(m_pFunctionDeclaration != NULL)
	{
		delete m_pFunctionDeclaration;
		m_pFunctionDeclaration = NULL;
	}

	// Delete the function definition.
	if(m_pFunctionDefinition != NULL)
	{
		delete m_pFunctionDefinition;
		m_pFunctionDefinition = NULL;
	}
}


// -----------------------------------------------------------------------------
// Properties
// -----------------------------------------------------------------------------
std::string CCI_LocalFunction::GetName()
{
	return m_szName;
}

std::string CCI_LocalFunction::GetSignature()
{
	return m_szSignature;
}

void CCI_LocalFunction::SetSignature(std::string szSignature)
{
	m_szSignature = szSignature;
}

std::string CCI_LocalFunction::GetPointerSignature()
{
	return m_szPointerSignature;
}

bool CCI_LocalFunction::HasClassName()
{
	return m_szClassName.length() > 0;
}

std::string CCI_LocalFunction::GetClassName()
{
	return m_szClassName;
}

bool CCI_LocalFunction::HasFunctionDefinition()
{
	return m_pFunctionDefinition != NULL;
}

CCI_FunctionDeclaration* CCI_LocalFunction::GetFunctionDeclaration()
{
	return m_pFunctionDeclaration;
}

CCI_FunctionDefinition* CCI_LocalFunction::GetFunctionDefinition()
{
	return m_pFunctionDefinition;
}

int CCI_LocalFunction::GetFlags()
{
	return m_iFlags;
}

bool CCI_LocalFunction::IsFlag(CCI_LocalFunctionFlag flag)
{
	return IsFlag(m_iFlags, flag);
}

bool CCI_LocalFunction::IsFlagStaticLinkage()
{
	return IsFlag(CCI_LFF_STATIC_LINKAGE);
}

bool CCI_LocalFunction::IsFlagMethod()
{
	return IsFlag(CCI_LFF_METHOD);
}

bool CCI_LocalFunction::IsFlagVirtual()
{
	return IsFlag(CCI_LFF_VIRTUAL);
}

bool CCI_LocalFunction::IsFlagStatic()
{
	return IsFlag(CCI_LFF_STATIC);
}

bool CCI_LocalFunction::IsFlagPublic()
{
	return IsFlag(CCI_LFF_PUBLIC);
}

bool CCI_LocalFunction::IsFlagProtected()
{
	return IsFlag(CCI_LFF_PROTECTED);
}

bool CCI_LocalFunction::IsFlagPrivate()
{
	return IsFlag(CCI_LFF_PRIVATE);
}

bool CCI_LocalFunction::IsFlagInline()
{
	return IsFlag(CCI_LFF_INLINE);
}

bool CCI_LocalFunction::IsFlag(int iFlags, CCI_LocalFunctionFlag flag)
{
	return (iFlags & flag) == flag;
}

bool CCI_LocalFunction::IsFlagStaticLinkage(int iFlags)
{
	return IsFlag(iFlags, CCI_LFF_STATIC_LINKAGE);
}

bool CCI_LocalFunction::IsFlagMethod(int iFlags)
{
	return IsFlag(iFlags, CCI_LFF_METHOD);
}

bool CCI_LocalFunction::IsFlagVirtual(int iFlags)
{
	return IsFlag(iFlags, CCI_LFF_VIRTUAL);
}

bool CCI_LocalFunction::IsFlagStatic(int iFlags)
{
	return IsFlag(iFlags, CCI_LFF_STATIC);
}

bool CCI_LocalFunction::IsFlagPublic(int iFlags)
{
	return IsFlag(iFlags, CCI_LFF_PUBLIC);
}

bool CCI_LocalFunction::IsFlagProtected(int iFlags)
{
	return IsFlag(iFlags, CCI_LFF_PROTECTED);
}

bool CCI_LocalFunction::IsFlagPrivate(int iFlags)
{
	return IsFlag(iFlags, CCI_LFF_PRIVATE);
}

bool CCI_LocalFunction::IsFlagInline(int iFlags)
{
	return IsFlag(iFlags, CCI_LFF_INLINE);
}

StringVector* CCI_LocalFunction::GetOverriddenMethodSignatures()
{
	return &m_overriddenMethodSignatures;
}


// -----------------------------------------------------------------------------
// Equality
// -----------------------------------------------------------------------------
bool CCI_LocalFunction::operator== (CCI_LocalFunction & other)
{
	return ( m_szSignature          ==  other.m_szSignature         ) &&
	       (*m_pFunctionDeclaration == *other.m_pFunctionDeclaration) &&
	       (
	           (m_pFunctionDefinition == NULL && other.m_pFunctionDefinition == NULL) ||
	           (m_pFunctionDefinition != NULL && other.m_pFunctionDefinition != NULL && *m_pFunctionDefinition == *other.m_pFunctionDefinition)
	       );
}

bool CCI_LocalFunction::operator!= (CCI_LocalFunction & other)
{
	return !(*this == other);
}


// -----------------------------------------------------------------------------
// Debugging
// -----------------------------------------------------------------------------

std::string CCI_LocalFunction::ToString(std::string szPrefix)
{
	std::stringstream ss;

	ss << szPrefix << "Local function"                                                    << std::endl;
	ss << szPrefix << "{"                                                                 << std::endl;

	// Print the function name.
	ss << szPrefix << "\tName               : " << m_szName                               << std::endl;
	
	// Print the function signature.
	ss << szPrefix << "\tSignature          : " << m_szSignature                          << std::endl;

	// Print the function call signature.
	ss << szPrefix << "\tPointer signature  : " << m_szPointerSignature                   << std::endl;

	// Print the values of the flags.
	ss << szPrefix << "\tIsFlagStaticLinkage: " << (IsFlagStaticLinkage() ? "Yes" : "No") << std::endl;
	ss << szPrefix << "\tIsFlagMethod       : " << (IsFlagMethod()        ? "Yes" : "No") << std::endl;
	ss << szPrefix << "\tIsFlagVirtual      : " << (IsFlagVirtual()       ? "Yes" : "No") << std::endl;
	ss << szPrefix << "\tIsFlagPublic       : " << (IsFlagPublic()        ? "Yes" : "No") << std::endl;
	ss << szPrefix << "\tIsFlagProtected    : " << (IsFlagProtected()     ? "Yes" : "No") << std::endl;
	ss << szPrefix << "\tIsFlagPrivate      : " << (IsFlagPrivate()       ? "Yes" : "No") << std::endl;
	ss << szPrefix << "\tIsFlagInline       : " << (IsFlagInline()        ? "Yes" : "No") << std::endl;

	// Print the methods that this method overrides.
	StringVector::iterator it = m_overriddenMethodSignatures.begin();
	while(it != m_overriddenMethodSignatures.end())
	{
		ss << szPrefix << "\tOverrides          : " << *it                                << std::endl;
		it++;
	}

	// Print the function declaration.
	if(m_pFunctionDeclaration != NULL)
	{
		ss << m_pFunctionDeclaration->ToString(szPrefix + "\t");
	}

	// Print the function definition.
	if(m_pFunctionDefinition != NULL)
	{
		ss << m_pFunctionDefinition->ToString(szPrefix + "\t");
	}

	ss << szPrefix << "}"                                                                 << std::endl;

	return ss.str();
}
