/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// -----------------------------------------------------------------------------
// Includes
// -----------------------------------------------------------------------------
#include "libcci/cci_serializer.h"
#include "libcci/cci_defines.h"
#include <iostream>
#include <assert.h>


// -----------------------------------------------------------------------------
// Construction & destruction
// -----------------------------------------------------------------------------
CCI_Serializer::CCI_Serializer() : m_os(std::cout)
{
	m_bUseFileStream = false;
	m_szFileName = "";
}

CCI_Serializer::CCI_Serializer(std::string szFileName) : m_os(m_ofs)
{
	m_bUseFileStream = true;
	m_szFileName = szFileName;
}


// -----------------------------------------------------------------------------
// Output stream
// -----------------------------------------------------------------------------
std::ios::openmode CCI_Serializer::GetStreamOpenMode()
{
	return std::ios::out | std::ios::trunc;
}


// -----------------------------------------------------------------------------
// Serialization
// -----------------------------------------------------------------------------
bool CCI_Serializer::Serialize(LocalFunctionMap* pLocalFunctions, FunctionCallVector* pInitializingCalls, FunctionCallVector* pFinalizingCalls)
{
	assert(pLocalFunctions    != NULL);
	assert(pInitializingCalls != NULL);
	assert(pFinalizingCalls   != NULL);

	if(m_bUseFileStream)
	{
		m_ofs.open(m_szFileName.c_str(), GetStreamOpenMode());
		if(!m_ofs.is_open())
		{
			std::cerr << "Error opening file '" << m_szFileName << "' for writing." << std::endl;
			return false;
		}
	}
	
	bool bResult = WriteCallInfo(pLocalFunctions, pInitializingCalls, pFinalizingCalls);

	if(m_bUseFileStream)
	{
		if(m_ofs.is_open())
			m_ofs.close();
	}

	return bResult;
}


bool CCI_Serializer::WriteInt(int iValue)
{
	return Write((void*)&iValue, sizeof(iValue));
}

bool CCI_Serializer::WriteBool(bool bValue)
{
	return Write((void*)&bValue, sizeof(bValue));
}

bool CCI_Serializer::WriteString(std::string szValue)
{
	// Write the length of the string.
	CCI_TRY(WriteInt(szValue.length()));

	// Write the string.
	CCI_TRY(Write((void*)szValue.c_str(), szValue.length()));

	return true;
}

bool CCI_Serializer::Write(void* pBuffer, int iLength)
{
	m_os.write((const char*)pBuffer, iLength);

	return m_os.good();
}

bool CCI_Serializer::Write(const void* pBuffer, int iLength)
{
	return Write((void*)pBuffer, iLength);
}
