/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// -----------------------------------------------------------------------------
// Includes
// -----------------------------------------------------------------------------
#include "libcci/cci_serializer_bin.h"
#include "libcci/cci_defines.h"


// -----------------------------------------------------------------------------
// Construction & destruction
// -----------------------------------------------------------------------------
CCI_SerializerBin::CCI_SerializerBin() : CCI_Serializer()
{
}

CCI_SerializerBin::CCI_SerializerBin(std::string szFileName) : CCI_Serializer(szFileName)
{
}


// -----------------------------------------------------------------------------
// Output stream
// -----------------------------------------------------------------------------
std::ios::openmode CCI_SerializerBin::GetStreamOpenMode()
{
	return CCI_Serializer::GetStreamOpenMode() | std::ios::binary;
}


// -----------------------------------------------------------------------------
// Serialization
// -----------------------------------------------------------------------------
bool CCI_SerializerBin::WriteCallInfo(LocalFunctionMap* pLocalFunctions, FunctionCallVector* pInitializingCalls, FunctionCallVector* pFinalizingCalls)
{
	CCI_TRY(WriteLocalFunctions(pLocalFunctions));
	CCI_TRY(WriteInitializingCalls(pInitializingCalls));
	CCI_TRY(WriteFinalizingCalls(pFinalizingCalls));

	return true;
}

bool CCI_SerializerBin::WriteLocalFunctions(LocalFunctionMap* pLocalFunctions)
{
	// Write the number of local functions.
	CCI_TRY(WriteInt(pLocalFunctions->size()));

	// Write the local functions.
	LocalFunctionMap::iterator itLocalFunction = pLocalFunctions->begin();
	while(itLocalFunction != pLocalFunctions->end())
	{
		CCI_TRY(WriteLocalFunction(itLocalFunction->second));
		itLocalFunction++;
	}

	return true;
}

bool CCI_SerializerBin::WriteInitializingCalls(FunctionCallVector* pInitializingCalls)
{
	// Write the number of initializing calls.
	CCI_TRY(WriteInt(pInitializingCalls->size()));

	// Write the initializing calls.
	FunctionCallVector::iterator itInitializingCall = pInitializingCalls->begin();
	while(itInitializingCall != pInitializingCalls->end())
	{
		CCI_TRY(WriteFunctionCall(*itInitializingCall));
		itInitializingCall++;
	}

	return true;
}

bool CCI_SerializerBin::WriteFinalizingCalls(FunctionCallVector* pFinalizingCalls)
{
	// Write the number of finalizing calls.
	CCI_TRY(WriteInt(pFinalizingCalls->size()));

	// Write the finalizing calls.
	FunctionCallVector::iterator itFinalizingCall = pFinalizingCalls->begin();
	while(itFinalizingCall != pFinalizingCalls->end())
	{
		CCI_TRY(WriteFunctionCall(*itFinalizingCall));
		itFinalizingCall++;
	}

	return true;
}

bool CCI_SerializerBin::WriteLocalFunction(CCI_LocalFunction* pLocalFunction)
{
	// Write the function name.
	CCI_TRY(WriteString(pLocalFunction->GetName()));

	// Write the function signature.
	CCI_TRY(WriteString(pLocalFunction->GetSignature()));

	// Write the function pointer signature.
	CCI_TRY(WriteString(pLocalFunction->GetPointerSignature()));

	// Write the function class name.
	CCI_TRY(WriteString(pLocalFunction->GetClassName()));
	
	// Write the function declaration.
	CCI_TRY(WriteFunctionDeclaration(pLocalFunction->GetFunctionDeclaration()));

	// Write the function definition.
	CCI_TRY(WriteFunctionDefinition(pLocalFunction->GetFunctionDefinition()));

	// Write the function flags.
	int iFlags = pLocalFunction->GetFlags();
	CCI_TRY(WriteInt(iFlags));

	// If this is a method and it is virtual, write the number of methods this
	// method overrides.
	if(CCI_LocalFunction::IsFlagMethod(iFlags) && CCI_LocalFunction::IsFlagVirtual(iFlags))
	{
		StringVector* pOverriddenMethodSignatures = pLocalFunction->GetOverriddenMethodSignatures();
		// Write the number of methods this method overrides.
		CCI_TRY(WriteInt(pOverriddenMethodSignatures->size()));

		// Write the signatures of the methods this methods overrides.
		StringVector::iterator itOverriddenMethodSignature = pOverriddenMethodSignatures->begin();
		while(itOverriddenMethodSignature != pOverriddenMethodSignatures->end())
		{
			CCI_TRY(WriteString(*itOverriddenMethodSignature));
			itOverriddenMethodSignature++;
		}
	}
	return true;
}

bool CCI_SerializerBin::WriteFunctionDeclaration(CCI_FunctionDeclaration* pFunctionDeclaration)
{
	// Write the CCI_SourceLoc of the function declaration.
	CCI_TRY(WriteSourceLoc(pFunctionDeclaration->GetSourceLoc()));

	return true;
}

bool CCI_SerializerBin::WriteFunctionDefinition(CCI_FunctionDefinition* pFunctionDefinition)
{
	// Write whether there is a CCI_FunctionDefinition.
	CCI_TRY(WriteBool(pFunctionDefinition != NULL));
	if(pFunctionDefinition == NULL)
		return true;

	// Write the CCI_SourceLoc of the function definition.
	CCI_TRY(WriteSourceLoc(pFunctionDefinition->GetSourceLoc()));

	FunctionCallVector* pFunctionCalls = pFunctionDefinition->GetFunctionCalls();
	// Write the number of function calls that are made in this function definition.
	CCI_TRY(WriteInt(pFunctionCalls->size()));

	// Write the function calls that are made in this function definition.
	FunctionCallVector::iterator itFunctionCall = pFunctionCalls->begin();
	while(itFunctionCall != pFunctionCalls->end())
	{
		CCI_TRY(WriteFunctionCall(*itFunctionCall));
		itFunctionCall++;
	}

	return true;
}

bool CCI_SerializerBin::WriteFunctionCall(CCI_FunctionCall* pFunctionCall)
{
	// Write the function call type.
	CCI_TRY(WriteInt((int)pFunctionCall->GetFunctionCallType()));

	// Write the FunctionCall's CCI_SourceLoc.
	CCI_TRY(WriteSourceLoc(pFunctionCall->GetSourceLoc()));

	// Write the signature of the called function.
	CCI_TRY(WriteString(pFunctionCall->GetSignature()));

	return true;
}

bool CCI_SerializerBin::WriteSourceLoc(CCI_SourceLoc* pSourceLoc)
{
	// Write the line number.
	CCI_TRY(WriteInt(pSourceLoc->GetLine()));

	// Write  the column number.
	CCI_TRY(WriteInt(pSourceLoc->GetColumn()));

	// Write  the filename.
	CCI_TRY(WriteString(pSourceLoc->GetFileName()));

	// Write the translation unit filename.
	CCI_TRY(WriteString(pSourceLoc->GetTranslationUnit()));

	return true;
}
