/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
 
 
// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "ccie_call_validator.h"
#include "ccie_call_visitor.h"
#include <libcci/cci.h>


// ----------------------------------------------------------------------------
// Static accessor for convenience
// ----------------------------------------------------------------------------
void CCIE_CallValidator::ValidateAvailability(/*CCIE_TranslationUnitInfo & tuInfo*/)
{
	CCIE_CallValidator validator/*(tuInfo)*/;
	validator._ValidateAvailability();
}

void CCIE_CallValidator::ValidateCompleteness(/*CCIE_TranslationUnitInfo & tuInfo*/)
{
	CCIE_CallValidator validator/*(tuInfo)*/;
	validator._ValidateCompleteness();
}

void CCIE_CallValidator::ValidateIntegrity(/*CCIE_TranslationUnitInfo & tuInfo*/)
{
	CCIE_CallValidator validator/*(tuInfo)*/;
	validator._ValidateIntegrity();
}


// ----------------------------------------------------------------------------
// Construction & destruction
// ----------------------------------------------------------------------------
CCIE_CallValidator::CCIE_CallValidator(/*CCIE_TranslationUnitInfo & tuInfo*/) /*:
	m_tuInfo(tuInfo)*/
{
}

CCIE_CallValidator::~CCIE_CallValidator()
{
	// Delete all remaining actual function calls.
	FunctionCallVector::iterator itFunctionCall = m_actualFunctionCalls.begin();
	while(itFunctionCall != m_actualFunctionCalls.end())
	{
		delete *itFunctionCall;
		itFunctionCall++;
	}
	m_actualFunctionCalls.clear();
}


// ----------------------------------------------------------------------------
// Availability validation
// ----------------------------------------------------------------------------
void CCIE_CallValidator::_ValidateAvailability()
{
	// Validate that each function that is called has an entry in the local
	// function repository.
	LocalFunctionMap* pLocalFunctions = CCIE_TranslationUnitInfo::m_localFunctionRepository.GetFunctions();
	LocalFunctionMap::iterator itLocalFunction = pLocalFunctions->begin();
	while(itLocalFunction != pLocalFunctions->end())
	{
		ValidateLocalFunction(itLocalFunction->second);
		itLocalFunction++;
	}
	
	// Validate the calls in the list of initializers.
	FunctionCallVector::iterator itInitializingCall = CCIE_TranslationUnitInfo::m_initializingCalls.begin();
	while(itInitializingCall != CCIE_TranslationUnitInfo::m_initializingCalls.end())
	{
		ValidateFunctionCall(*itInitializingCall);
		itInitializingCall++;
	}

	// Validate the calls in the list of finalizers.
	FunctionCallVector::iterator itFinalizingCall = CCIE_TranslationUnitInfo::m_finalizingCalls.begin();
	while(itFinalizingCall != CCIE_TranslationUnitInfo::m_finalizingCalls.end())
	{
		ValidateFunctionCall(*itFinalizingCall);
		itFinalizingCall++;
	}
}

void CCIE_CallValidator::ValidateLocalFunction(CCI_LocalFunction* pLocalFunction)
{
	cci_assert(pLocalFunction != NULL);
	
	if(pLocalFunction->HasFunctionDefinition())
	{
		CCI_FunctionDefinition* pFunctionDefinition = pLocalFunction->GetFunctionDefinition();
		FunctionCallVector* pFunctionCalls = pFunctionDefinition->GetFunctionCalls();

		FunctionCallVector::iterator itFunctionCall = pFunctionCalls->begin();
		while(itFunctionCall != pFunctionCalls->end())
		{
			ValidateFunctionCall(*itFunctionCall);
			itFunctionCall++;
		}
	}
}

void CCIE_CallValidator::ValidateFunctionCall(CCI_FunctionCall* pFunctionCall)
{
	cci_assert(pFunctionCall != NULL);

	if(!pFunctionCall->IsTypePointer())
	{
		std::string signature = pFunctionCall->GetSignature();

		// Validate that the signature of this function call can be found in
		// the local function repository.
		if(!CCIE_TranslationUnitInfo::m_localFunctionRepository.ContainsFunction(signature))
			ReportFunctionNotFound(pFunctionCall);
	}
}

void CCIE_CallValidator::ReportFunctionNotFound(CCI_FunctionCall* pFunctionCall)
{
	cci_assert(pFunctionCall != NULL);

	StringVector builtinFunctions;
	builtinFunctions.push_back("__builtin_");

	StringVector::iterator itBuiltinFunction = builtinFunctions.begin();
	while(itBuiltinFunction != builtinFunctions.end())
	{
		std::string szSignature = pFunctionCall->GetSignature();
		if(szSignature.find(*itBuiltinFunction) != std::string::npos)
			return;
		itBuiltinFunction++;
	}

	std::cerr << "VALIDATION ERROR: Function call without corresponding declaration." << std::endl;
	std::cerr << "{" << std::endl;
	std::cerr << pFunctionCall->ToString("\t");
	std::cerr << "}" << std::endl;
}


// ----------------------------------------------------------------------------
// Completeness validation
// ----------------------------------------------------------------------------
void CCIE_CallValidator::_ValidateCompleteness()
{
	// First, count how many function calls there actually are in this
	// translation unit.
	int iActualFunctionCallCount = CountActualFunctionCalls();

	// Second, count how many function calls we found.
	int iFoundFunctionCallCount = CountFoundFunctionCalls();

	// Finally, compare the numbers of function calls.
	if(iFoundFunctionCallCount < iActualFunctionCallCount)
	{
		// We did not find all function calls.
		std::cerr << "VALIDATION ERROR: We only found " << iFoundFunctionCallCount << " of the " << iActualFunctionCallCount << " function calls in this translation unit." << std::endl;
		PrintMissingFunctionCalls();
	}
	else if(iFoundFunctionCallCount > iActualFunctionCallCount)
	{
		// We found more function calls than actually exist?
		std::cerr << "VALIDATION ERROR: Somehow we found " << iFoundFunctionCallCount << " of the " << iActualFunctionCallCount << " function calls in this translation unit." << std::endl;
	}
	else
	{
		std::cerr << "VALIDATION SUCCESSFUL: We found " <<  iFoundFunctionCallCount << " of the " << iActualFunctionCallCount << " function calls in this translation unit." << std::endl;
		// Good...
	}
}

int CCIE_CallValidator::CountActualFunctionCalls()
{
	m_actualFunctionCalls.clear();
	CCIE_CallVisitor callVisitor(&m_actualFunctionCalls);
	callVisitor.Traverse(CCIE_TranslationUnitInfo::m_pTranslationUnit);
	return m_actualFunctionCalls.size();
}

int CCIE_CallValidator::CountFoundFunctionCalls()
{
	int iFunctionCallCount = 0;
	
	LocalFunctionMap* pLocalFunctions = CCIE_TranslationUnitInfo::m_localFunctionRepository.GetFunctions();
	LocalFunctionMap::iterator itLocalFunction = pLocalFunctions->begin();
	while(itLocalFunction != pLocalFunctions->end())
	{
		CCI_LocalFunction* pLocalFunction = itLocalFunction->second;
		iFunctionCallCount += pLocalFunction->HasFunctionDefinition() ?
		                      pLocalFunction->GetFunctionDefinition()->GetFunctionCalls()->size() :
		                      0;

		itLocalFunction++;
	}

	return iFunctionCallCount + CCIE_TranslationUnitInfo::m_initializingCalls.size() + CCIE_TranslationUnitInfo::m_finalizingCalls.size();
}

void CCIE_CallValidator::PrintMissingFunctionCalls()
{
	// Throw out all actual function calls that we found in function definitions.
	LocalFunctionMap* pLocalFunctions = CCIE_TranslationUnitInfo::m_localFunctionRepository.GetFunctions();
	LocalFunctionMap::iterator itLocalFunction = pLocalFunctions->begin();
	while(itLocalFunction != pLocalFunctions->end())
	{
		CCI_LocalFunction* pLocalFunction = itLocalFunction->second;
		if(pLocalFunction->HasFunctionDefinition())
		{
			FunctionCallVector* pFunctionCalls = pLocalFunction->GetFunctionDefinition()->GetFunctionCalls();
			FunctionCallVector::iterator itFunctionCall = pFunctionCalls->begin();
			while(itFunctionCall != pFunctionCalls->end())
			{
				DeleteFunctionCall(*itFunctionCall);
				itFunctionCall++;
			}
		}

		itLocalFunction++;
	}

	// Throw out all actual function calls that we found in static initialization.
	FunctionCallVector::iterator itFunctionCall = CCIE_TranslationUnitInfo::m_initializingCalls.begin();
	while(itFunctionCall != CCIE_TranslationUnitInfo::m_initializingCalls.end())
	{
		DeleteFunctionCall(*itFunctionCall);
		itFunctionCall++;
	}

	// Throw out all actual function calls that we found in static finalization.
	itFunctionCall = CCIE_TranslationUnitInfo::m_finalizingCalls.begin();
	while(itFunctionCall != CCIE_TranslationUnitInfo::m_finalizingCalls.end())
	{
		DeleteFunctionCall(*itFunctionCall);
		itFunctionCall++;
	}

	// Now we are left with all actual function calls that we did not find, so
	// print them.
	std::cerr << "The following function calls were not found:" << std::endl;
	std::cerr << "{" << std::endl;
	itFunctionCall = m_actualFunctionCalls.begin();
	while(itFunctionCall != m_actualFunctionCalls.end())
	{
		std::cerr << (*itFunctionCall)->ToString("\t");
		itFunctionCall++;
	}
	std::cerr << "}" << std::endl;
}

void CCIE_CallValidator::DeleteFunctionCall(CCI_FunctionCall* pFunctionCall)
{
	FunctionCallVector::iterator itFunctionCall = m_actualFunctionCalls.begin();
	while(itFunctionCall != m_actualFunctionCalls.end())
	{
		CCI_FunctionCall* pActualFunctionCall = *itFunctionCall;
		if(*pActualFunctionCall == *pFunctionCall)
		{
			delete pActualFunctionCall;
			m_actualFunctionCalls.erase(itFunctionCall);
			return;
		}

		itFunctionCall++;
	}
}


// ----------------------------------------------------------------------------
// Completeness validation
// ----------------------------------------------------------------------------
void CCIE_CallValidator::_ValidateIntegrity()
{
	// Validate that all functions in the function repository have a signature
	// that is equal to the key under which they are published.
	LocalFunctionMap* pLocalFunctions = CCIE_TranslationUnitInfo::m_localFunctionRepository.GetFunctions();
	LocalFunctionMap::iterator itLocalFunction = pLocalFunctions->begin();
	while(itLocalFunction != pLocalFunctions->end())
	{
		if(itLocalFunction->first != itLocalFunction->second->GetSignature())
		{
			std::cerr << "VALIDATION ERROR: Key/signature mismatch:" << std::endl;
			std::cerr << "{" << std::endl;
			std::cerr << itLocalFunction->second->ToString("\t") << std::endl;
			std::cerr << "}" << std::endl;
		}

		itLocalFunction++;
	}	
}
