/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "ccie_cmd.h"
#include "oink_util.h"  // USER_ERROR_ExitCode


// ----------------------------------------------------------------------------
// Initialization of static members
// ----------------------------------------------------------------------------
StringVector CCIE_Cmd::m_filterDefinitionsKeepSignatures;
std::string CCIE_Cmd::m_szOutputFileName         = "";
CCIE_Cmd::CCIE_OutputType CCIE_Cmd::m_outputType = CCIE_OT_Binary;
bool CCIE_Cmd::m_bFilterDeclarations             = true;
bool CCIE_Cmd::m_bFilterDefinitions              = true;
bool CCIE_Cmd::m_bFilterDefinitionsConservative  = false;
bool CCIE_Cmd::m_bPrintCallInformation           = false;
bool CCIE_Cmd::m_bPrintConservativenessWarnings  = true;
bool CCIE_Cmd::m_bVerbose                        = false;
bool CCIE_Cmd::m_bPrintDebugInformation          = false;


// ----------------------------------------------------------------------------
// OinkCmd overrides
// ----------------------------------------------------------------------------
void CCIE_Cmd::readOneArg(int &argc, char **&argv)
{
	// First, see if Oink needs to parse this argument. If so, we don't need to.
	int old_argc = argc;
	OinkCmd::readOneArg(argc, argv);
	if(old_argc != argc)
		return;

	// If Oink did not parse anything, than this was not an Oink argument, so
	// try to parse it.
	char* pszArg = argv[0];


	// Flags that require arguments.
	if(streq(pszArg, "-c-output"))
	{
		shift(argc, argv);
		m_szOutputFileName = shift(argc, argv);
	}
	else if(streq(pszArg, "-c-output-type"))
	{
		shift(argc, argv);
		sm::string szNext = shift(argc, argv);
		if(streq(szNext, "binary"))
			m_outputType = CCIE_OT_Binary;
		else if(streq(szNext, "xml"))
			m_outputType = CCIE_OT_Xml;
		else
			throw UserError(USER_ERROR_ExitCode, stringc << "Unrecognized output type '" << szNext << "'. Please try --help for valid values.");
	}
	else if(streq(pszArg, "-c-filter-defn-keep"))
	{
		shift(argc, argv);
		std::string szSignature = shift(argc, argv);
		m_filterDefinitionsKeepSignatures.push_back(szSignature);
	}

	// Filter related boolean flags.
	else if(streq(pszArg, "-fc-filter-decl"))
	{
		shift(argc, argv);
		m_bFilterDeclarations = true;
	}
	else if(streq(pszArg, "-fc-no-filter-decl"))
	{
		shift(argc, argv);
		m_bFilterDeclarations = false;
	}
	else if(streq(pszArg, "-fc-filter-defn"))
	{
		shift(argc, argv);
		m_bFilterDefinitions = true;
	}
	else if(streq(pszArg, "-fc-no-filter-defn"))
	{
		shift(argc, argv);
		m_bFilterDefinitions = false;
	}
	else if(streq(pszArg, "-fc-filter-defn-conservative"))
	{
		shift(argc, argv);
		m_bFilterDefinitionsConservative = true;
	}
	else if(streq(pszArg, "-fc-no-filter-defn-conservative"))
	{
		shift(argc, argv);
		m_bFilterDefinitionsConservative = false;
	}

	// Printing related boolean flags.
	else if(streq(pszArg, "-fc-print-call-information"))
	{
		shift(argc, argv);
		m_bPrintCallInformation = true;
	}
	else if(streq(pszArg, "-fc-no-print-call-information"))
	{
		shift(argc, argv);
		m_bPrintCallInformation = false;
	}
	else if(streq(pszArg, "-fc-print-consv-warnings"))
	{
		shift(argc, argv);
		m_bPrintConservativenessWarnings = true;
	}
	else if(streq(pszArg, "-fc-no-print-consv-warnings"))
	{
		shift(argc, argv);
		m_bPrintConservativenessWarnings = false;
	}

	// Debug printing related boolean flags.
	else if(streq(pszArg, "-fc-print-debug-information"))
	{
		shift(argc, argv);
		m_bPrintDebugInformation = true;
	}
	else if(streq(pszArg, "-fc-no-print-debug-information"))
	{
		shift(argc, argv);
		m_bPrintDebugInformation = false;
	}

	// Verboseness related boolean flags.
	else if(streq(pszArg, "-fc-verbose"))
	{
		shift(argc, argv);
		m_bVerbose = true;
	}
	else if(streq(pszArg, "-fc-no-verbose"))
	{
		shift(argc, argv);
		m_bVerbose = false;
	}
}

void CCIE_Cmd::dump()
{
	OinkCmd::dump();

	std::cout << "c-output: " << m_szOutputFileName                                       << std::endl;
	std::cout << "c-output-type: " << (m_outputType == CCIE_OT_Binary ? "binary" : "xml") << std::endl;
	std::cout << "c-filter-defn-keep:"                                                    << std::endl;
	StringVector::iterator itSignature = m_filterDefinitionsKeepSignatures.begin();
	while(itSignature != m_filterDefinitionsKeepSignatures.end())
	{
		std::cout << "\t" << (*itSignature) << std::endl;
		itSignature++;
	}
	std::cout << "fc-filter-decl: " << boolToStr(m_bFilterDeclarations)                         << std::endl;
	std::cout << "fc-filter-defn: " << boolToStr(m_bFilterDefinitions)                          << std::endl;
	std::cout << "fc-filter-defn-conservative: " << boolToStr(m_bFilterDefinitionsConservative) << std::endl;
	std::cout << "fc-print-call-information: " << boolToStr(m_bPrintCallInformation)            << std::endl;
	std::cout << "fc-verbose: " << boolToStr(m_bVerbose)                                        << std::endl;
}

void CCIE_Cmd::printHelp()
{
	OinkCmd::printHelp();
	std::cout << std::endl << std::endl;
	std::cout << "ccie flags that take an argument:                                             " << std::endl;
	std::cout << "  -c-output FILE                : Use FILE as output filename instead of      " << std::endl;
	std::cout << "                                  default.                                    " << std::endl;
	std::cout << "  -c-output-type TYPE           : Specify the output type; one of: binary,    " << std::endl;
	std::cout << "                                  xml.                                        " << std::endl;
	std::cout << "  -c-filter-defn-keep SIGNATURE : Prevent all function definitions in the sub-" << std::endl;
	std::cout << "                                  call-graph starting at SIGNATURE from being " << std::endl;
	std::cout << "                                  filtered. All defs with non-static linkage  " << std::endl;
	std::cout << "                                  are never filtered; only defs with static   " << std::endl;
	std::cout << "                                  linkage are candidates for filtering.       " << std::endl;

	std::cout << std::endl;
	std::cout << "ccie boolean flags; precede by '-fc-no-' for the negative sense.              " << std::endl;
	std::cout << "  -fc-filter-decl              : Filter out declarations that are never used. " << std::endl;
	std::cout << "  -fc-filter-defn              : Filter out definitions that are never used.  " << std::endl;
	std::cout << "                                 This may filter too many definitions if not  " << std::endl;
	std::cout << "                                 in conservative mode.                        " << std::endl;
	std::cout << "  -fc-filter-defn-conservative : Be conservative in filtering definitions.    " << std::endl;
	std::cout << "                                 Only definitions that are guaranteed not to  " << std::endl;
	std::cout << "                                 be used are filtered. If ccc is giving you   " << std::endl;
	std::cout << "                                 warnings, specify this flag.                 " << std::endl;
	std::cout << "  -fc-print-call-information   : Print all extracted call information.        " << std::endl;
	std::cout << "  -fc-print-consv-warnings     : Print warnings about potential call          " << std::endl;
	std::cout << "                                 candidates being filtered.                   " << std::endl;
	std::cout << "  -fc-print-debug-information  : Print debug information.                     " << std::endl;
	std::cout << "  -fc-verbose                  : Print what is being done.                    " << std::endl;
}


// ----------------------------------------------------------------------------
// Properties
// ----------------------------------------------------------------------------
std::string CCIE_Cmd::OutputFileName()
{
	return m_szOutputFileName;
}

CCIE_Cmd::CCIE_OutputType CCIE_Cmd::OutputType()
{
	return m_outputType;
}

StringVector* CCIE_Cmd::FilterDefinitionsKeepSignatures()
{
	return &m_filterDefinitionsKeepSignatures;
}

bool CCIE_Cmd::FilterDeclarations()
{
	return m_bFilterDeclarations;
}

bool CCIE_Cmd::FilterDefinitions()
{
	return m_bFilterDefinitions;
}

bool CCIE_Cmd::FilterDefinitionsConservative()
{
	return m_bFilterDefinitionsConservative;
}

bool CCIE_Cmd::PrintCallInformation()
{
	return m_bPrintCallInformation;
}

bool CCIE_Cmd::PrintConservativenessWarnings()
{
	return m_bPrintConservativenessWarnings;
}

bool CCIE_Cmd::PrintDebugInformation()
{
	return m_bPrintDebugInformation;
}

bool CCIE_Cmd::Verbose()
{
	return m_bVerbose;
}

