/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "ccie_declaration_filter.h"
#include "ccie_debug_printing.h"


// ----------------------------------------------------------------------------
// Static accessor for convenience
// ----------------------------------------------------------------------------
void CCIE_DeclarationFilter::Filter(/*CCIE_TranslationUnitInfo & tuInfo*/)
{
	CCIE_DeclarationFilter filter/*(tuInfo)*/;
	filter._Filter();
}


// ----------------------------------------------------------------------------
// Construction & destruction
// ----------------------------------------------------------------------------
CCIE_DeclarationFilter::CCIE_DeclarationFilter(/*CCIE_TranslationUnitInfo & tuInfo*/) /*:
	CCIE_TranslationUnitInfo::(tuInfo)*/
{
}


// ----------------------------------------------------------------------------
// Filtering
// ----------------------------------------------------------------------------
void CCIE_DeclarationFilter::_Filter()
{
	// A CCI_LocalFunction can be thrown out iff:
	//
	// 1. It has no definition, and
	// 2. It is never called in this translation unit.
	//
	// This is sufficient, because:
	// * The removed declarations have no definition, so there is no usage of
	//   the function from another translation unit that is going to be linked
	//   to this declaration (Linking is done to a definition, not a
	//   declaration).
	// * If another translation unit uses the function represented by this
	//   declaration, it would need to have its own declaration anyway.
	//   Therefore it will never need this declaration.

	// Make a working copy of the CCI_LocalFunction map. This working copy will
	// contain all the functions that are going to be removed. Before that
	// happens, all functions that do not satisfy both requirement 1 and 2 will
	// be removed from the working copy.
	LocalFunctionMap* pLocalFunctions = CCIE_TranslationUnitInfo::m_localFunctionRepository.GetFunctions();
	LocalFunctionMap cpyLocalFunctions(*pLocalFunctions);

	// Delete all CCI_LocalFunctions from the map that are called in this
	// translation unit, since they do not satisfy requirement 2.
	ExcludeFunctionsCalledInDefinitions(cpyLocalFunctions);

	// Delete all CCI_LocalFunctions from the map that are called during
	// static initialization in this translation unit, since they do not
	// satisfy requirement 2 either.
	ExcludeFunctionsCalledInInitialization(cpyLocalFunctions);

	// Delete all CCI_LocalFunctions from the map that are called during
	// static finalization in this translation unit, since they do not satisfy
	// requirement 2 either.
	ExcludeFunctionsCalledInFinalization(cpyLocalFunctions);

	// Delete all CCI_LocalFunctions from the map that have a function
	// definition, since they do no satisfy requirement 1.
	ExcludeFunctionsWithDefinition(cpyLocalFunctions);

	// Now the map contains only functions that are guaranteed to have no
	// definition and are guaranteed to never be called from this translation
	// unit. Delete these functions from the repository.
	RemoveUnusedDeclarations(cpyLocalFunctions);
}

void CCIE_DeclarationFilter::ExcludeFunctionsCalledInDefinitions(LocalFunctionMap & localFunctionsToRemove)
{
	// Iterate through all CCI_LocalFunctions.
	LocalFunctionMap* pLocalFunctions = CCIE_TranslationUnitInfo::m_localFunctionRepository.GetFunctions();
	LocalFunctionMap::iterator itLocalFunction = pLocalFunctions->begin();
	while(itLocalFunction != pLocalFunctions->end())
	{
		// If this CCI_LocalFunction has a CCI_FunctionDefinition, retrieve it
		// and iterate through all its function calls.
		if(itLocalFunction->second->HasFunctionDefinition())
		{
			CCI_FunctionDefinition* pFunctionDefinition = itLocalFunction->second->GetFunctionDefinition();
			FunctionCallVector* pFunctionCalls = pFunctionDefinition->GetFunctionCalls();

			FunctionCallVector::iterator itFunctionCall = pFunctionCalls->begin();
			while(itFunctionCall != pFunctionCalls->end())
			{
				// All CCI_LocalFunctions that are possible call candidates for
				// this function call must be removed from the working copy of
				// CCI_LocalFunctions that are going to be filtered out.
				StringVector signatures;
				(*itFunctionCall)->GetCallCandidates(CCIE_TranslationUnitInfo::m_callCandidates, signatures);
				RemoveCallCandidateSignatures(localFunctionsToRemove, signatures);

				// Move on to the next function call.
				itFunctionCall++;
			}
		}

		// Move on to the next CCI_LocalFunction.
		itLocalFunction++;
	}
}

void CCIE_DeclarationFilter::ExcludeFunctionsCalledInInitialization(LocalFunctionMap & localFunctionsToRemove)
{
	FunctionCallVector::iterator itFunctionCall = CCIE_TranslationUnitInfo::m_initializingCalls.begin();
	while(itFunctionCall != CCIE_TranslationUnitInfo::m_initializingCalls.end())
	{
		// All CCI_LocalFunctions that are possible call candidates for this
		// function call must be removed from the working copy of
		// CCI_LocalFunctions that are going to be filtered out.
		StringVector signatures;
		(*itFunctionCall)->GetCallCandidates(CCIE_TranslationUnitInfo::m_callCandidates, signatures);
		RemoveCallCandidateSignatures(localFunctionsToRemove, signatures);

		// Move on to the next function call.
		itFunctionCall++;
	}

}

void CCIE_DeclarationFilter::ExcludeFunctionsCalledInFinalization(LocalFunctionMap & localFunctionsToRemove)
{
	FunctionCallVector::iterator itFunctionCall = CCIE_TranslationUnitInfo::m_finalizingCalls.begin();
	while(itFunctionCall != CCIE_TranslationUnitInfo::m_finalizingCalls.end())
	{
		// All CCI_LocalFunctions that are possible call candidates for this
		// function call must be removed from the working copy of
		// CCI_LocalFunctions that are going to be filtered out.
		StringVector signatures;
		(*itFunctionCall)->GetCallCandidates(CCIE_TranslationUnitInfo::m_callCandidates, signatures);
		RemoveCallCandidateSignatures(localFunctionsToRemove, signatures);

		itFunctionCall++;
	}
}

void CCIE_DeclarationFilter::ExcludeFunctionsWithDefinition(LocalFunctionMap & localFunctionsToRemove)
{
	// Delete all CCI_LocalFunctions that have a definition from the map, since
	// they do not satisfy requirement 1.
	LocalFunctionMap::iterator itLocalFunction = localFunctionsToRemove.begin();
	while(itLocalFunction != localFunctionsToRemove.end())
	{
		// If this CCI_LocalFunction has a function definition, it does not
		// satisfy requirement 1, and must be deleted from the copy map.
		if(itLocalFunction->second->HasFunctionDefinition())
			localFunctionsToRemove.erase(itLocalFunction++);
		else
			itLocalFunction++;
	}
}

void CCIE_DeclarationFilter::RemoveCallCandidateSignatures(LocalFunctionMap & localFunctions, StringVector & signatures)
{
	// Remove all signatures that are possible call-candidates from
	// the map of functions that will be removed.
	StringVector::iterator itSignature = signatures.begin();
	while(itSignature != signatures.end())
	{
		LocalFunctionMap::iterator itLocalFunction = localFunctions.find(*itSignature);
		if(itLocalFunction != localFunctions.end())
			localFunctions.erase(itLocalFunction);

		itSignature++;
	}
}

void CCIE_DeclarationFilter::RemoveUnusedDeclarations(LocalFunctionMap & localFunctionsToRemove)
{
	LocalFunctionMap::iterator itLocalFunction = localFunctionsToRemove.begin();
	DEBUG_PRINT("Removed Unused Declarations");
	DEBUG_PRINT("{");
	while(itLocalFunction != localFunctionsToRemove.end())
	{
		CCIE_TranslationUnitInfo::m_callCandidates.RemoveFunction(itLocalFunction->first);
		DEBUG_PRINT("\t" << itLocalFunction->first);

		itLocalFunction++;
	}
	DEBUG_PRINT("}");
}
