/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef CCIE_FUNCTION_CALL_FACTORY_H
#define CCIE_FUNCTION_CALL_FACTORY_H


// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "cc_ast.h"
#include "ccie_sourceloc_factory.h"
#include "ccie_function_signature.h"
#include <libcci/cci.h>
#include <vector>


// ----------------------------------------------------------------------------
// Class CCIE_FunctionCallFactory
// ----------------------------------------------------------------------------
class CCIE_FunctionCallFactory
{
	// ------------------------------------------------------------------------
	// Creation
	// ------------------------------------------------------------------------
private:
	static void Create(FunctionCallVector* pFunctionCalls, SourceLoc sourceLoc, Variable* pVariable)
	{
		cci_assert(pVariable != NULL);
		cci_assert(pVariable->type != NULL);

		// Determine what type of function we're dealing with.
		CCI_FunctionCallType functionCallType;
		if(pVariable->type->isFunctionType())
		{
			FunctionType* pFunctionType = pVariable->type->asFunctionType();
			if(pFunctionType->isMethod())
				functionCallType = CCI_FCT_DirectMethod;
			else if(pFunctionType->isConstructor())
				functionCallType = CCI_FCT_Constructor;
			else if(pFunctionType->isDestructor())
				functionCallType = CCI_FCT_Destructor;
			else
				functionCallType = CCI_FCT_DirectFunction;
		}
		else if(pVariable->type->isPointerType())
		{
			functionCallType = CCI_FCT_PointerToFunction;
		}
		else if(pVariable->type->isPointerToMemberType())
		{
			functionCallType = CCI_FCT_PointerToMember;
		}
		else if(pVariable->type->isReferenceType())
		{
			if(pVariable->type->asReferenceType()->atType->isPointerType())
			{
				functionCallType = CCI_FCT_PointerToFunction;
			}
			else if(pVariable->type->asReferenceType()->atType->isPointerToMemberType())
			{
				functionCallType = CCI_FCT_PointerToMember;
			}
			else
			{
				// If the type of the supplied variable is a reference type, then
				// this must be a call through a pointer! This should not happen.
				CCI_SourceLoc loc(sourceLocManager->getFile(sourceLoc), sourceLocManager->getLine(sourceLoc), sourceLocManager->getCol(sourceLoc), CCIE_TranslationUnitInfo::m_szSourceFileName);
				cci_assert_msg
				(
					false,
					std::endl <<
					"\t{"                                                                              << std::endl <<
					"\t\tProblem          : Call through reference variable, but not on pointer type." << std::endl <<
					"\t\tVariable type    : " << BaseType::getNameOfTag(pVariable->type->getTag())     << std::endl <<
				 	"\t\tReferenced type  : " << BaseType::getNameOfTag(pVariable->type->asReferenceType()->atType->getTag()) << std::endl <<
					"\t\tSignature of call: " << CCIE_FunctionSignature::GetSignature(pVariable)       << std::endl <<
					"\t\tLocation of call : " << loc.ToString("")                                      << std::endl <<
					"\t}"
				);
			}
		}
		else
		{
			CCI_SourceLoc loc(sourceLocManager->getFile(sourceLoc), sourceLocManager->getLine(sourceLoc), sourceLocManager->getCol(sourceLoc), CCIE_TranslationUnitInfo::m_szSourceFileName);
			cci_assert_msg
			(
				false,
				std::endl <<
				"\t{"                                                                          << std::endl <<
				"\t\tProblem          : Unknown type of (standard) function call"              << std::endl <<
				"\t\tVariable type    : " << BaseType::getNameOfTag(pVariable->type->getTag()) << std::endl <<
				"\t\tSignature of call: " << CCIE_FunctionSignature::GetSignature(pVariable)   << std::endl <<
				"\t\tLocation of call : " << loc.ToString("")                                  << std::endl <<
				"\t}"
			);
		}

		Create(pFunctionCalls, sourceLoc, pVariable, functionCallType);
	}


public:
	// Creates a CCI_FunctionCall from the function call represented by
	// pExpression. Returns NULL if no function call could be created. In that
	// case, the caller should continue without error.
	static void Create(FunctionCallVector* pFunctionCalls, SourceLoc sourceLoc, Expression* pExpression)
	{
		cci_assert(pExpression != NULL);

		// At first, the supplied expression is either of the following two:
		// 1. An E_constructor.
		// 2. The expression immediately under E_funCall.
		
		//
		if(pExpression->type->isPointerType() || pExpression->type->isPointerToMemberType())
		{
			Create(pFunctionCalls, sourceLoc, pExpression->type);
			return;
		}

		// Case 1.
		if(pExpression->isE_constructor())
		{
			E_constructor* pE_constructor = pExpression->asE_constructor();
			Variable* pVariable = pE_constructor->ctorVar;
			
			// Every now and then pVariable is NULL. It seems that this occurs when
			// a C-struct is copied, e.g., as a return value. For some reason, Elsa
			// indicates that as a constructor expression, but without a ctorVar.
			// It makes sense that there is no constructor though, since it is a C-
			// struct, not a C++-struct.
			if(pVariable != NULL)
			{
				//std::cerr << "[0\n";
				Create(pFunctionCalls, sourceLoc, pVariable);
				//std::cerr << "0]\n";
				return;
			}
		}

		// Case 2.
		else if(pExpression->isE_variable())
		{
			// This is a call to a 'free' (non-method) function, or a call
			// through a pointer-to-function.
			E_variable* pE_variable = pExpression->asE_variable();
			Variable* pVariable = pE_variable->var;

			// When in 'permissive' mode, every now and then pVariable is NULL.
			// This seems to be caused by compiler builtins like
			// '__synch_fetch_and_add'. This is in fact an error, but since we
			// are in permissive mode, it is tolerated. We cannot do anything
			// with this function call however, so ignore it.
			if(pVariable != NULL)
			{
				//std::cerr << "[1\n";
				Create(pFunctionCalls, sourceLoc, pVariable);
				//std::cerr << "1]\n";
				return;
			}
		}
		else if(pExpression->isE_fieldAcc())
		{
			// This is a call to a 'field accessor'. Known accessors:
			// * Destructor
			// * Operator=
			// * Method call (through object, object pointer or object reference).
			//std::cerr << "[2\n";
			Create(pFunctionCalls, sourceLoc, pExpression->asE_fieldAcc());
			//std::cerr << "2]\n";
			return;
		}
		else if(pExpression->isE_binary())
		{
			// This is a call through a pointer-to-member.
			E_binary* pE_binary = pExpression->asE_binary();

			Expression* pEx2 = pE_binary->e2;
			cci_assert(pEx2 != NULL);

			if(pEx2->isE_variable())
			{
				E_variable* pE_variable = pEx2->asE_variable();

				Variable* pVariable = pE_variable->var;
				cci_assert(pVariable != NULL);

				//std::cerr << "[3\n";
				//pExpression->debugPrint(std::cout, 0);
				Create(pFunctionCalls, sourceLoc, pVariable);
				//std::cerr << "3]\n";
				return;
			}
		}
		else if(pExpression->isE_deref())
		{
			//std::cerr << "[4\n";
			
			// If this is a pointer-to-function or pointer-to-member, we don't need
			// to look any further.
			//if(pExpression->type->isPointerType() || pExpression->type->isPointerToMemberType())
			//{
			//	Create(pFunctionCalls, sourceLoc, pExpression->type);
			//	return;
			//}

			// If this is something else, we need to dig down further by recursing
			// into the thing that the pointer points at.
			Create(pFunctionCalls, sourceLoc, pExpression->asE_deref()->ptr);
			//std::cerr << "4]\n";
			return;
		}
		else if(pExpression->isE_cond())
		{
			// This function call is of this form:
			// (condition ? function1 : function2)(parameters of function1 and function2)
			// We need to add calls to both function1 and function2.
			//std::cerr << "[5\n";
			// Add a call to function1.
			Create(pFunctionCalls, sourceLoc, pExpression->asE_cond()->th);
			// Add a call to function2.
			Create(pFunctionCalls, sourceLoc, pExpression->asE_cond()->el);
			//std::cerr << "5]\n";
			return;
		}
		else
		{
			//pExpression->debugPrint(std::cout, 0);
			//std::cerr << "[6\n";
			Create(pFunctionCalls, sourceLoc, pExpression->type);
			//std::cerr << "6]\n";
			return;
		}
		
/*		else if(pExpression->isE_cond())
		{
			pExpression->debugPrint(std::cerr, 0);
			CCI_SourceLoc loc(sourceLocManager->getFile(sourceLoc), sourceLocManager->getLine(sourceLoc), sourceLocManager->getCol(sourceLoc));
			cci_assert_msg_allow
			(
				false,
				std::endl <<
				"\t{"                                                                                                        << std::endl <<
				"\t\tSituation                : E_funCall->E_cond"                                                           << std::endl <<
				"\t\tSignature of E_cond->type: " << CCIE_FunctionSignature::GetCallSignature(pExpression->asE_cond()->type) << std::endl <<
				"\t\tLocation of call         : " << loc.ToString("")                                                        << std::endl <<
				"\t}"
			);
		}
		else if(pExpression->isE_cast())
		{
			pExpression->debugPrint(std::cerr, 0);
			CCI_SourceLoc loc(sourceLocManager->getFile(sourceLoc), sourceLocManager->getLine(sourceLoc), sourceLocManager->getCol(sourceLoc));
			cci_assert_msg_allow
			(
				false,
				std::endl <<
				"\t{"                                                                                                        << std::endl <<
				"\t\tSituation                : E_funCall->E_cast"                                                           << std::endl <<
				"\t\tSignature of E_cast->type: " << CCIE_FunctionSignature::GetCallSignature(pExpression->asE_cast()->type) << std::endl <<
				"\t\tLocation of call         : " << loc.ToString("")                                                        << std::endl <<
				"\t}"
			);
			if(pExpression->asE_cast()->type->isReferenceType())
			{
				std::cout << "!!!!: " << CCIE_FunctionSignature::GetCallSignature(pExpression->asE_cast()->type->asReferenceType()->atType) << std::endl;
			}
		}
		else
		{
			// What is this?
			pExpression->debugPrint(std::cerr, 0);
			cci_assert(false);
		}*/
	}
	
	static void Create(FunctionCallVector* pFunctionCalls, SourceLoc sourceLoc, E_fieldAcc* pFieldAcc)
	{
		cci_assert(pFieldAcc != NULL);
		cci_assert(pFieldAcc->obj != NULL);
		cci_assert(pFieldAcc->field != NULL);
		cci_assert(pFieldAcc->field->type != NULL);

		// In any case, the member (method, or pointer-to-*) that causes the
		// actual function call is now in pFieldAcc->field.
		Variable* pVariable = pFieldAcc->field;

		// If the call is though a pointer-to-function, or through a
		// pointer-to-member, then it really does not matter whether the call
		// was made on a 'direct' object, an object pointer or an object
		// reference.
		// Finally, there is the case that the thing that is being called, is a
		// reference type. Now, something that is not a pointer-to-function or a
		// pointer-to-member is a plain method. A plain method can, obviously,
		// never be a reference type. So, if the variable is a reference type, then
		// it is also a pointer type.
		if(pVariable->type->isPointerType() || pVariable->type->isPointerToMemberType() || pVariable->type->isReferenceType())
		{
			//std::cerr << "[7\n";
			Create(pFunctionCalls, sourceLoc, pVariable);
			//std::cerr << "7]\n";
			return;
		}

		// Now we are left to determine what to do now.
		if(pFieldAcc->obj->isE_variable())
		{
			// The call was made either on a 'direct' instance of an object, or
			// the call was made on a reference to an object. We need to find
			// out which is the case.
			if(pFieldAcc->obj->asE_variable()->var->type->isReferenceType())
			{
				//std::cerr << "[8\n";
				Create(pFunctionCalls, sourceLoc, pVariable, CCI_FCT_ObjectReference);
				//std::cerr << "8]\n";
				return;
			}
			else
			{
				// Can't this be handled by the default handler?
				//std::cerr << "[9\n";
				Create(pFunctionCalls, sourceLoc, pVariable, CCI_FCT_DirectMethod);
				//std::cerr << "9]\n";
				return;
			}
		}
		else if(pFieldAcc->obj->isE_fieldAcc())
		{
			// The object on which the call was made is a member of an
			// encapsulating object. Now, that is not really interesting. What
			// is interesting, though, is the fact that this means that the call
			// was made on a 'direct' instance of an object, or on a reference
			// to an object. We need to find out which is the case.
			if(pFieldAcc->obj->asE_fieldAcc()->field->type->isReferenceType())
			{
				//std::cerr << "[10\n";
				Create(pFunctionCalls, sourceLoc, pVariable, CCI_FCT_ObjectReference);
				//std::cerr << "10]\n";
				return;
			}
			else
			{
				//std::cerr << "[11\n";
				// Can't this be handled by the default handler?
				Create(pFunctionCalls, sourceLoc, pVariable, CCI_FCT_DirectMethod);
				//std::cerr << "11]\n";
				return;
			}
		}
		else if(pFieldAcc->obj->isE_deref())
		{
			// 'pFieldAcc->obj' represents a pointer to an object. Now, it might
			// be that the object that 'pFieldAcc->obj' points to is itself a
			// member of another object, but that is not of interest. All we
			// care about, is that the call was made on a pointer to an object.
			//std::cerr << "[12\n";
			Create(pFunctionCalls, sourceLoc, pVariable, CCI_FCT_ObjectPointer);
			//std::cerr << "12]\n";
			return;
		}
		else
		{
			// In all other cases, just use the pVariable to extract the function call.
			cci_assert(pVariable != NULL);
			//std::cerr << "[13\n";
			Create(pFunctionCalls, sourceLoc, pVariable);
			//std::cerr << "13]\n";
			return;
			
			// I have never seen this before. Print some information so we can
			// investigate and then halt.
			//pFieldAcc->debugPrint(std::cerr, 0);
			//cci_assert(false);
		}
	}

	/*static CCI_FunctionCall* Create(SourceLoc sourceLoc, Variable* pVariable, Expression* pObject)
	{
		cci_assert(pVariable != NULL);
		cci_assert(pObject != NULL);

		// Before we check whether we need to handle a special case (of a call
		// on a reference to an object or a pointer to an object), first check
		// whether the call is made through a pointer-to-function or a
		// pointer-to-member.
		// If that is the case, then it does not matter if the call was made on
		// an object reference or object pointer, since that will not have any
		// influence on what its call candidates are going to be: Its call
		// candidates will be those functions that are the call candidates of
		// the pointer-to-function or pointer-to-member signature. Therefore, in
		// that case, this call must not receive special treatment and must be
		// processed in the default manner.
		if(!pVariable->type->isPointerType() && !pVariable->type->isPointerToMemberType())
		{
			// Handle the special case where a function is called on a reference to an object.
			if(pObject->isE_variable())
			{
				cci_assert(pObject->asE_variable()->var != NULL);
				cci_assert(pObject->asE_variable()->var->type != NULL);
				if(pObject->asE_variable()->var->type->isReferenceType())
				{
					return Create(sourceLoc, pVariable, CCI_FCT_ObjectReference);
				}
				else
				{
					std::cerr << "OTHER TYPE\n";
				}
			}

			// Handle the special case where the function is called on a pointer to an object.
			else if(pObject->isE_deref())
			{
				// Something is being dereferenced, lets find out what it is.
				// Determine what type of call we're dealing with.
				if(pVariable->type->isFunctionType())
				{
					// This applies to destructors now as well (through a 'delete' call),
					// but that's good, since destructors can be virtual as well.
					return Create(sourceLoc, pVariable, CCI_FCT_ObjectPointer);
				}
				else
				{
					// This is not a call through a pointer-to-function or a
					// pointer-to-member, and it is not a 'normal' function
					// call. Whatever it is, I did not expect it here.
					CCI_SourceLoc loc(sourceLocManager->getFile(sourceLoc), sourceLocManager->getLine(sourceLoc), sourceLocManager->getCol(sourceLoc));
					std::cerr<<type_toString(pVariable->type)<<std::endl;
					cci_assert_msg
					(
						false,
						std::endl <<
						"\t{"                                                                          << std::endl <<
						"\t\tProblem          : Unknown type of function call using this-pointer"      << std::endl <<
						"\t\tVariable type    : " << BaseType::getNameOfTag(pVariable->type->getTag()) << std::endl <<
						"\t\tSignature of call: " << CCIE_FunctionSignature::GetSignature(pVariable)   << std::endl <<
						"\t\tLocation of call : " << loc.ToString("")                                  << std::endl <<
						"\t}"
					);
				}
			}
			else
			{
				std::cerr << "laskdjflskdfj\n";
			}
		}

		// This call was not a special case, so handle it in the default manner.
		return Create(sourceLoc, pVariable);
	}*/

private:
	static void Create(FunctionCallVector* pFunctionCalls, SourceLoc sourceLoc, Variable* pVariable, CCI_FunctionCallType functionCallType)
	{
		std::string szSignature = "";

		switch(functionCallType)
		{
			case CCI_FCT_DirectFunction:
			case CCI_FCT_DirectMethod:
			case CCI_FCT_ObjectPointer:
			case CCI_FCT_ObjectReference:
			case CCI_FCT_Constructor:
			case CCI_FCT_Destructor:
				szSignature = CCIE_FunctionSignature::GetSignature(pVariable);
				break;
			case CCI_FCT_PointerToFunction:
				if(pVariable->type->isPointerType())
					szSignature = CCIE_FunctionSignature::GetCallSignature(pVariable->type->asPointerType()->atType);
				else if(pVariable->type->isReferenceType() && pVariable->type->asReferenceType()->atType->isPointerType())
					szSignature = CCIE_FunctionSignature::GetCallSignature(pVariable->type->asReferenceType()->atType->asPointerType()->atType);
				else
					cci_assert(false);

				break;
			case CCI_FCT_PointerToMember:
				if(pVariable->type->isPointerToMemberType())
					szSignature = CCIE_FunctionSignature::GetCallSignature(pVariable->type->asPointerToMemberType()->atType);
				else if(pVariable->type->isReferenceType() && pVariable->type->asReferenceType()->atType->isPointerToMemberType())
					szSignature = CCIE_FunctionSignature::GetCallSignature(pVariable->type->asReferenceType()->atType->asPointerToMemberType()->atType);
				break;
			default:
				CCI_SourceLoc loc(sourceLocManager->getFile(sourceLoc), sourceLocManager->getLine(sourceLoc), sourceLocManager->getCol(sourceLoc), CCIE_TranslationUnitInfo::m_szSourceFileName);
				cci_assert_msg
				(
					false,
					std::endl <<
					"\t{"                                                                             << std::endl <<
					"\t\tProblem             : Illegal CCI_FunctionCallType"                          << std::endl <<
					"\t\tCCI_FunctionCallType: " << functionCallType                                  << std::endl <<
					"\t\tVariable type       : " << BaseType::getNameOfTag(pVariable->type->getTag()) << std::endl <<
					"\t\tSignature of call   : " << CCIE_FunctionSignature::GetSignature(pVariable)   << std::endl <<
					"\t\tLocation of call    : " << loc.ToString("")                                  << std::endl <<
					"\t}"
				);
		}

		pFunctionCalls->push_back(new CCI_FunctionCall(functionCallType, CCIE_SourceLocFactory::Create(sourceLoc), szSignature));
	}
	
	static void Create(FunctionCallVector* pFunctionCalls, SourceLoc sourceLoc, Type* pType)
	{
		cci_assert(pType != NULL);
		
		CCI_FunctionCallType functionCallType;
		std::string szSignature;
		
		if(pType->isPointerType())
		{
			functionCallType = CCI_FCT_PointerToFunction;
			szSignature = CCIE_FunctionSignature::GetCallSignature(pType->asPointerType()->atType);
		}
		else if(pType->isPointerToMemberType())
		{
			functionCallType = CCI_FCT_PointerToMember;
			szSignature = CCIE_FunctionSignature::GetCallSignature(pType->asPointerToMemberType()->atType);
		}
		else if(pType->isReferenceType())
		{
			Create(pFunctionCalls, sourceLoc, pType->asReferenceType()->atType);
			return;
		}
		else
		{
			CCI_SourceLoc loc(sourceLocManager->getFile(sourceLoc), sourceLocManager->getLine(sourceLoc), sourceLocManager->getCol(sourceLoc), CCIE_TranslationUnitInfo::m_szSourceFileName);
			cci_assert_msg
			(
				false,
				std::endl <<
				"\t{"                                                                          << std::endl <<
				"\t\tProblem          : Attempt to create CCI_FunctionCall using Type from non-pointer, non-reference type." << std::endl <<
				"\t\tType             : " << BaseType::getNameOfTag(pType->getTag())           << std::endl <<
				"\t\tSignature of type: " << CCIE_FunctionSignature::GetCallSignature(pType)   << std::endl <<
				"\t\tLocation of call : " << loc.ToString("")                                  << std::endl <<
				"\t}"
			);
		}
		
		pFunctionCalls->push_back(new CCI_FunctionCall(functionCallType, CCIE_SourceLocFactory::Create(sourceLoc), szSignature));
	}
};


#endif
