/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "ccie_linkage_semantics.h"
//#include "ci_function_call.h"
#include <libcci/cci.h>


// ----------------------------------------------------------------------------
// Static accessor for convenience
// ----------------------------------------------------------------------------
void CCIE_LinkageSemantics::MangleStaticLinkageSignatures(/*CCIE_TranslationUnitInfo & tuInfo*/)
{
	CCIE_LinkageSemantics linkageSemantics/*(tuInfo)*/;
	linkageSemantics._MangleStaticLinkageSignatures();
}


// ----------------------------------------------------------------------------
// Construction
// ----------------------------------------------------------------------------
CCIE_LinkageSemantics::CCIE_LinkageSemantics(/*CCIE_TranslationUnitInfo & tuInfo*/) /*:
	m_tuInfo(tuInfo)*/
{
}


// ----------------------------------------------------------------------------
// Mangling
// ----------------------------------------------------------------------------
void CCIE_LinkageSemantics::_MangleStaticLinkageSignatures()
{
	// When a function has static linkage it is only visible in this
	// translation unit. That means that functions in another translation unit,
	// with the same signature, must not be linked to this function. For that
	// reason we need to mangle the names of functions with static linkage.

	// 1. Iterate over all function calls, to find their corresponding function.
	//    If that corresponding function has static linkage, the signature of
	//    the call needs to be mangled.
	MangleFunctionCallSignatures();

	// 2. Iterate over all functions. If a function overriddes another function,
	//    check whether the signature of the overridden function (held by the
	//    first function) needs to be mangled.
	MangleVirtualOverriddenSignatures();

	// 3.
	MangleVirtualSignatureMapping();

	// 4.
	MangleCallSignatureMapping();

	// 5. Iterate over all functions. If a function has static linkage, mangle
	//    its signature.
	MangleLocalFunctionSignatures();
}

void CCIE_LinkageSemantics::MangleFunctionCallSignatures()
{
	//
	// Mangles the reference-signatures held by all function calls.
	//

	// Mangle the signatures of function calls in function definitions.
	LocalFunctionMap* pLocalFunctions = CCIE_TranslationUnitInfo::m_localFunctionRepository.GetFunctions();
	LocalFunctionMap::iterator itLocalFunction = pLocalFunctions->begin();
	while(itLocalFunction != pLocalFunctions->end())
	{
		CCI_LocalFunction* pLocalFunction = itLocalFunction->second;

		// A function is only interesting if it has a function definition,
		// because only function definitions contain function calls.
		if(pLocalFunction->HasFunctionDefinition())
		{
			CCI_FunctionDefinition* pFunctionDefinition = pLocalFunction->GetFunctionDefinition();
			FunctionCallVector* pFunctionCalls = pFunctionDefinition->GetFunctionCalls();
			FunctionCallVector::iterator itFunctionCall = pFunctionCalls->begin();
			while(itFunctionCall != pFunctionCalls->end())
			{
				// If necessary, mangle the signature of this function call.
				MangleFunctionCall(*itFunctionCall);
				itFunctionCall++;
			}
		}

		itLocalFunction++;
	}

	// Mangle the signatures of static initializing calls.
	FunctionCallVector::iterator itFunctionCall = CCIE_TranslationUnitInfo::m_initializingCalls.begin();
	while(itFunctionCall != CCIE_TranslationUnitInfo::m_initializingCalls.end())
	{
		// If necessary, mangle the signature of this function call.
		MangleFunctionCall(*itFunctionCall);
		itFunctionCall++;
	}

	// Mangle the signatures of static finalizing calls.
	itFunctionCall = CCIE_TranslationUnitInfo::m_finalizingCalls.begin();
	while(itFunctionCall != CCIE_TranslationUnitInfo::m_finalizingCalls.end())
	{
		// If necessary, mangle the signature of this function call.
		MangleFunctionCall(*itFunctionCall);
		itFunctionCall++;
	}
}

void CCIE_LinkageSemantics::MangleFunctionCall(CCI_FunctionCall* pFunctionCall)
{
	// Retrieve the signature of the function call.
	std::string signature = pFunctionCall->GetSignature();

	if(IsStaticLinkage(signature))
	{
		// Mangle the signature of the function call.
		pFunctionCall->SetSignature(GetMangledSignature(signature));
	}
}

void CCIE_LinkageSemantics::MangleVirtualOverriddenSignatures()
{
	//
	// Mangles the reference-signatures held by all virtual functions that
	// override another function.
	//
	
	LocalFunctionMap* pLocalFunctions = CCIE_TranslationUnitInfo::m_localFunctionRepository.GetFunctions();
	LocalFunctionMap::iterator itLocalFunction = pLocalFunctions->begin();
	while(itLocalFunction != pLocalFunctions->end())
	{
		CCI_LocalFunction* pLocalFunction = itLocalFunction->second;

		StringVector* pOverriddenMethodSignatures = pLocalFunction->GetOverriddenMethodSignatures();
		StringVector::iterator itOverriddenMethodSignature = pOverriddenMethodSignatures->begin();
		while(itOverriddenMethodSignature != pOverriddenMethodSignatures->end())
		{
			std::string szSignature = *itOverriddenMethodSignature;
			if(IsStaticLinkage(szSignature))
			{
				// Mangle the signature of the overridden method.
				*itOverriddenMethodSignature = GetMangledSignature(szSignature);
			}
			itOverriddenMethodSignature++;
		}
		
		itLocalFunction++;
	}
}

void CCIE_LinkageSemantics::MangleVirtualSignatureMapping()
{
	//
	// Mangles the reference-signatures held by the virtual signature mapping.
	// Also mangles the signatures that are the keys into the mapping.
	//

	// First, mangle the signatures of the overriding functions.
	StringVectorMap* pOverriders = CCIE_TranslationUnitInfo::m_callCandidates.GetOverridersMap();
	StringVectorMap::iterator itSignatures = pOverriders->begin();
	while(itSignatures != pOverriders->end())
	{
		// Iterate over all signatures of the functions overriding this function.
		StringVector* pSignatures = itSignatures->second;
		StringVector::iterator itSignature = pSignatures->begin();
		while(itSignature != pSignatures->end())
		{
			std::string szSignature = *itSignature;
			if(IsStaticLinkage(szSignature))
				*itSignature = GetMangledSignature(szSignature);

			itSignature++;
		}

		itSignatures++;
	}

	// Second, mangle the key-signatures.
	StringVectorMap tmpSignatures;
	itSignatures = pOverriders->begin();
	while(itSignatures != pOverriders->end())
	{
		std::string szSignature = itSignatures->first;
		if(IsStaticLinkage(szSignature))
		{
			// Add the key/value pair to the temporary map, only with a mangled
			// key-signature.
			tmpSignatures[GetMangledSignature(szSignature)] = itSignatures->second;

			// Remove the current key/value pair, since it must be reinserted
			// with a mangled key.
			pOverriders->erase(itSignatures++);
		}
		else
		{
			itSignatures++;
		}
	}
	// Now reinsert the key/value pairs from the temporary map into the
	// m_virtualSignatureToOverridingSignatures map.
	pOverriders->insert(tmpSignatures.begin(), tmpSignatures.end());
}

void CCIE_LinkageSemantics::MangleCallSignatureMapping()
{
	//
	// Mangles the reference-signatures held by the call-signature mapping.
	//

	// Mangle the signatures of the call-candidates.
	StringVectorMap* pPointerMatches = CCIE_TranslationUnitInfo::m_callCandidates.GetPointerMatchesMap();
	StringVectorMap::iterator itSignatures = pPointerMatches->begin();
	while(itSignatures != pPointerMatches->end())
	{
		// Iterate over all signatures of the call-candidates.
		StringVector* pSignatures = itSignatures->second;
		StringVector::iterator itSignature = pSignatures->begin();
		while(itSignature != pSignatures->end())
		{
			std::string szSignature = *itSignature;
			if(IsStaticLinkage(szSignature))
				*itSignature = GetMangledSignature(szSignature);

			itSignature++;
		}

		itSignatures++;
	}
}

void CCIE_LinkageSemantics::MangleLocalFunctionSignatures()
{
	//
	// Mangles the signatures of the CCI_LocalFunctions themselves. Also mangles
	// the signatures that are the keys into the mapping.
	//
	
	// NOTE: Call-signatures must never be mangled! Why? Functions with static
	//       linkage can still be called via a call through pointer-to-function
	//       from another translation unit. Therefore, functions with static
	//       linkage are still valid call-candidates for calls through
	//       pointers-to-functions and must therefore be included in the set
	//       of call-candidates and must therefore not be mangled.

	LocalFunctionMap* pLocalFunctions = CCIE_TranslationUnitInfo::m_localFunctionRepository.GetFunctions();
	LocalFunctionMap::iterator itLocalFunction = pLocalFunctions->begin();
	LocalFunctionMap tmpLocalFunctions;
	while(itLocalFunction != pLocalFunctions->end())
	{
		CCI_LocalFunction* pLocalFunction = itLocalFunction->second;

		if(pLocalFunction->IsFlagStaticLinkage())
		{
			// Mangle this local function's signature name.
			std::string szSignature = pLocalFunction->GetSignature();
			std::string szMangledSignature = GetMangledSignature(szSignature);

			// Update the signature of the CCI_LocalFunction.
			pLocalFunction->SetSignature(szMangledSignature);

			// Add the CCI_LocalFunction to the temporary map of CCI_LocalFunctions.
			tmpLocalFunctions[szMangledSignature] = pLocalFunction;

			// Remove the CCI_LocalFunction from the repository, since it must
			// be reinserted with a different key.
			pLocalFunctions->erase(itLocalFunction++);
		}
		else
		{
			itLocalFunction++;
		}
	}

	// Add all CCI_LocalFunctions in the temporary map to the local function
	// repository.
	pLocalFunctions->insert(tmpLocalFunctions.begin(), tmpLocalFunctions.end());
}

bool CCIE_LinkageSemantics::IsStaticLinkage(std::string szSignature)
{
	// Retrieve the function belonging to the signature.
	CCI_LocalFunction* pLocalFunction = CCIE_TranslationUnitInfo::m_localFunctionRepository.GetFunction(szSignature);
	if(pLocalFunction != NULL)
	{
		// Return whether the function has static linkage.
		return pLocalFunction->IsFlagStaticLinkage();
	}
	return false;
}

std::string CCIE_LinkageSemantics::GetMangledSignature(std::string szSignature)
{
	return szSignature + "{sl:" + CCIE_TranslationUnitInfo::m_szSourceFileName + "}";
}
