/***************************************************************************
 *   Copyright (C) 2009 by Hessel Hoogendorp                               *
 *   bugs.ccc@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


// ----------------------------------------------------------------------------
// Includes
// ----------------------------------------------------------------------------
#include "ccie_local_function_repository.h"
#include "ccie_local_function_factory.h"
#include <libcci/cci.h>


// ----------------------------------------------------------------------------
// Construction & destruction
// ----------------------------------------------------------------------------
CCIE_LocalFunctionRepository::CCIE_LocalFunctionRepository()
{
}

CCIE_LocalFunctionRepository::~CCIE_LocalFunctionRepository()
{
	LocalFunctionMap::iterator it = m_functions.begin();
	while(it != m_functions.end())
	{
		delete it->second;
		it++;
	}
	m_functions.clear();
}


// ----------------------------------------------------------------------------
// Repository management
// ----------------------------------------------------------------------------
bool CCIE_LocalFunctionRepository::ContainsFunction(std::string signature)
{
	return m_functions.find(signature) != m_functions.end();
}

bool CCIE_LocalFunctionRepository::ContainsFunction(Function* pFunction)
{
	cci_assert(pFunction != NULL);
	std::string signature = CCIE_FunctionSignature::GetSignature(pFunction);
 	return ContainsFunction(signature);
}

CCI_LocalFunction* CCIE_LocalFunctionRepository::GetFunction(std::string signature)
{
	if(!ContainsFunction(signature)) return NULL;
	return m_functions[signature];
}

LocalFunctionMap* CCIE_LocalFunctionRepository::GetFunctions()
{
	return &m_functions;
}

void CCIE_LocalFunctionRepository::AddFunction(Function* pFunction)
{
	cci_assert(pFunction != NULL);
	
	//
	// This is called ONLY when adding a function from a DEFINITION.
	//

	std::string signature = CCIE_FunctionSignature::GetSignature(pFunction);
	
	// If the signature was already present in the repository, we have a
	// duplicate definition.
	if(ContainsFunction(signature))
	{
		// Test whether the new and the existing definition are truly different;
		// sometimes duplicate definitions are caused by template partial
		// specializations, and are not really duplicate definitions.

		// Retrieve the CCI_SourceLoc of the new definition.
		SourceLoc loc = pFunction->getLoc();
		CCI_SourceLoc newLoc(sourceLocManager->getFile(loc), sourceLocManager->getLine(loc), sourceLocManager->getCol(loc), CCIE_TranslationUnitInfo::m_szSourceFileName);

		// Retrieve the CCI_SourceLoc of the existing definition.
		cci_assert_msg(GetFunction(signature)->GetFunctionDefinition() != NULL, signature);
		CCI_SourceLoc* pOldLoc = GetFunction(signature)->GetFunctionDefinition()->GetSourceLoc();

		// If the CCI_SourceLocs are different, then we really have a duplicate
		// definition.
		if(*pOldLoc != newLoc)
		{
			cci_assert_msg_allow
			(
			 	false,
			 	std::endl <<
				"\t{"                                                   << std::endl <<
				"\t\tProblem               : Function redefinition"     << std::endl <<
				"\t\tSignature             : " << signature             << std::endl <<
				"\t\tNew definition is here: " << newLoc.ToString("")   << std::endl <<
				"\t\tOld definition is here: " << pOldLoc->ToString("") << std::endl <<
				"\t}"
			);
		}
		
		// If the CCI_SourceLocs are equal, then we already have this definition
		// and we do not need to do anything.
	}
	else
	{
		//m_functions[signature] = new CCI_LocalFunction(pFunction);
		m_functions[signature] = CCIE_LocalFunctionFactory::Create(pFunction);
	}
}

void CCIE_LocalFunctionRepository::AddFunction(Variable* pVariable)
{
	cci_assert(pVariable != NULL);
	
	//
	// This is called ONLY when adding a function from a DECLARATION. Note also,
	// that this only happens AFTER ALL DEFINITIONS have been added. Therefore,
	// if this is called, that means an attempt is made to add a function from a
	// declaration, which might or might not already have a definition or
	// declaration attached to it.
	//

	std::string signature = CCIE_FunctionSignature::GetSignature(pVariable);

	// If there is no definition nor declaration yet, add a declaration now.
	if(!ContainsFunction(signature))
	{
		//m_functions[signature] = new CCI_LocalFunction(pVariable);
		m_functions[signature] = CCIE_LocalFunctionFactory::Create(pVariable);
	}
}

void CCIE_LocalFunctionRepository::DeleteFunction(std::string signature)
{
	// If the function with the specified signature exists in the repository,
	// then delete it from the repository.
	LocalFunctionMap::iterator itLocalFunction = m_functions.find(signature);
	if(itLocalFunction != m_functions.end())
	{
		delete itLocalFunction->second;
		m_functions.erase(itLocalFunction);
	}
}


// ----------------------------------------------------------------------------
// Debugging
// ----------------------------------------------------------------------------
void CCIE_LocalFunctionRepository::Print(std::string prefix)
{
	std::cout << std::endl << prefix << "Local Function Repository";
	std::cout << std::endl << prefix << "{";

	LocalFunctionMap::iterator it = m_functions.begin();
	while(it != m_functions.end())
	{
		if(it != m_functions.begin())
			std::cout << std::endl;

		it->second->PrintOut(prefix + "\t");
		it++;
	}

	std::cout << std::endl << prefix << "}";
}
