// see License.txt for copyright and terms of use

#include "nuker_cmd.h"    // NukerCmd
#include "oink.gr.gen.h"        // CCParse_Oink
#include "strutil.h"            // quoted
#include "oink_util.h"
#include "expr_visitor.h"
#include "squash_util.h"
#include <sstream>
#include <cppundolog.h>
#include <set>
#include <fstream>
#include "patcher.h"

using namespace std;

NukerCmd *nukerCmd = NULL;
typedef vector<StringRef> NameVector;
typedef set<NameVector> ClassSet;
/* Can rename class names
x */
class NukerVisitor : public ExpressionVisitor {
public:
  NukerVisitor(Patcher &patcher, const ClassSet &targetClasses): 
    patcher(patcher),
    targetClasses(targetClasses)
  {
  }

  bool visitTypeSpecifier(TypeSpecifier *ts);
  bool visitTopForm(TopForm *f);
  void postvisitMember(Member *m);
  void maybeNukeDtor(Variable *v, SourceLoc loc, SourceLoc endloc);
  bool isTarget(Scope *s);
  bool isDerivedTarget(Scope *s);
  bool visitFunction(Function *f);
  bool visitStatement(Statement *f);
private:
  Patcher &patcher;
  const ClassSet &targetClasses;
};

NameVector tokenize(std::string const &line, std::string const delim = "::") {
  NameVector v;
  string::size_type from = 0;
  do {
    string::size_type n = line.find(delim, from);
    if (n == string::npos) {
      n = line.size();
    }
    v.push_back(globalStrTable(line.substr(from, n - from).c_str()));
    from = n + delim.size();
  } while (from < line.size());
  return v;
}

void NukerRunner::run() {
  if (nukerCmd->inputFile.empty()) {
    cerr << "Must specify a list of classes to process" << endl;
  }
  ClassSet targetClasses;
  ifstream fin(nukerCmd->inputFile.c_str());
  if(!fin.is_open()) {
    cerr << "Could not open file list: " << nukerCmd->inputFile << endl;
    exit(1);
  }
  string line;
  while(getline(fin, line)) {
    targetClasses.insert(tokenize(line));
  }

  Patcher patcher;
  NukerVisitor nuker(patcher, targetClasses);
  foreachSourceFile {
    File *file = files.data();
    patcher.setFile(file->name);
    
    maybeSetInputLangFromSuffix(file);
    TranslationUnit *unit = file2unit.get(file);
    unit->traverse(nuker.loweredVisitor);
  }
}

bool NukerVisitor::visitTypeSpecifier(TypeSpecifier *ts) {
  try {
    TS_classSpec *cs = ts->asTS_classSpec();
    PQ_name *name = SOME(cs->name)->asPQ_name();
    xmatch(isTarget(cs->ctype->getTypedefVar()->getDenotedScope()));
    CPPSourceLoc csl(name->loc);
    int offset = strlen(name->name);
    if (csl.macroExpansion) {
      cerr << csl.loc() << ": macro " << csl.macroExpansion->name <<
        " prevents inhertiance mod for " << name->name << endl;
      return true;
    } else if (csl.loc() == SL_UNKNOWN) {
      cerr << "Elsa is missing position info in order to do inhertiance mod for "
           << name->name << endl;
      return true;
    }
    StringRef insertDerived = " : public XPCOMGCObject";
    //if there is a base class insert 'public XPCOMGCObject,'
    if (cs->bases->first()) {
      stringstream ss;
      UnboxedLoc ul;
      StringRef file = ul.set(csl.loc());
      string line = patcher.getLine(ul.line, file);
      ul.col += offset;
      string::size_type pos = line.find(":", ul.col - 1);
      insertDerived = " public XPCOMGCObject,";
      if (pos == string::npos) {
        cerr << csl.loc() << 
          ": Could not find ':' to insert '" << insertDerived << "'" << endl;
        return true;
      }
      offset = 0;
      // 1 to change into elsa positions..+1 to move after :
      ul.col = pos + 2;
      csl.overrideLoc(ul.toSourceLoc(file));
    }
    patcher.insertBefore(csl, insertDerived,
                         offset);

  } catch (x_match&) {
  }
  return true;
}

void NukerVisitor::maybeNukeDtor(Variable *v, SourceLoc loc, SourceLoc endloc) {
  StringRef name = v->name;
  if (v->isImplicitMemberFunc()
       || !(name && name[0] == '~' && isDerivedTarget(v->scope))) return;
  PairLoc p(loc, endloc);
  if (p.hasExactPosition()) 
    patcher.printPatch("", p);
  else {
    MacroUndoEntry *m = p.getMacro();
    if (m) {
      cerr << loc << ": " << m->name
           << " prevented from nuking " << name << endl;
    } else {
      cerr << "Elsa has insufficient position info to nuke " << v->name << endl;
    }
  }

}

/* nuke ~Foo(); and ~Foo() {} */
void NukerVisitor::postvisitMember(Member *m) {
  if (MR_func *mf = m->ifMR_func()) {
    maybeNukeDtor(mf->f->nameAndParams->var, m->loc, m->endloc);
  } else if (MR_decl *md = m->ifMR_decl()) {
    Declarator *d = md->d->decllist->first();
    if (d)
      maybeNukeDtor(d->var, m->loc, m->endloc);
  }
}

/* nuke Foo::~Foo() */
bool NukerVisitor::visitTopForm(TopForm *t) {
  if (TF_func *tf = t->ifTF_func())
    if (tf->f->body)
      maybeNukeDtor(tf->f->nameAndParams->var, tf->loc, tf->f->body->endloc);
  return true;
}

bool NukerVisitor::isTarget(Scope *s) {
  if (!s)
    return false;

  for(ClassSet::iterator it = targetClasses.begin();
      it != targetClasses.end();
      ++it) {
    NameVector const &n = *it;
    Scope *targetScope = s;
    NameVector::const_reverse_iterator rit = n.rbegin();
    for (; rit != n.rend(); ++rit) {
      if (*rit != targetScope->getTypedefName()->name)
        break;
      targetScope = targetScope->parentScope;
    }
    // everything in the list matched
    if (rit == n.rend() && targetScope->scopeKind == SK_GLOBAL)
      return true;
  }
  return false;
}

bool NukerVisitor::isDerivedTarget(Scope *s) {
  if (isTarget(s)) return true;
  TS_classSpec *ts = s->curCompound->syntax;
  if (!ts) return false;
  for (BaseClassSpec *base = ts->bases->first();
       base; base = base->next) {
    if (isDerivedTarget(base->type->getTypedefVar()->getDenotedScope()))
      return true;
  }
  return false;
}

bool NukerVisitor::visitFunction(Function *f) {
  Variable *v = f->nameAndParams->var;
  FunctionType *t = v->type->asFunctionType();
  // make sure we only go into bodies of constructors
  return t->isConstructor() && !v->isImplicitMemberFunc()
    && isDerivedTarget(v->scope);
}

bool NukerVisitor::visitStatement(Statement *s) {
  CPPSourceLoc csl(s->loc);
  static StringRef str_MOZ_COUNT_CTOR = globalStrTable("MOZ_COUNT_CTOR");
  MacroUndoEntry *m = csl.macroExpansion;
  if (m && m->name == str_MOZ_COUNT_CTOR) {
    UnboxedLoc l1, l2; 
    StringRef file;
    file = l1.set(m->preStartLoc);
    xassert(file == l2.set(m->preEndLoc));
    patcher.printPatch("", UnboxedPairLoc(file, l1, l2));
  }
  return true;
}
