// see License.txt for copyright and terms of use

#include "outparamdel.h"        // this module
#include "outparamdel_cmd.h"    // OutparamdelCmd
#include "outparamdel_global.h"
#include "oink.gr.gen.h"        // CCParse_Oink
#include "strutil.h"            // quoted
#include "oink_util.h"
#include "expr_visitor.h"
#include "squash_util.h"
#include "patcher.h"
#include "outparamdel_cmd.h"
#include "cppundolog.h"
#include <fstream>
#include <list>
#include <ctype.h>
#include <sstream>
#include <memory>
#include <set>

#define STRINGIFY(x) #x
#define TOSTRING(x) STRINGIFY(x)

using namespace std;

// used to see if we eliminated all usage of an nsresult var
class VarUsageCounter : public ExpressionVisitor {
public:
  VarUsageCounter(Variable *v):
    sdecl(NULL), v(v), counter(0) {
  }

  void skipStatement(Statement *toSkip) {
    skipSet.insert(toSkip);
  }

  bool visitStatement(Statement *s) {
    if (skipSet.find(s) != skipSet.end())
      return false;
    return ExpressionVisitor::visitStatement(s);
  }

  // use of a var counts as use :)
  void postvisitE_variable(E_variable *ev) {
    counter += ev->var == v;
  }

  // initialization of a var also counts as use
  void postvisitS_decl(S_decl *sdecl) {
    try {
      Declarator *d = SOME(sdecl->decl->decllist)->first();
      if (d->var == v) {
        // save declaration location
        this->sdecl = sdecl;
        if (d->init)
          counter++;
      }
    } catch (x_match&) {
    }
  }
  
  S_decl *sdecl;
  Variable *v;
  set<Statement*> skipSet;
  int counter;
};

typedef pair<string, string> str_pair;
struct OutParamInfo;
typedef map<Variable*, OutParamInfo*> outparam_map;

struct OutParamInfo {
  OutParamInfo(): isVirtual(false), already_AddRefed(false), outparamType(NULL) {
  }
  string toString() const {
    stringstream ss;
    ss << className << "::" << memberName << '(';
    for(vector<string>::const_iterator it = paramTypes.begin();
        it != paramTypes.end();
        it++) {
      if (it != paramTypes.begin())
        ss << ',';
      ss << *it;
    }
    ss << ")," << param << ',';
    if (already_AddRefed)
      ss << "already_AddRefed";
    return ss.str();
  }
  StringRef className;
  string memberName;
  // index of the param to remove
  unsigned int param;
  // name used in getter
  string outparamName;
  // name for local var
  string retvarName;
  bool isVirtual;
  vector<string> paramTypes;
  bool already_AddRefed;
  Type *outparamType;
};

typedef list<OutParamInfo> outparam_list;

class OutparamDel : public ExpressionVisitor {
public:
  OutparamDel(Patcher &patcher, OutparamdelCmd const &cmd): 
    patcher(patcher),
    cmd(cmd),
    currentOpi(NULL),
    outmostExpression(NULL),
    outmostStatement(NULL),
    str_NS_ENSURE_ARG_POINTER(globalStrTable("NS_ENSURE_ARG_POINTER")),
    currentFunction(NULL),
    s_skippedTopNS_ENSURE_ARG_POINTER(NULL)
  {
    readParams(outparams, cmd.outparamFile);
  }
  virtual bool visitTypeSpecifier(TypeSpecifier *ts);
  virtual bool visitFunction(Function *f);
  virtual void postvisitFunction(Function *f);
  virtual bool visitS_return(S_return *e);
  virtual bool visitE_funCall(E_funCall *e);
  virtual bool visitExpression(Expression *e);
  virtual void postvisitExpression(Expression *e);
  virtual bool visitStatement(Statement *s);
  virtual bool visitS_if(S_if *s);
  virtual bool visitS_while(S_while *s);
  virtual bool visitS_for(S_for *s);
  virtual bool visitS_doWhile(S_doWhile *s);
  virtual void postvisitStatement(Statement *s);
  virtual bool visitE_deref(E_deref *e);
  virtual void postvisitE_variable(E_variable *e);
private:
  void readParams(outparam_list &outparams, string const &file);
  string getOutparamWithCast(Expression *inExpr, 
                             string &strGetterAddrefsCast,
                             string &strOutparamExpr,
                             bool &outGetterAddRefs,
                             OutParamInfo const &opi);
  void early_visitMR_decl_and_func(Member *m);
  OutParamInfo* isMatch(D_func *df, Variable *member, ASTTypeId *&outparam,
			Type *thisType, SourceLoc loc);
  // check callsites
  OutParamInfo* isMatchingCall(E_funCall *e);
  str_pair* rewriteFunctionDeclaration(OutParamInfo const &opi,
				      unsigned int paramCount,
				      CPPSourceLoc cslFunctionNameStart,
				      CPPSourceLoc cslStart,
				      CPPSourceLoc cslParamStart,
				      //  using middle,because don't have endloc for that
				      CPPSourceLoc cslParamMiddle,
				      CPPSourceLoc cslEnd);
  bool isDanglingStatement();
  MacroUndoEntry* doFollowedByNS_ENSURE_SUCCESS(S_expr *sexpr_assign, S_decl *sdecl, int &outCounter);
  string macroParam2String(MacroDefinition *md);
private:
  outparam_list outparams;
  Patcher &patcher;
  OutparamdelCmd const &cmd;
  // this is empty unless inside of a body of func that's being rewritten
  OutParamInfo *currentOpi;
  outparam_map mapOutparams;
  // used to trigger rewrites for things that act upon ret vals
  Expression *outmostExpression;
  Statement *outmostStatement;
  const StringRef str_NS_ENSURE_ARG_POINTER;
  Function *currentFunction;
  Statement *s_skippedTopNS_ENSURE_ARG_POINTER;
};

void OutparamDel::readParams(outparam_list &outparams, string const &file) {
  ifstream f(file.c_str());
  if (!f.is_open()) {
    cerr << "Couldn't open configuration file '"<<file<<"'" << endl;
    exit(1);
  }

  string line;
  unsigned int lineNum = 0;
  while (getline(f, line)) {
    lineNum++;
    OutParamInfo opi;
    string::size_type old_n, n;

    if (line.empty() || line[0] == '#') continue;

    // skip dehydra location info
    n = line.find(" ");
    if (n != string::npos) line.erase(0, n + 1);

    n = line.find("::");
    if(n == string::npos)
      goto error;
    opi.className = globalStrTable(line.substr(0, n).c_str());
    n = line.find('(', old_n = n + 2);
    if(n == string::npos)
      goto error;
    opi.memberName = line.substr(old_n, n - old_n);
    n = line.find(')', old_n = n + 1);
    if(n == string::npos)
      goto error;
    string paramTypes = line.substr(old_n, n - old_n);
    string::size_type n2 = paramTypes.find(',');
    string::size_type old_n2 = 0;
    while(old_n2 < paramTypes.size()) {
      if (n2 == string::npos) n2 = paramTypes.size();
      opi.paramTypes.push_back(paramTypes.substr(old_n2, n2 - old_n2));
      old_n2 = n2 + 1;
      n2 = paramTypes.find(',', old_n2);
    }
    n = line.find(',', old_n = n + 2);
    if (n == string::npos)
      n = line.size();
    string param = line.substr(old_n, n - old_n);
    opi.param = atoi(param.c_str());
    if (n != line.size())
      opi.already_AddRefed = line.substr(n+1) == "already_AddRefed";
    if (cmd.debug) {
      cerr << "Parsed " << opi.toString() << endl;
    }

    outparams.push_back(opi);
  }
  f.close();
  return;
 error:
  f.close();
  cerr << file << ":" << lineNum << ": Parse error, invalid input format:"
       << line << endl;
  exit(1);
}

void Outparamdel::process(OutparamdelCmd const &cmd) {
  Patcher patcher;
  OutparamDel od(patcher, cmd);

  foreachSourceFile {
    File *file = files.data();
    patcher.setFile(file->name);
    
    TranslationUnit *unit = file2unit.get(file);
    unit->traverse(od.loweredVisitor);
  }
}

bool OutparamDel::visitTypeSpecifier(TypeSpecifier *ts) {
  TS_classSpec *sp = ts->ifTS_classSpec();
  if (!sp) return true;
  for (ASTListIterNC<Member> i(sp->members->list);
       ! i.isDone();
       i.adv()) {
    Member *m = i.data();
    switch(m->kind()) {
    case Member::MR_DECL:
    case Member::MR_FUNC:
      early_visitMR_decl_and_func(m);
    default:
      break;
    }
  }
  return true;
}

/* called from visitTypeSpecifier before any code using these member signatures
   is encountered */
void OutparamDel::early_visitMR_decl_and_func(Member *m) {
  DeclFlags dflags;
  Declarator *d = NULL;
  // non-NULL iff MR_decl
  Declaration *dn = NULL;
  if (MR_decl *mdecl = m->ifMR_decl()) {
    dn = mdecl->d;
    d = dn->decllist->first();
    dflags = dn->dflags;
  } else if (MR_func *mfunc = m->ifMR_func()) {
    d = mfunc->f->nameAndParams;
    dflags = mfunc->f->dflags;
  }

  if (!d) return;
  D_func *df = d->decl->ifD_func();

  Variable *v = d->var;
  // these are from the type system
  if (!d->type->isFunctionType()) return;
  
  FunctionType *f = d->type->asFunctionType();
  SObjListIter<Variable> typedParams(f->params);
  ASTTypeId *outparam = NULL;
  Type *thisType = NULL;
  if (!typedParams.isDone()) thisType = typedParams.data()->type;
  OutParamInfo *opi = isMatch(df, v, outparam, thisType, m->loc);
  if ( !(opi && outparam)) return;
  // not sure if this is the right place to initialize this
  opi->outparamType = f->params.nth(opi->param + (int)f->isMethod())->type;

  mapOutparams[v] = opi;
  opi->isVirtual = dflags & DF_VIRTUAL;
  
  // only do MR_decl. MR_funcs get rewriten in visitFunction
  if (dn) {
    rewriteFunctionDeclaration(*opi, f->params.count() - 1,
			       dn->spec->asTS_name()->name->loc,
			       d->decl->loc,
			       outparam->spec->loc,
			       outparam->decl->decl->loc,
			       m->endloc);
  }
}

// Based on a func with the same name in squash_expr_rewriter.cc
static bool matchesType(StringRef type, Type *t, bool checkAncestors) {
  static StringRef strStar = globalStrTable("*");
  if (type == strStar) return true;
  if (t->isPtrOrRef() || t->isArrayType()) {
    return matchesType(type, t->getAtType(), checkAncestors);
  }
  if (!t->isCompoundType()) {
    return false;
  }
  CompoundType *ct = t->asCompoundType();
  TS_classSpec *c = ct->syntax;
  if (type == c->name->getName())
    return true;
  
  if (!checkAncestors)
    return false;
  for (FakeList<BaseClassSpec > *bs = c->bases;
       !bs->isEmpty();
       bs = bs->butFirst()) {
    BaseClassSpec *base = bs->first();
    if(matchesType(type, base->type->selfType, true))
      return true;
  }
  return false;
}

// Check function declaration if it is a rewrite candidate
OutParamInfo* OutparamDel::isMatch(D_func *df, 
				   Variable *member,
				   ASTTypeId *&outparam,
				   Type *thisType, SourceLoc loc) {
  for (outparam_list::iterator i = outparams.begin();
       i != outparams.end();
       ++i) {
    if (member->name != i->memberName) continue;
    
    // type of "this" parameter is the same as the class name
    if (! matchesType(i->className, thisType, i->isVirtual)) {
      continue; 
    }
    unsigned int param = i->param;
    for (FakeList<ASTTypeId>* params = df->params;
	 ! params->isEmpty();
	 params = params->butFirst()) {
      ASTTypeId *p = params->first();
      if (param--) continue;
      outparam = p;
      break;
    } 
    if (!outparam) continue;

    // now check if this is the correct overload
    FunctionType *t = skipPtrOrRef(member->type)->asFunctionType();

    SObjListIter<Variable> params(t->params);
    if (t->isMethod()) params.adv();

    for (param = 0; 
         !params.isDone() && param < i->paramTypes.size();
         params.adv(), param++) {
      Type *t = params.data()->type;
      sm::string typeName = t->toString();
      if (i->paramTypes[param] != typeName.c_str()) {
        if (cmd.debug) {
          cerr << loc << ": In param #" << param << " type '" 
               << i->paramTypes[param] << "'" << " != " 
               << "'" << typeName << "'" << endl;
        }
        break;
      }
    }

    // if param types don't match, move on
    if (param < i->paramTypes.size()) {
      if (cmd.debug) {
        cerr << loc << ": " << i->className << "::" << i->memberName
             << " doesn't match param types" << endl;
      }
      continue;
    }
    return &(*i);
  }
  return NULL;
}

str_pair* OutparamDel::rewriteFunctionDeclaration(OutParamInfo const &opi,
					     unsigned int paramCount,
					     CPPSourceLoc cslStart,
					     CPPSourceLoc cslFunctionNameStart,
					     CPPSourceLoc cslParamType,
					     //  using middle,because don't have endloc for that
					     CPPSourceLoc cslParamName,
					     CPPSourceLoc cslEnd) {
  CPPSourceLoc *validateLocs[] = {&cslStart, &cslFunctionNameStart, 
				&cslParamType, &cslParamName, &cslEnd};
  // this is non-NULL if method is decorated with NS_IMETHOD[IMP]
  MacroUndoEntry *macroNS_IMETHOD = NULL;
  for (unsigned int i = 0; 
       i < sizeof(validateLocs)/sizeof(validateLocs[0]);
       i++) {
    if (validateLocs[i]->hasExactPosition()) continue;
    if (validateLocs[i] == &cslStart) {
      macroNS_IMETHOD = cslStart.macroExpansion;
      while (macroNS_IMETHOD->parent) macroNS_IMETHOD = macroNS_IMETHOD->parent;

      continue;
    }
    cerr << toString(validateLocs[i]->loc()) << ": " << opi.memberName
	 <<" declared within " 
	 << validateLocs[i]->macroExpansion->name << endl;;
    return NULL;
  }
  // first strip the outparam
  PairLoc pairMemberToOutparam(cslFunctionNameStart, cslParamType);
  PairLoc pairOutparamToMember(cslParamName, cslEnd);

  string strMemberToOutparam = patcher.getRange(pairMemberToOutparam);
  string::size_type pos = strMemberToOutparam.find_last_of("(,");
  xassert(pos != string::npos);

  // Erase text prior to outparam.
  // don't erase '(', but do erase ',' and everything else
  if (strMemberToOutparam[pos] == '(') {
    pos++;
  }
  strMemberToOutparam.erase(pos);
  string strOutparamToMember = patcher.getRange(pairOutparamToMember);
  pos = strOutparamToMember.find_first_of(",)");
  xassert(pos != string::npos);
  
  // if outparam is the first paramter out of many, drop the comma after it
  if (paramCount > 1 && opi.param == 0) {
    do {
      pos++;
      // also skip whitespace after comma
    } while (pos < strOutparamToMember.size() && isspace(strOutparamToMember[pos]));
  }
  strOutparamToMember.erase(0,pos);
  string strFunc = strMemberToOutparam + strOutparamToMember;

  // now figure out type & name of outparam
  string strOutparamType = patcher.getRange(PairLoc(cslParamType, cslParamName));
  string strOutparamName = patcher.getRange(PairLoc(cslParamName, cslEnd));
  
  pos = strOutparamName.find_first_of(",)");
  xassert(pos != string::npos);
  // get rid of trailing spaces
  while(pos && isspace(strOutparamName[pos - 1])) pos--;
  strOutparamName.erase(pos);
  xassert(pos != string::npos);
  pos = strOutparamName.find_last_of("*&");
  xassert(pos != string::npos);
  strOutparamType += strOutparamName.substr(0,pos);
  pos = strOutparamName.find_last_of(" &*");
  xassert(pos != string::npos);
  strOutparamName.erase(0, pos+1);
  
  if (!cslStart.hasExactPosition()) cslStart.overrideLoc(cslStart.loc());

  string strFuncType = patcher.getRange(PairLoc(cslStart, cslFunctionNameStart));
  pos = strFuncType.size() - 1;
  while (isspace(strFuncType[pos])) {
    pos--;
  }
  
  string strRetType = strOutparamType;
  if (opi.already_AddRefed) {
    string::size_type spos = strRetType.find_last_of("&*");
    xassert(spos != string::npos);
    for(;spos && isspace(strRetType[spos-1]);--spos);
    strRetType.erase(spos);
    strRetType = "already_AddRefed<" + strRetType + ">";
  }
  if (macroNS_IMETHOD)
    strRetType = macroNS_IMETHOD->name +("_(" + strRetType) + ")";
  // preserve whitespace after the type
  patcher.printPatch(strRetType + strFuncType.substr(pos + 1) + strFunc,
		     PairLoc(cslStart, cslEnd));

  return new str_pair(strOutparamType, strOutparamName);
}


struct VarFinderVisitor : public ASTVisitor {
  OutParamInfo *opi;
  VarFinderVisitor(OutParamInfo *opi):opi(opi) {
  }
  void postvisitExpression(Expression *e) {
    E_variable *ev = e->ifE_variable();
    if (ev && ev->var && ev->var->name == opi->outparamName)
      throw true;
  }
};

//convert aFoo into foo
static string prettyRetvarName(string const &orig) {
  if (orig.size() > 1 && orig[0] == 'a') {
    string ret = orig;
    ret.erase (0, 1);
    ret[0] = tolower(ret[0]);
    // avoid keywords
    if (ret == "return")
      return "ret";
    return ret;
  } 
  return "__" + orig;
}

bool OutparamDel::visitFunction(Function *f) {
  PQName  *pqname = f->nameAndParams->getDeclaratorId();
  PQ_qualifier *q = pqname->ifPQ_qualifier();
  // either there is a qualifier or we are in an MR_function so pqname is a PQ_name
  PQ_name *pq = (q ? q->rest : pqname)->ifPQ_name();
  if (!pq) return true;
  D_func *df = f->nameAndParams->decl->ifD_func();
  ASTTypeId *outparam = NULL;

  FunctionType *funcType = f->funcType;
  if (!funcType->isMethod()) return true;

  OutParamInfo *opi = isMatch(df, f->nameAndParams->var, outparam, 
			      funcType->getClassOfMember()->selfType, pq->loc);
  currentOpi = opi;
  if (!opi) return true;

  SourceLoc bodyLoc = SL_UNKNOWN;
  {
    StringRef file;
    int line;
    int col;
    sourceLocManager->decodeLineCol(f->body->loc, file, line, col);
    bodyLoc = sourceLocManager->encodeLineCol(file, line, col + 1);
  }
  CPPSourceLoc cslType(f->retspec->loc);
  CPPSourceLoc endcslType(f->nameAndParams->decl->loc);
  CPPSourceLoc cslOutparam(outparam->spec->loc);
  CPPSourceLoc cslOutparamName(outparam->decl->decl->loc);
  CPPSourceLoc endcsl(bodyLoc);
  
  auto_ptr<str_pair> type_name( 
    rewriteFunctionDeclaration(*opi, 
			       f->nameAndParams->type->asFunctionType()->params.count() - 1, 
			       cslType,
			       endcslType,
			       cslOutparam,
			       cslOutparamName,	      
			       endcsl));
  if (!type_name.get())
    return true;
  opi->outparamName = type_name->second;
  opi->retvarName = prettyRetvarName(type_name->second);
  // don't bother rewriting more if the func has no body
  
  ASTList<Statement> &ls = f->body->asS_compound()->stmts;
  if (ls.isEmpty()) return true;
  
  StringRef file;
  int line;
  int col;
  CPPSourceLoc cslIndent(ls.first()->loc);
  // we don't care if cslIndent points within a macro
  cslIndent.overrideLoc(cslIndent.loc());
  // Don't need an exact position if we are just
  // inserting stuff infront
  // xassert(cslIndent.hasExactPosition());
  sourceLocManager->decodeLineCol(cslIndent.loc(), file, line, col);

  string strIndent = patcher.getLine(line, file).substr(0, col - 1);
  stringstream ss;
  ss <<  type_name->first << " " <<  opi->retvarName << " = nsnull;";
  ss << "\n" << strIndent <<type_name->first << "* " 
     << type_name->second << " = &" << opi->retvarName << ";";

  s_skippedTopNS_ENSURE_ARG_POINTER = NULL;
  if (cslIndent.macroExpansion
      && cslIndent.macroExpansion->name == str_NS_ENSURE_ARG_POINTER) {
    try {
      VarFinderVisitor vfv(opi);
      ls.first()->traverse(vfv);
    } catch (bool b) {
      s_skippedTopNS_ENSURE_ARG_POINTER  = ls.first();
    }
  }
  if (s_skippedTopNS_ENSURE_ARG_POINTER) {
    // if the first part of func is a pesky NS_ENSURE_ARG_POINTER
    // replace it altogether instead of prepending stuff
    CPPSourceLoc endloc = cslIndent;
    endloc.overrideLoc(cslIndent.macroExpansion->preEndLoc);
    patcher.printPatch(ss.str(), PairLoc(cslIndent, endloc));
  } else {
    ss << "\n" << strIndent;
    patcher.insertBefore(cslIndent, ss.str());
  }
  currentFunction = f;
  return true;
}

void OutparamDel::postvisitFunction(Function *f) {
  currentOpi = NULL;
  currentFunction = NULL;
  s_skippedTopNS_ENSURE_ARG_POINTER = NULL;
}

bool OutparamDel::visitS_return(S_return *s) {
  if (!currentOpi) return true;

  Expression *e = s->expr->expr;
  CPPSourceLoc csl(e->loc);
  CPPSourceLoc endcsl(e->endloc);
  CPPSourceLoc cslReturn(s->loc);
  bool lostInMacro = !(csl.hasExactPosition() && endcsl.hasExactPosition());
  if (!cslReturn.hasExactPosition() && lostInMacro) {
    // in case NS_ENSURE_TRUE(..., err_code)
    // replace err_code with nsnull
    static StringRef strEnsureTrue = globalStrTable("NS_ENSURE_TRUE");
    if (csl.macroExpansion == endcsl.macroExpansion
        && csl.macroExpansion->name == strEnsureTrue) {
      // the macros have the original position
      // no need to recalculate
      csl.overrideLoc(csl.loc());
      endcsl = csl;
      endcsl.overrideLoc(csl.macroExpansion->preEndLoc);
      PairLoc pairMacro(csl, endcsl);
      string strMacro = patcher.getRange(pairMacro);
      
      string::size_type pos = strMacro.rfind(',');
      xassert(pos != string::npos);
      while(isspace(strMacro[pos + 1])) pos++;

      string::size_type endpos = strMacro.find(')', pos + 1);
      patcher.printPatch(strMacro.substr(0, pos + 1)
                         + "nsnull"
                         + strMacro.substr(endpos), pairMacro);
      return false;
    }
    CPPSourceLoc &bad = !csl.hasExactPosition() ? csl : endcsl;
    CPPSourceLoc firstStatementLoc(currentFunction->body->asS_compound()->stmts.first()->loc);

    cerr << toString(bad.loc())
	 << ": getter's return statement is within an unsupported macro: " 
	 << csl.macroExpansion->name << endl;
    return true;
  } else if (cslReturn.hasExactPosition() && lostInMacro) {
    MacroUndoEntry *macro = (csl.macroExpansion ? csl : endcsl).macroExpansion;
    cerr << toString(cslReturn.loc())
	 << ": getter's return expression within a non-obvious macro. Guessing that "
	 << macro->name << " is some error \"constant\"" << endl;
    csl.overrideLoc(macro->preStartLoc);
    endcsl.overrideLoc( macro->preEndLoc);
    patcher.printPatch("nsnull", PairLoc(csl, endcsl));
    return false;
  }

  PairLoc pairRet(csl, endcsl);
  // TODO, check that nsresult is the cast expr
  if (E_cast *c = e->ifE_cast()) {
    e = c->expr;
  }
  E_intLit *i = e->ifE_intLit();

  stringstream ss;  
  static StringRef str_NS_ENSURE_SUCCESS_param2 =
    globalStrTable("NS_ENSURE_SUCCESS:0-1");
  // is return is being used to signal an error
  if ((i && i->i != 0)
      || (csl.macroExpansion
          && csl.macroExpansion->name == str_NS_ENSURE_SUCCESS_param2)) {
    ss << "nsnull";
  } else if (!i) {
    E_funCall *func = e->ifE_funCall();
    if (func && isMatchingCall(func)) {
      Restorer<Statement*> restorer(outmostStatement, s);
      visitE_funCall(func);
      return false;
    }
    ss << patcher.getRange(pairRet) << " == NS_OK ? "
       << currentOpi->retvarName << " : nsnull";
  } else {
    ss << currentOpi->retvarName;
  }
  if (cmd.debug) 
    ss << "/*" << __FILE__ << ":" << __LINE__ << "*/";
  
  patcher.printPatch(ss.str(), pairRet);
  return false;
}

static Expression* unrollParam(Expression *inExpr, Expression *&outOuterExpr,
                               OutParamInfo const &opi) {
  Expression *outExpr = inExpr;
  outOuterExpr = outExpr;
  // loop does "recursion"
  while(true) {
    switch(outExpr->kind()) {
    case Expression::E_VARIABLE:
    case Expression::E_FIELDACC:
      return outExpr;
    case Expression::E_ADDROF:
      outOuterExpr = outExpr;
      outExpr = outExpr->asE_addrOf()->expr;
      break;
    case Expression::E_CAST:
      outOuterExpr = outExpr;
      outExpr = outExpr->asE_cast()->expr;
      break;
    case Expression::E_KEYWORDCAST:
      outExpr = outExpr->asE_keywordCast()->expr;
      break;
    case Expression::E_FUNCALL:
      {
	E_funCall *f = outExpr->asE_funCall();
	static const StringRef strGetterAddRefs = globalStrTable("getter_AddRefs");
	static const StringRef strGetter_Transfers = globalStrTable("getter_Transfers");
	E_variable *v = f->func->ifE_variable();
	if (v && (v->var->name == strGetterAddRefs ||  v->var->name == strGetter_Transfers)) {
	  outOuterExpr = outExpr;
	  outExpr = f->args->first()->expr;
	  break;
	} else if (v) {
          cerr << opi.memberName << " used with a weird wrapper: "
               << v->var->name << endl;
        }
        
      }
    default:
      xassert(false && "Unhandled outparameter argument expression");
    }
  }
  
  return outExpr;
}

/* to is the outparam type..from is the variable passed into it 
   in the original code outparam type could've been null, so need to upcast to from
 */
static string maybeCast(Type *varType, bool finalType, Type *paramType, SourceLoc loc) {
  string strType;
  if (varType->isReferenceType())
    varType = varType->getAtType();
  // get rid of extra *
  if (!finalType)
    varType = varType->getAtType();
  if (varType->toString() != paramType->toString()) {
    cerr << loc <<  ":Failed to deduce case through type system. Need '"
         << paramType->toString() 
         << "' got '" << varType->toString() << "'"<< endl;
    strType = "(" + string(varType->toString().c_str()) + ") ";
  }
  return strType;
}

// Takes outparam expression and returns LHS of form outparam = (foo)
string OutparamDel::getOutparamWithCast(Expression *inExpr,
                                        string &strGetterAddrefsCast,
                                        string &strOutparamExpr,
                                        bool &outGetterAddRefs,
                                        OutParamInfo const &opi)
{
  // this will contain the most important outer expr
  // ie E_funCall, E_*cast
  Expression *outerExpr = NULL;
  //outExpr is the variable within
  Expression *outExpr = unrollParam(inExpr, outerExpr, opi);
  string strType;
  outGetterAddRefs = false;
  Type *retType = opi.outparamType->getAtType();
  strGetterAddrefsCast.resize(0);
  strOutparamExpr.resize(0);
  
  switch(outerExpr->kind()) {
  case Expression::E_ADDROF:
    strType = maybeCast(outExpr->type, true, retType, inExpr->loc);
    break;
  // this is for getter_AddRefs
  case Expression::E_FUNCALL:
    {
      // work with inExpr, since it's a simplified version in case of getter_AddRefs
      TemplateInfo *ti = 
	outerExpr->type->asCVAtomicType()->atomic->asCompoundType()->templateInfo();
      static const StringRef str_nsGetterAddRefs = globalStrTable("nsGetterAddRefs");
      static const StringRef str_nsRefPtrGetterAddRefs = 
        globalStrTable("nsRefPtrGetterAddRefs");
      static const StringRef str_nsAutoPtrGetterTransfers = 
        globalStrTable("nsAutoPtrGetterTransfers");
      static const StringRef str_nsAutoArrayPtrGetterTransfers = 
        globalStrTable("nsAutoArrayPtrGetterTransfers");
      StringRef var_name = ti->var->name;
      xassert((var_name == str_nsGetterAddRefs
               || var_name == str_nsRefPtrGetterAddRefs
               || var_name == str_nsAutoPtrGetterTransfers
               || var_name == str_nsAutoArrayPtrGetterTransfers)
	      && ti->arguments.count() == 1);
      STemplateArgument *ta = ti->arguments.first();
      xassert(ta->isType());
      Type *innerType = ta->getType();
      strType = innerType->toString().c_str();
      outGetterAddRefs = true;
    }
    break;
  case Expression::E_VARIABLE:
  case Expression::E_CAST:
  case Expression::E_KEYWORDCAST:
    strType = maybeCast(outExpr->type, false, retType, inExpr->loc);
    //value needs to be dereferenced on LHS
    strOutparamExpr += "*";
    break;
  default:
    xassert(false && "unhandled outparamter argument..when looking for type");
    break;
  }
  
  CPPSourceLoc cslOutparam(outExpr->loc);
  CPPSourceLoc endcslOutparam(outExpr->endloc);
  strOutparamExpr += patcher.getRange(PairLoc(cslOutparam, endcslOutparam));
  string strExpr = strOutparamExpr;
  strExpr.append(" = ");
  // then we aren't rewriting into identical code
  // and creating a leak
  if (outGetterAddRefs && !opi.already_AddRefed) {
    OutParamInfo myopi = opi;
    myopi.already_AddRefed = true;
    cerr << outExpr->loc << ": Rewrite needs to specify already_AddRefed to avoid memory leak: "
         << myopi.toString() << endl;
  }
  if (!outGetterAddRefs) {
    strExpr.append(strType);
    //don't need to use special version of getter if types already match
  } else if (opi.paramTypes[opi.param].find(strType) != 0) {
    strGetterAddrefsCast = strType;
  }
  return strExpr;
}

OutParamInfo* OutparamDel::isMatchingCall(E_funCall *e) {
  E_fieldAcc *f = e->func->ifE_fieldAcc();
  if (!f || !f->field) return NULL;
  
  outparam_map::iterator it = mapOutparams.find(f->field);
  if (it == mapOutparams.end()) return NULL;

  return it->second;
}

// rewrite callers of outparamed func...might be recursive
bool OutparamDel::visitE_funCall(E_funCall *e) {
  OutParamInfo *opi = isMatchingCall(e);
  if (!opi) return true;

  unsigned int counter = opi->param;
  Expression *paramExpr = NULL;
  // point at the argument in front of outparam
  Expression *priorParamExpr = NULL;
  for (FakeList<ArgExpression>* args = e->args;
	 ! args->isEmpty();
	 args = args->butFirst()) {
    if (0 != counter--) {
      priorParamExpr = args->first()->expr;
      continue;
    }
    paramExpr = args->first()->expr;
    break;
  }
  
  CPPSourceLoc cslStart(e->loc);
  CPPSourceLoc cslParamStart(paramExpr->loc);
  CPPSourceLoc cslParamEnd(paramExpr->endloc);
  CPPSourceLoc cslEnd(e->endloc);
  // first strip the outparam
  PairLoc pairFuncToOutparam(cslStart, cslParamStart);
  PairLoc pairParamToEnd(cslParamEnd, cslEnd);

  MacroUndoEntry *macro = NULL;
  if (!pairFuncToOutparam.hasExactPosition()) {
    macro = pairFuncToOutparam.getMacro();
  } else if (!pairParamToEnd.hasExactPosition()) {
    macro = pairParamToEnd.getMacro();
  }
  if (macro) {
    cerr << toString(cslStart.loc()) << ": "
	 << opi->memberName << " called within "<< macro->name << endl;
    return false;
  }

  // "expressions" of nsresult type end up really ugly
  // split up the two "easy" into separate rv assignment and return
  bool splitStatement = false;
  // sexpr != NULL && splitStatement -> this is a simple assignment
  S_expr *sexpr = NULL;
  S_decl *sdecl = NULL;
  try {
    // not in the body of the function being de-outparamed
    xmatch (!currentOpi);
    // must be either a return statement or an assignment
    if (!SOME(outmostStatement)->isS_return()) {
      sexpr = outmostStatement->ifS_expr();
      sdecl = outmostStatement->ifS_decl();
      if (sexpr)
        xmatch(sexpr->expr->expr->asE_assign()->src == e);
      else if (sdecl)
        xmatch(SOME(sdecl->decl->decllist)->first()->init->asIN_expr()->e == e);
      else
        xmatch(false);
    }
    splitStatement = true;
  } catch (x_match&) {
    
  }
  PairLoc pairExpression(splitStatement ? outmostStatement->loc : cslStart,
                         splitStatement ? outmostStatement->endloc : cslEnd);
  // Give up if expression within return is free of macros, but not entire return
  // above guards satisfy first clause
  if (!pairExpression.hasExactPosition()) {
    pairExpression = PairLoc(cslStart, cslEnd);
    xassert(pairExpression.hasExactPosition());
    splitStatement = false;
  }
  string strFuncToOutparam = patcher.getRange(pairFuncToOutparam);
  string::size_type pos = strFuncToOutparam.find_last_of("(,");
  xassert(pos != string::npos);

  // Erase text prior to outparam.
  // don't erase '(', but do erase ',' and everything else
  if (strFuncToOutparam[pos] == '(') {
    pos++;
  }
  strFuncToOutparam.erase(pos);
  string strOutparamToEnd = patcher.getRange(pairParamToEnd);
  pos = strOutparamToEnd.find_first_of(",)");
  xassert(pos != string::npos);
  
  // if outparam is the first paramter out of many, drop the comma after it
  if (e->args->count() > 1 && opi->param == 0) {
    do {
      pos++;
      // also skip whitespace after comma
    } while (pos < strOutparamToEnd.size() && isspace(strOutparamToEnd[pos]));
  }
  strOutparamToEnd.erase(0, pos);

  string strGetterAddrefsCast;
  string strOutparamExpr;
  bool getterAddrefs;
  string strLHS = getOutparamWithCast(paramExpr, strGetterAddrefsCast,
                                      strOutparamExpr, getterAddrefs, *opi);
  // holds replacement text
  stringstream ss;
  ss << strLHS << strFuncToOutparam << strOutparamToEnd;
  // raw pointers don't use getterAddrefs, this validates that codde
  // nsCOMPtrs don't use getterAddrefs when they want the referece count to go up(ugly yeah). check :)
  if (opi->already_AddRefed && !getterAddrefs)
    ss << ".get()";
  bool outerExpHandled = false;

  if (!strGetterAddrefsCast.empty()) {
    CPPSourceLoc cslEoFunc(e->func->endloc);
    PairLoc pairFunc(cslStart, cslEoFunc);
    string strFuncToParam = patcher.getRange(pairFunc); 
    ss.str("");
    ss << strLHS;
      
    // this is a QI-specific hack
    if (opi->memberName == "QueryInterface") {
      ss << strFuncToParam << "AAR<" << strGetterAddrefsCast << ">";
    } else {
      ss << "(" << strGetterAddrefsCast << "*)" << strFuncToParam;
    }
    if (priorParamExpr) {
      CPPSourceLoc endcslPrior(priorParamExpr->endloc);
      if (endcslPrior.macroExpansion) {
        endcslPrior.overrideLoc(endcslPrior.macroExpansion->preEndLoc);
      }
      ss << patcher.getRange(PairLoc(cslEoFunc, endcslPrior));
    } else {
      ss << "(";
    }
    ss << strOutparamToEnd;
  } else if (currentOpi && outmostStatement && outmostStatement->isS_return()) {
    ss.str("");
    ss << strFuncToOutparam << strOutparamToEnd;
    outerExpHandled = true;
  } else {
    try {
      Variable *v =
        SOME(SOME(outmostExpression)->asE_funCall()->func->asE_variable()->var);
      static const StringRef strNS_FAILED = globalStrTable("NS_FAILED");
      static const StringRef strNS_SUCCEEDED = globalStrTable("NS_SUCCEEDED");

      xmatch(v->name == strNS_FAILED || v->name == strNS_SUCCEEDED);
      string strExpr = ss.str();
      ss.str("");
      ss << (v->name == strNS_FAILED ? "! (" : "!! (");
      ss << strExpr << ")";
      cslStart = CPPSourceLoc(outmostExpression->loc);
      cslEnd = CPPSourceLoc(outmostExpression->endloc);
      outerExpHandled = true;
    } catch (x_match&) {
    }
  } 
  UnboxedPairLoc unboxedPair(pairExpression);
  if (splitStatement) {
    // this would be "return " or "lhs = "
    PairLoc pairLHS(outmostStatement->loc, cslStart);
    if (!pairLHS.hasExactPosition()) {
      cerr << toString(pairLHS.getMacro()->postStartLoc) << ": " 
           <<  pairLHS.getMacro()->name << " prevents rewriting" << endl;
      return false;
    }
    string strIndent = patcher.getLine(unboxedPair.first.line,
                                            unboxedPair.file);
    strIndent.erase(unboxedPair.first.col - 1);
    // hack to avoid dealing with having stuff before "return" issue
    for(string::size_type i = 0; i < strIndent.size();i++)
      if (!isspace(strIndent[i]))
        strIndent[i]=' ';
    
    const bool danglingStatement = isDanglingStatement();
    // abtrary !0 value, used to figure out when rv is no longer used
    // such that 'nsresult rv =' part can be nuked
    int usageCounter = -1;
    MacroUndoEntry *followedByNS_ENSURE_SUCCESS = 
      doFollowedByNS_ENSURE_SUCCESS(sexpr, sdecl, usageCounter);
    // do not do anything is there are move uses of 'nsresult rv'
    if (sdecl && usageCounter != 0)
      followedByNS_ENSURE_SUCCESS = NULL;

    if (followedByNS_ENSURE_SUCCESS) {
      unboxedPair.second.set(followedByNS_ENSURE_SUCCESS->preEndLoc);
    }
    string strExpr = ss.str();
    ss.str("");
    string strIndentIndent = strIndent;
    if (danglingStatement) {
      strIndentIndent += "  ";
      ss << "{" << "\n" << strIndentIndent ;
    }
    ss << strExpr << ";\n"
       << strIndentIndent;
    if (followedByNS_ENSURE_SUCCESS) {
      ss << "NS_ENSURE_TRUE(" << strOutparamExpr << ", "
         << cmd.outparamError << ")";//; will already exist
    } else {
      ss <<  patcher.getRange(pairLHS) << strOutparamExpr << " ? NS_OK : " 
         << cmd.outparamError << ";";
    }
    if (danglingStatement)
      ss << "\n" << strIndent << "}";
    outerExpHandled = true;
  }
  
  // rewriting everything that isnt a plain function call
  // to the compatibility ternary
  if (!outerExpHandled) {
    bool justOuparamFcall = false;
    try {
      justOuparamFcall = SOME(outmostStatement)->asS_expr()->expr->expr == e;
    } catch (x_match &) {
    }
    if ((outmostExpression || outmostStatement)
        && !justOuparamFcall) {
      string strExpr = ss.str();
      ss.str("");
      ss << "((" << strExpr << ") ? NS_OK : " << cmd.outparamError << ")";
    }
  }

  if (cmd.debug) {
    ss << "/*" << __FILE__ << ":" << __LINE__ << "*/";
  }
  
  patcher.printPatch(ss.str(), unboxedPair);
  return false;
}

bool OutparamDel::visitExpression(Expression *e) {
  if (ExpressionVisitor::visitExpression(e)) {
    if (!outmostExpression) outmostExpression = e;
    return true;
  }
  return false;
}

void OutparamDel::postvisitExpression(Expression *e) {
  if (outmostExpression == e) {
    outmostExpression = NULL;
  }
  ExpressionVisitor::postvisitExpression(e);
}

bool OutparamDel::visitStatement(Statement *s) {
  if (s == s_skippedTopNS_ENSURE_ARG_POINTER)
    return false;
  switch(s->kind()) {
  case Statement::S_EXPR:
  case Statement::S_DECL:
  case Statement::S_RETURN:
    if (!outmostStatement) {
      outmostStatement = s;
    }
  default:
    bool ret = ExpressionVisitor::visitStatement(s);
    // when ret is false postvisitStatement isn't called :(
    // gotta cleanup manually
    if (!ret && s == outmostStatement) outmostStatement = NULL;
    return ret;
  }
  return true;
}

bool OutparamDel::visitS_if(S_if *s) {
  // give special treatment to conditional expr
  if (s->cond) { 
    Restorer<Statement*> restorer(outmostStatement, s);
    s->cond->traverse(this->loweredVisitor); 
  }

  // rest is just like S_if::traverse
  if (s->thenBranch) { 
    s->thenBranch->traverse(this->loweredVisitor); 
  }
  if (s->elseBranch) { 
    s->elseBranch->traverse(this->loweredVisitor); 
  }
  return false;
}

bool OutparamDel::visitS_while(S_while *s) {
  // give special treatment to conditional expr
  if (s->cond) { 
    Restorer<Statement*> restorer(outmostStatement, s);
    s->cond->traverse(this->loweredVisitor); 
  }

  // rest is just like S_while::traverse
  if (s->body) { 
    s->body->traverse(this->loweredVisitor); 
  }
  return false;
}

bool OutparamDel::visitS_doWhile(S_doWhile *s) {
  // just like S_doWhile::traverse
  if (s->body) { 
    s->body->traverse(this->loweredVisitor); 
  }
  
  // give special treatment to conditional expr
  if (s->expr) { 
    Restorer<Statement*> restorer(outmostStatement, s);
    s->expr->traverse(this->loweredVisitor); 
  }
  return false;
}

bool OutparamDel::visitS_for(S_for *s) {
  // give special treatment to conditional expr
  { 
    Restorer<Statement*> restorer(outmostStatement, s);
    if (s->init) s->init->traverse(this->loweredVisitor);
    if (s->cond) s->cond->traverse(this->loweredVisitor);
  }  
  // rest is just like S_for::traverse
  if (s->after) { 
    s->after->traverse(this->loweredVisitor); 
  }
  if (s->body) { 
    s->body->traverse(this->loweredVisitor); 
  }
  return false;
}

void OutparamDel::postvisitStatement(Statement *s) {
  if (outmostStatement == s) outmostStatement = NULL;
  ExpressionVisitor::postvisitStatement(s);
}

/* Walk the statement stack to figure if we are in a 
if (foo)
   bla;
type situation where adding an extra statement without {} 
would be unsafe . Return true in that case
*/
bool OutparamDel::isDanglingStatement() {
  list<Statement*>::reverse_iterator i = statementStack.rbegin();
  //skip first element
  for (++i; i != statementStack.rend(); ++i) {
    S_compound *s = (*i)->ifS_compound();
    if (!s) continue;
    // >1 statement implies {}
    if (s->stmts.count() != 1)
      return false;
    // statement that might be dandling
    Statement *last = statementStack.back();
    //pigeonhole principle..no other nodes but me
    //implies it's me!
    //i'm pretty sure this isn't always valid due to E_statement
    //so assert :)
    xassert(last == s->stmts.first());
    // if locations match then s is a synthenic statement inserted
    // during elaboration
    return s->loc == last->loc && s->endloc == last->endloc;
  }
  xassert(false && "Should never get here");
}

string OutparamDel::macroParam2String(MacroDefinition *md) {
  // initialize these with bogus values
  CPPSourceLoc from(md->fromLoc);
  CPPSourceLoc to(from);
  // then override them with exact values
  from.overrideLoc(md->fromLoc);
  to.overrideLoc(md->toLoc);
  return patcher.getRange(PairLoc(from, to));
}

/* extract lhs' variable name from sexpr_assign, then check if the statement
   is followed by NS_ENSURE_SUCCESS(rv, rv)
   This also detects when nsresult rv; is no longer needed AND eliminates it in case of
   sexpr_assign.
*/
MacroUndoEntry* OutparamDel::doFollowedByNS_ENSURE_SUCCESS(S_expr *sexpr_assign, S_decl *sdecl, int &outCounter) {
  list<Statement*>::reverse_iterator i = statementStack.rbegin();
  Variable *v = NULL;
  Statement *sself = NULL;
  if (sexpr_assign) {
    sself = sexpr_assign;
    if (E_variable *evar = sexpr_assign->expr->expr->asE_assign()->target->ifE_variable())
      v = evar->var;
  } else if (sdecl) {
    sself = sdecl;
    v = sdecl->decl->decllist->first()->var;
  }
  if (!v)
    return NULL;

  bool prevWasSself = false;
  static StringRef str_NS_ENSURE_SUCCESS = globalStrTable("NS_ENSURE_SUCCESS");
  //skip first element
  for (++i; i != statementStack.rend(); ++i) {
    S_compound *s = (*i)->ifS_compound();
    if (!s) continue;
    for (ASTListIterNC<Statement> i(s->stmts);
         ! i.isDone();
         i.adv()) {
      Statement *s = i.data();
      if (prevWasSself) {
        if (!s->isS_doWhile() || s->loc == SL_UNKNOWN) return NULL;
        CPPSourceLoc csl(s->loc);
        if (!csl.macroExpansion 
            || csl.macroExpansion->name != str_NS_ENSURE_SUCCESS) return NULL;
        string firstArg = macroParam2String(csl.macroExpansion->params.first());
        string secondArg = 
          macroParam2String(csl.macroExpansion->params.nth(1));
        if (v->name == firstArg && firstArg == secondArg) {
          VarUsageCounter vuc(v);
          vuc.skipStatement(sself);
          vuc.skipStatement(s);
          statementStack.front()->traverse(vuc.loweredVisitor);
          outCounter = vuc.counter;
          if (!vuc.counter && vuc.sdecl) {
            PairLoc pair(vuc.sdecl->loc, vuc.sdecl->endloc);
            if (pair.hasExactPosition()) {
              patcher.printPatch(cmd.debug ?"/* nuked cos " __FILE__":"TOSTRING( __LINE__ )"*/" : "", pair);
            } else {
              cerr << toString(pair.first.loc()) 
                   << ": Could not remove declaration of " << v->name << endl;
            }
          }
          return csl.macroExpansion;
        }
      } else if (s == sself) {
        prevWasSself = true;
        continue;
      }
    }    
  }
  return NULL;
}

bool OutparamDel::visitE_deref(E_deref *e) {
  if (!currentOpi) return true;
  E_variable *v = e->ptr->ifE_variable();
  if (v && v->var && v->var->name == currentOpi->outparamName) {
    PairLoc ploc(e->loc, e->endloc);
    if (ploc.hasExactPosition()) {
      patcher.printPatch(currentOpi->retvarName, ploc);
      return false;
    }
  }
  return true;
}

void OutparamDel::postvisitE_variable(E_variable *v) {
  if (currentOpi && v->var && v->var->name == currentOpi->outparamName) {
    PairLoc ploc(v->loc, v->endloc);
    if (ploc.hasExactPosition()) {
      patcher.printPatch("&" + currentOpi->retvarName, ploc);
    }
  }
}
