// see License.txt for copyright and terms of use

#include "renamer_cmd.h"    // RenamerCmd
#include "oink.gr.gen.h"        // CCParse_Oink
#include "strutil.h"            // quoted
#include "oink_util.h"
#include "expr_visitor.h"
#include "squash_util.h"
#include <sstream>
#include <cppundolog.h>
#include "patcher.h"

using namespace std;

RenamerCmd *renamerCmd = NULL;

/* Can rename class names
 */
class RenameVisitor : public ExpressionVisitor {
public:
  RenameVisitor(Patcher &patcher): 
    patcher(patcher)
  {
  }

  void postvisitTypeSpecifier(TypeSpecifier *ts);
  void postvisitDeclarator(Declarator *);
  void postvisitPQName(PQName *);
  void postvisitBaseClassSpec(BaseClassSpec *);
  void renameAt(CPPSourceLoc loc, string const &renameTo, int offset = 0);
  bool isRenameClassTarget(Scope *scope);
  void postvisitE_variable(E_variable *v);
private:
  Patcher &patcher;
};

void RenamerRunner::run() {
  if (!renamerCmd->renameClass.from) {
    cerr << "Must specify a class to rename" << endl;
  }
  Patcher patcher;
  RenameVisitor renamer(patcher);
  foreachSourceFile {
    File *file = files.data();
    patcher.setFile(file->name);
    
    maybeSetInputLangFromSuffix(file);
    TranslationUnit *unit = file2unit.get(file);
    unit->traverse(renamer.loweredVisitor);
  }
}

void RenameVisitor::renameAt(CPPSourceLoc csl, string const &renameTo, int offset) {
  if (!csl.hasExactPosition()) {
    cerr << toString(csl.loc()) << ": " <<csl.macroExpansion->name
         << " macro prevents rename" << endl;
    return;
  }
  UnboxedLoc uloc;
  StringRef filename = uloc.set(csl.loc());
  uloc.col += offset;
  UnboxedLoc ulocEnd = uloc;
  ulocEnd.col += renamerCmd->renameClass.fromLen;
  UnboxedPairLoc upair(filename, uloc, ulocEnd);
  string existing = patcher.getRange(upair);
  if (existing != renamerCmd->renameClass.from) {
    cerr << toString(csl.loc()) << ": Expected '"  << renamerCmd->renameClass.from
         << "', got '" << existing << "'" ;
    if (csl.macroExpansion)
      cerr << ". Macro " << csl.macroExpansion->name << " may be to blame";
    cerr << endl;
    return;
  }
  patcher.printPatch(renameTo, upair);
}

/* Returns true when scope is that of a class member of the class being renamed */
bool RenameVisitor::isRenameClassTarget(Scope *s) {
  return (s && s->scopeKind == SK_CLASS
          && s->getTypedefName()->name == renamerCmd->renameClass.from
          && s->parentScope->scopeKind == SK_GLOBAL);
}

void RenameVisitor::postvisitTypeSpecifier(TypeSpecifier *ts) {
  if (TS_classSpec *cs = ts->ifTS_classSpec()) {
    try {
      PQ_name *name = SOME(cs->name)->asPQ_name();
      Variable *v = cs->ctype->getTypedefVar();
      xmatch(isRenameClassTarget(v->getDenotedScope()));
      renameAt(name->loc, renamerCmd->renameClass.toShort);
    } catch (x_match&) {
    }
  } else if (TS_name *n = ts->ifTS_name()) {
    if (isRenameClassTarget(n->var->getDenotedScope()))
      renameAt(n->name->loc, renamerCmd->renameClass.to);
  } 
}

void RenameVisitor::postvisitDeclarator(Declarator *d) {
  Variable *v = d->var;
  if (v->isImplicitMemberFunc()) return;
  
  // gets set in first try block
  PQName *pqname = NULL;
  // try to rename as class
  try {
    // filter out implicit/generated crap
    // filter everything but members of class being renamed
    pqname = SOME(d->decl->getD_func())->base->asD_name()->name;
    
    xmatch (isRenameClassTarget(v->scope));

    while (PQ_qualifier *qual = pqname->ifPQ_qualifier()) {
      if (qual->qualifier == renamerCmd->renameClass.from)
        renameAt(qual->loc, renamerCmd->renameClass.to);
      pqname = qual->rest;
    }
    PQ_name *name = pqname->asPQ_name();
    xmatch(name->name);
    // detect constructor
    if (name->name == renamerCmd->renameClass.from) {
      renameAt(name->loc, renamerCmd->renameClass.toShort);
    } else if (name->name[0] == '~' 
               && !strcmp(name->name + 1, renamerCmd->renameClass.from)) {
      renameAt(name->loc, renamerCmd->renameClass.toShort, 1);
    }
  } catch(x_match &) {
    //d->debugPrint(cerr, 0); 
    //cerr << endl;
  }

  // now try to rename as function
  if (pqname && v->scope && v->scope->scopeKind == SK_GLOBAL
      && v->name == renamerCmd->renameClass.from)
    renameAt(pqname->loc, renamerCmd->renameClass.toShort, 0);
}

void RenameVisitor::postvisitBaseClassSpec(BaseClassSpec *d) {
  if (isRenameClassTarget(d->type->getTypedefName()->getDenotedScope())) {
    renameAt(d->name->loc, renamerCmd->renameClass.to);
  }
}

void RenameVisitor::postvisitPQName(PQName *d) {
  if (PQ_qualifier *q = d->ifPQ_qualifier()) {
    if (q->qualifierVar
        && isRenameClassTarget(q->qualifierVar->getDenotedScope()))
      renameAt(q->loc, renamerCmd->renameClass.to);
  }
}

void RenameVisitor::postvisitE_variable(E_variable *ev) {
  Variable *v = ev->var;
// now try to rename as function
  if (v->scope && v->scope->scopeKind == SK_GLOBAL
      && v->name == renamerCmd->renameClass.from)
    renameAt(ev->loc, renamerCmd->renameClass.toShort, 0);
  
}
