#include "include/pointcloud.h"
#include "include/skelft.h"
#include "include/sparsematrix.h"
#include "include/fullmatrix.h"
#include "include/sortederrmatrix.h"
#include "include/grouping.h"
#include <cuda_runtime_api.h>
#include <algorithm>
#include <string>
#include <vector>
#include <map>
#include <deque>

#define REAL double									//Needed for Triangle; note also that Triangle is a C API

#define TRILIBRARY
#define NO_TIMER
#define ANSI_DECLARATORS
#define VOID void

extern "C" {										
#include "triangle.h"
}


using namespace std;



float Point2d::edgeAngle(const Point2d& p, const Point2d& q)
{
			const Point2d& r  = q-p;
			float n           = r.norm();
			if (n<1.0e-4) return -1;							//Cannot determine angle, edge is of zero length
						
			float cos_r       = r.x/n;
			float sin_r       = r.y/n;
			float alpha       = acos(cos_r);
			if (sin_r<0)  alpha = 2*M_PI - alpha; 
			return alpha;  
}

float Point2d::angle(const Point2d& r)
{
			float n           = r.norm();
			if (n<1.0e-4) return -1;							//Cannot determine angle, edge is of zero length
						
			float cos_r       = r.x/n;
			float sin_r       = r.y/n;
			float alpha       = acos(cos_r);
			if (sin_r<0)  alpha = 2*M_PI - alpha; 
			return alpha;
}

Point2d Point2d::center(const Point2d& a,const Point2d& b,const Point2d& c)
{
	Point2d ab = b-a; ab.normalize();
	Point2d ac = c-a; ac.normalize();
	Point2d cb = b-c; cb.normalize();
	Point2d ca = a-c; ca.normalize();
	
	Point2d  A = (ab+ac); A.normalize();
	Point2d  C = (ca+cb); C.normalize();
	
	float t2 = (A.x*(c.y-a.y)-A.y*(c.x-a.x))/(A.y*C.x-A.x*C.y);
	
	Point2d r;
	r.x = C.x*t2 + c.x;
	r.y = C.y*t2 + c.y;
	
	return r;
}

//-------------------------------------------------------------------------------------------------------------------------


PointCloud::PointCloud(int size): kdt(0),kdt_points(0),siteMax(0),fboSize(size),DT_max(0),
								  point_scalars_min(0),point_scalars_max(0),
								  avgdist(0),num_labels(0)
{
	cudaMallocHost((void**)&siteFT,size*size*2*sizeof(short));    
	cudaMallocHost((void**)&siteParam,size*size*sizeof(float));
	cudaMallocHost((void**)&siteDT,size*size*sizeof(float));
	
	buff_triangle_id = new unsigned int[fboSize*fboSize];
}

PointCloud::~PointCloud()
{
	if (kdt) { delete kdt; kdt=0; }
	if (kdt_points) annDeallocPts(kdt_points);
	delete edges;
	delete sorted_edges;
	delete distmatrix;
	delete sorted_errors;
	for(int i=0;i<attributes.size();++i)
	   delete attributes[i];
	
	delete[] buff_triangle_id;
	
	cudaFreeHost(siteFT);
	cudaFreeHost(siteParam);
	cudaFreeHost(siteDT);	
}

void PointCloud::makeKDT()
{
	delete kdt;														//Deallocate whatever we hold currently
	if (kdt_points) annDeallocPts(kdt_points);	
		
	kdt_points = annAllocPts(points.size(), 2);						//Put all vertices in the ANN search structure
		
	for(int i=0;i<points.size();i++)								//Copy surf vertices in an ANN-compatible structure
	{
		kdt_points[i][0]=points[i].x;
		kdt_points[i][1]=points[i].y;
	}
		
	kdt = new ANNtree(kdt_points, points.size(), 2, 5);
}	

int PointCloud::searchR(const Point2d& seed,float rad,int nn_max,vector<int>& result) const
{
	ANNcoord query_pt[2];
	static ANNidx   nn_idx[10000];	
	
	query_pt[0] = seed.x;
	query_pt[1] = seed.y;
	
	int nn = kdt->annkFRSearch(query_pt,rad*rad,nn_max,nn_idx);				//Search up to nn_max points within rad from seed.
	
	if (nn>nn_max) nn=nn_max;												//If there are more than nn_max points in the ball, beware that
																			//we only returned nn_max of them.
	result.resize(nn); 
	for(int i=0;i<nn;++i) 
		result[i] = nn_idx[i]; 
	
	return nn;
}


int PointCloud::closest(int pid,float& d) const
{
	ANNcoord query_pt[2];
	ANNidx   nn_idx[2];	
    ANNdist  nn_dist[2];
	
	query_pt[0] = points[pid].x;
	query_pt[1] = points[pid].y;
	
	kdt->annkSearch(query_pt,2,nn_idx,nn_dist);
	
	d = sqrt(nn_dist[1]);
													
	return (nn_idx[1]!=pid)? nn_idx[1]:nn_idx[0];
}

int PointCloud::searchNN(const Point2d& seed,int k,vector<int>& result,vector<float>* result_dist) const
{
	ANNcoord query_pt[2];
	static ANNidx   nn_idx[10000];	
    static ANNdist  nn_dist[10000];
	
	query_pt[0] = seed.x;
	query_pt[1] = seed.y;
	
	kdt->annkSearch(query_pt,k,nn_idx,nn_dist);
	
	result.resize(k);
	if (result_dist) result_dist->resize(k);
	for(int i=0;i<k;++i) 
	{
		result[i] = nn_idx[i]; 
		if (result_dist) (*result_dist)[i] = nn_dist[i];
	}
			
	return k;
}

void PointCloud::triangulate()
{
   triangulateio ti,to;													
   double pts[50000];													//In: 2D points to be triangulated
   int tris[50000];														//Out: triangles created by the 2D triangulation			
   int tedges[600000];													//Out: edges of triangulation

   ti.pointlist				   = pts;									//Set up triangulation of projected points
   ti.pointmarkerlist          = 0;										//
   ti.numberofpointattributes  = 0;										//
   ti.numberofpoints		   = points.size();	 

   to.trianglelist             = tris;									//All we want are the triangles..
   to.edgelist				   = tedges;								//..and edges
   to.pointmarkerlist          = 0;										//
   
   for(int i=0;i<ti.numberofpoints;++i)
   {
     pts[2*i]   = points[i].x;
	 pts[2*i+1] = points[i].y;
   }
   
   ::triangulate("zePBNQYY",&ti,&to,0);									//Call Triangle-lib to do the triangulation of the projected skel-points

   triangles.resize(to.numberoftriangles);								//Get triangles:
   point2tris.resize(points.size());									//In the same time, construct point2tris[]
   for(int i=0;i<to.numberoftriangles;++i)
   {
	  int t = 3*i;
	  int a = to.trianglelist[t];
	  int b = to.trianglelist[t+1];
	  int c = to.trianglelist[t+2];
      triangles[i] = Triangle(a,b,c);
	  
	  point2tris[a].insert(i);
	  point2tris[b].insert(i);
	  point2tris[c].insert(i);
   }		
   
   for(int i=0;i<to.numberofedges;++i)									//Get edges:
   {
	  int i1 = tedges[2*i], i2 = tedges[2*i+1];
	  (*edges)(i1,i2) = 1;
	  (*edges)(i2,i1) = 1;
   }	  
}




struct EdgeCompare													//Compares two edges vs their angles [0..M_PI] with the +x axis
{																	//Used to sort edges anticlockwise
		 EdgeCompare() {}	
	bool operator() (const PointCloud::Edge& ei,const PointCloud::Edge& ej) 
					{ return ei.angle < ej.angle; }
};


void PointCloud::sortErrors()										//For each matrix row (point in cloud), sort dist-errors to all other points.
{																	//Like this, it's next easy to find out, for any point, which are its most
	int NP = points.size();											//evident false positives (large negative dist-errors) and false negatives
	sorted_errors = new SortedErrorMatrix(NP);						//(large positive dist-errors)
	
	for(int i=0;i<NP;++i)
	{
		const DistMatrix::Row& row  = (*distmatrix)(i);
		SortedErrorMatrix::Row&  srow = (*sorted_errors)(i); 

		int idx = 0;	
		for(DistMatrix::Row::const_iterator it=row.begin();it!=row.end();++it,++idx)
		{
			float val = *it;
			srow.insert(make_pair(val,idx));
		}	
	}
}


void PointCloud::initEnd()
{
	int NP = points.size();
	
	memset(siteParam,0,fboSize*fboSize*sizeof(float));				//Site parameterization:			
																	//0: no site at current pixel
	for(int i=0;i<NP;++i)											//i: site i-1 at current pixel
	{
		const Point2d& p = points[i];
		siteParam[int(p.y)*fboSize+int(p.x)] = i+1;
	}
	
	siteMax = NP+1;
	
	skelft2DFT(siteFT,siteParam,0,0,fboSize,fboSize,fboSize);		//Compute FT of the sites
	skelft2DDT(siteDT,0,0,fboSize,fboSize);							//Compute DT of the sites (from the resident FT)	
	
		
	edges = new SparseMatrix(NP);									//Allocate Delaunay adjacency matrix 
	sorted_edges = new EdgeMatrix(NP);								//Allocate ordered Delaunay adj. matrix (edges ordered by angle anticlockwise)
	false_negative_error.resize(NP);								//Allocate space for various other data vectors that we can build now
	aggregate_error.resize(NP);										//
	aggregate_fp_error.resize(NP);									//
	aggregate_fn_error.resize(NP);									//
	label_mix.resize(NP);
	
	DT_max = 0;														//Compute max of DT
	for(int i=0;i<fboSize*fboSize;++i)
	{
		float dt = siteDT[i];
		if (dt>DT_max) DT_max = dt;
	}
	
	triangulate();													//Compute Delaunay triangulation of this
	
	EdgeCompare edge_comp;											//Used to compare edges vs their angles
	for(int i=0;i<NP;++i)											//Construct sorted_edges[]: Delaunay edges[], but sorted anticlockwise around each point
	{
		const SparseMatrix& ed       = *edges;						
		const SparseMatrix::Row& row = ed(i);						//This is what we want to sort
		EdgeMatrix::Row&		orow = (*sorted_edges)(i);			//This is where the sorted output goes
		const Point2d&             p = points[i];					//This is the current vertex, for which we'll sort edges
		
		orow.resize(row.size());									//Make room for the sorted output		
		int j=0;
		for(SparseMatrix::Row::const_iterator it=row.begin();it!=row.end();++it)
		{
			const int	  pid = it->first;							//Find the angle [0..M_PI] of edge (p,points[it->first]) with the +x axis
			float alpha = Point2d::edgeAngle(p,points[pid]);
			orow[j++] = Edge(pid,alpha);							//Copy the unsorted data to the sorted output, prior to sort
		}
		   
		std::sort(orow.begin(),orow.end(),edge_comp);				//Sort orow[] using the anticlockwise edge order
	}	
		
	makeKDT();

	hash_set<int> lbls;
	avgdist = 0;													//Compute average distance points in the cloud
	for(int i=0;i<NP;++i)											//For this, we estimate the avg radius of the NSZ-nb, with a small NSZ
	{																//(for some reason, NSZ=approx 5 gives good results)
		const int NSZ = 5;
		vector<int> nn; vector<float> nnd;
		searchNN(points[i],NSZ,nn,&nnd);
		float dist = sqrt(nnd[NSZ-1]);
		avgdist += dist;
		lbls.insert(point_scalars[i]);
	}
	avgdist /= NP;
	num_labels = lbls.size();

	sortErrors();	
}



Grouping* PointCloud::groupByLabel()								//Construct grouping of points in this based on equal-label
{
	SimpleGrouping* grp = new SimpleGrouping(this);					//Make new grouping based on point-labeling of this
	map<int,int> gkey2gidx;											//Maps the group-ids to 0-based ids

	int NG=0,NP=size();	
	for(int i=0;i<NP;++i)											//Read all points:
	{
		int gkey = int(point_scalars[i]);							//See if we have a new group-id
		if (gkey2gidx.find(gkey)==gkey2gidx.end())					//If so, map it to a 0-based increasing integer
		{
		   gkey2gidx.insert(make_pair(gkey,NG)); ++NG;
		}
	}
	
	grp->resize(NG);												//We now know how many groups we have to make
	for(int i=0;i<NP;++i)											//Scan the points and add them to their right groups
	{
		int gkey = int(point_scalars[i]);
		int gid  = gkey2gidx.find(gkey)->second;
		grp->group(gid).insert(i);		
	}
	
	return grp;
}



bool PointCloud::loadPex(const char* file,const char* proj,bool load_nd)	//Load triplet of Pex files: nD, 2D, error
{
	const float t = 0.04;											//Border between the points and image-border (needed for safe DT computations)

	string fnd = file; fnd += ".nd";								//n-dimensional points file
	string f2d = file;												//2-dimensional projection file
	if (proj) { f2d += "."; f2d += proj; }			
	f2d += ".2d";
	
	string fer = file;												//error matrix (wrt projection of n-D points to 2-D points)
	if (proj) { fer += "."; fer += proj; }
	fer += ".err";
	char line[1024];
	
	FILE* fp = fopen(f2d.c_str(),"r");								//1. Read 2D projections:
	if (!fp)
	{
		cout<<"Error: Cannot open "<<f2d<<endl;
		return false;
	}	
	
	fgets(line,1024,fp);											//Skip first line 'DY'
	int NP;
	fscanf(fp,"%d",&NP);											//Get #points in file
	int dim;
	fscanf(fp,"%d",&dim);											//Get point dimensions (should be 2)
	if (dim!=2)
	{
	   cout<<"Warning: 2D projection dimension="<<dim<<", expected 2"<<endl;
	}

	points.resize(NP);
	point_scalars.resize(NP);
	point_scalars_min = min_p.x = min_p.y = 1.0e+8;
	point_scalars_max = max_p.x = max_p.y = -1.0e+8;
		
	for(int i=0;i<NP;++i)											//Read all 2D point projections:
	{
		Point2d& p = points[i];
		fscanf(fp,"%*[^;];%f;%f;%f",&p.x,&p.y,&point_scalars[i]);	//REMARK: apparently, first item (point-ID) can be a string..
		point_scalars_min = std::min(point_scalars_min,point_scalars[i]);
		point_scalars_max = std::max(point_scalars_max,point_scalars[i]);
		min_p.x = std::min(min_p.x,p.x);
		min_p.y = std::min(min_p.y,p.y);
		max_p.x = std::max(max_p.x,p.x);
		max_p.y = std::max(max_p.y,p.y);	
	}
	fclose(fp);
		
	Point2d range(max_p-min_p);

	for(int i=0;i<NP;++i)											//Normalize read points within available image space (fboSize)
	{
		Point2d& p = points[i];
		p.x = (p.x-min_p.x)/range.x*(1-2*t)*fboSize + t*fboSize;
		p.y = (p.y-min_p.y)/range.y*(1-2*t)*fboSize + t*fboSize;
	}
	
	for(int i=0;i<NP;++i)
	{
		Point2d& p = points[i];
		min_p.x = std::min(min_p.x,p.x);
		min_p.y = std::min(min_p.y,p.y);
		max_p.x = std::max(max_p.x,p.x);
		max_p.y = std::max(max_p.y,p.y);	
	}	
	
	fp = fopen(fer.c_str(),"r");									//2. Read projection-error matrix:
	if (!fp)
	{
		cout<<"Error: Cannot open"<<fer<<endl;
		return false;
	}		
	
	distmatrix = new DistMatrix(NP);								
	int nrows;
	fscanf(fp,"%d",&nrows);
	if (nrows!=NP)
	{
		cout<<"Error: error-matrix #rows "<<nrows<<" != #points "<<NP<<endl;
		return false;
	}
	
	for(int i=0;i<NP;++i)
	  for(int j=0;j<=i;++j)
	  {
		float val;
		if (j<i) fscanf(fp,"%f;",&val);								//Take care, last elem on line not followed by ';'
		else														//Also, note that the matrix stored in the file is symmatric lower-diagonal
		{															//REMARK: We could store this more compactly..
			fscanf(fp,"%f",&val);	
			continue;
		}
				
		(*distmatrix)(i,j) = val;
		(*distmatrix)(j,i) = val;
	  }
	fclose(fp);			

	distmatrix->minmax();											//3. Compute range of distance matrix	
																	//   -the range [min,0]: false positives (points too close in 2D w.r.t. nD)
																	//   -the range [0,max]: false negatives (points too far in 2D w.r.t. nD)
	cout<<"Error matrix: ["<<distmatrix->min()<<","<<distmatrix->max()<<"]"<<endl;
	
	if (load_nd)													//4. Read the nD data values:
	{
		fp = fopen(fnd.c_str(),"r");									
		fgets(line,1024,fp);											//Skip first line 'DY'
		int NP_n;
		fscanf(fp,"%d",&NP_n);											//Get #points in file
		if (NP_n!=NP)
		{
		   cout<<"Error: "<<NP_n<<" nD points, "<<NP<<" 2D points"<<endl;
		   return false;
		}
		
		int ND;
		fscanf(fp,"%d",&ND);											//Get nD point dimensions 
		attributes.resize(ND);											//Allocate space for attributes
		attributes_min.resize(ND);
		attributes_max.resize(ND);
		for(int i=0;i<ND;++i)	
		{
		   attributes[i] = new vector<float>(NP);
		   attributes_min[i] = 1.0e+6;
		   attributes_max[i] = -1.0e+6;
		}
			
		char pid_nd[128];
		for(int i=0;i<NP;++i)											//Read all n-D points:
		{
			fscanf(fp,"%[^;];",pid_nd);									//REMARK: Apparently, point-ID can be a string..
			for(int j=0;j<ND;++j)										//Read all n-D dimensions for point 'i'
			{
				float dim_j;
				fscanf(fp,"%f;",&dim_j);
				(*attributes[j])[i] = dim_j;
				attributes_min[j] = std::min(attributes_min[j],dim_j);
				attributes_max[j] = std::max(attributes_max[j],dim_j);
			}
			
			float label;												//Sanity check: label must be the same in 2D and nD
			fscanf(fp,"%f",&label);
			if (label!=point_scalars[i])
			{
			   cout<<"Error: point "<<i<<"("<<pid_nd<<") has label "<<label<<" in nD and label "<<point_scalars[i]<<" in 2D"<<endl;
			   return false;
			}
		}
		fclose(fp);
	}
	
	return true;
}



void PointCloud::closestEdges(const Point2d& x,int pid,const Edge*& e1,float& d1,const Edge*& e2,float& d2) const
{
	const Point2d& p = points[pid];							//'center' of the edge-fan we search into
	float      alpha = Point2d::edgeAngle(p,x);				//angle of line from p to current x	

	const EdgeMatrix::Row& row = (*sorted_edges)(pid);		//all edges in edge-fan

	if (row.size()<2) 
	{
		cout<<"ERROR: point "<<pid<<" has only "<<row.size()<<" edges"<<endl;
		e1 = e2 = 0;
		return;
	}

	int i=0;		
	for(;i<row.size();++i)									//search where the current angle falls (note that row is sorted on angles..)
	{
	   if (alpha <= row[i].angle) break;
	}   
	
	i = i % row.size();
	int j = i-1;
	if (j<0) j += row.size();								//next edge after edge 'i'

	e2 = &row[i];
	e1 = &row[j];
	d2 = x.distance2line(p,points[e2->pid]);
	d1 = x.distance2line(p,points[e1->pid]);
}



float PointCloud::blendDistance(const Point2d& pix,const Triangle& tr) const
{
	int i0 = tr(0);
	int i1 = tr(1);
	int i2 = tr(2);
	
	const Point2d& p0 = points[i0];						//closest cloud-point to 'pix'
	const Point2d& p1 = points[i1];
	const Point2d& p2 = points[i2];
	
	float d1 = pix.distance2line(p0,p1);					
	float d2 = pix.distance2line(p0,p2);					
	float d3 = pix.distance2line(p1,p2);					
	float tmin = min(d1,min(d2,d3));					//DT of triangle

	Point2d c = Point2d::center(p0,p1,p2);
	float dmin = c.dist(pix);							//DT of center
	float B = 0.5*(((dmin)? min(tmin/dmin,1.0f):1.0f) + ((tmin)? max(1-dmin/tmin,0.0f):0.0f));
	return pow(B,0.5f);	
}


/*
bool PointCloud::findTriangle(const Point2d& x,int& pid,const Edge*& e1,const Edge*& e2) const
{
	vector<int> nbs;
	searchNN(x,1,nbs);	
	pid = nbs[0];															//Closest point to 'x'

	std::deque<int> q;
	std::vector<bool> visited(triangles.size());
		
	const TrisOfPoint& fan = point2tris[pid];
	for(TrisOfPoint::const_iterator it=fan.begin();it!=fan.end();++it)
	{   }
	   
	while(q.size())   
	{
		int t  = *(q.begin()); q.pop_front();
		const Triangle& tr = triangles[t];
		int p0 = tr.idx[0];
		int p1 = tr.idx[1];
		int p2 = tr.idx[2];
		if (x.inTriangle(points[p0],points[p1],points[p2]))
		{
			return hitTriangle(x,pid,e1,e2);
		}
		
		const TrisOfPoint& f0 = point2tris[p0];
		for(TrisOfPoint::const_iterator it=f0.begin();it!=f0.end();++it)
		   if (!visited[*it]) { q.push_back(*it); visited[*it]=true; }
		
		const TrisOfPoint& f1 = point2tris[p1];
		for(TrisOfPoint::const_iterator it=f1.begin();it!=f1.end();++it)
		   if (!visited[*it]) { q.push_back(*it); visited[*it]=true; }
		
		const TrisOfPoint& f2 = point2tris[p2];
		for(TrisOfPoint::const_iterator it=f2.begin();it!=f2.end();++it)
		   if (!visited[*it]) { q.push_back(*it); visited[*it]=true; }
	
	}
	
	return false;
}
*/




inline float errorDistWeighting(float d,float delta)
{
   return (d<delta)? 1 : exp(-0.3*(d-delta)*(d-delta));
}


float PointCloud::interpolateDistMatrix(const Point2d& pix,float& certainty,float delta) const
{
	int tid = hitTriangle(pix);
	if (tid<0)										//pix outside the point cloud triangulation: nothing to show there, really
	{
		certainty = 0;
		return 0;
	}

	const Point2d* p[3]; float d[3]; int i[3]; float w[3];
	const DistMatrix& dm = *distmatrix;
	const Triangle& tr = triangles[tid];

	i[0] = tr(0);
	i[1] = tr(1);
	i[2] = tr(2);

	p[0] = &points[i[0]];							//p1 = other end of edge (p,p1)
	p[1] = &points[i[1]];							//p2 = other end of edge (p,p2)
	p[2] = &points[i[2]];
													//Get edge weights:
	w[0] = -std::min(dm(i[2],i[0]),0.0f);			//As we ONLY want to emphasize false-positives here,
	w[1] = -std::min(dm(i[2],i[1]),0.0f);			//so we simply eliminate false-negatives from this data.
	w[2] = -std::min(dm(i[0],i[1]),0.0f);   
	
	d[0] = pix.distance2line(*p[2],*p[0]);
	d[1] = pix.distance2line(*p[2],*p[1]);
	d[2] = pix.distance2line(*p[0],*p[1]);

	w[0] *= errorDistWeighting(p[2]->dist(*p[0]),delta); 	
	w[1] *= errorDistWeighting(p[2]->dist(*p[1]),delta); 	
	w[2] *= errorDistWeighting(p[0]->dist(*p[1]),delta); 	

	certainty = 1;
	if (d[0]<1) return w[0];						//point on (p,p1): return w1
	if (d[1]<1) return w[1];						//point on (p,p2): return w2
	if (d[2]<1) return w[2];						//point on (p,p2): return w2
	
	float B   = blendDistance(pix,tr);
	certainty = 1-B;								//Record the certainty (0 for deepest point in triangle, 1 for being on the edges)
	
	float l[3],A[3];								//Use barycentric coordinates to interpolate
	l[0] = p[0]->dist(*p[2]);
	l[1] = p[1]->dist(*p[2]);
	l[2] = p[0]->dist(*p[1]);
	A[0] = 1/(d[0]*l[0]);																																					
	A[1] = 1/(d[1]*l[1]);																																					
	A[2] = 1/(d[2]*l[2]);																																					

	float val = (w[0]*A[0]+w[1]*A[1]+w[2]*A[2])/(A[0]+A[1]+A[2]);
	return val;
}



void PointCloud::computeAggregateError(float norm)								//Compute aggregated projection error for a point wrt all its neighbors
{
	float aggregate_error_max    = 0;
	float aggregate_fp_error_max = 0;
	float aggregate_fn_error_max = 0;
	const int NP = points.size();
	
	for(int i=0;i<NP;++i)
	{
		const DistMatrix::Row& row = (*distmatrix)(i);
		aggregate_error[i]    = 0;
		aggregate_fp_error[i] = 0;
		aggregate_fn_error[i] = 0;
		for(DistMatrix::Row::const_iterator it=row.begin();it!=row.end();++it)
		{
			float val              = *it;
			aggregate_error[i]    += fabs(val);											
			aggregate_fp_error[i] += (val<0)? -val:0;
			aggregate_fn_error[i] += (val>0)? val:0;
		}
		aggregate_error_max     = std::max(aggregate_error_max,aggregate_error[i]);
		aggregate_fp_error_max  = std::max(aggregate_fp_error_max,aggregate_fp_error[i]);
		aggregate_fn_error_max  = std::max(aggregate_fn_error_max,aggregate_fn_error[i]);
	}

	for(int i=0;i<NP;++i)														//Normalize aggregated error
	{
	    aggregate_error[i]    /= NP;
		aggregate_fp_error[i] /= NP;
		aggregate_fn_error[i] /= NP;
	}
	aggregate_error_max    /= NP;	
	aggregate_fp_error_max /= NP;	
	aggregate_fn_error_max /= NP;	
	
	if (aggregate_error_max<1.0e-6) aggregate_error_max=1;
	if (aggregate_fp_error_max<1.0e-6) aggregate_fp_error_max=1;
	if (aggregate_fn_error_max<1.0e-6) aggregate_fn_error_max=1;

	cout<<"Aggregate err max: "<<aggregate_error_max<<endl;

	if (norm)																	//Normalization to user-specified range
	{
		aggregate_error_max    = norm;
		aggregate_fp_error_max = norm;
		aggregate_fn_error_max = norm;
	}

	for(int i=0;i<NP;++i)														//Normalize aggregated error
	{
	    aggregate_error[i]    = std::min(aggregate_error[i]/aggregate_error_max,1.0f);
	    aggregate_fp_error[i] = std::min(aggregate_fp_error[i]/aggregate_fp_error_max,1.0f);
	    aggregate_fn_error[i] = std::min(aggregate_fn_error[i]/aggregate_fn_error_max,1.0f);
	}
}



float PointCloud::averageNeighborDist(int pid) const							//Return average dist to geometric nbs of point 'pid'
{
		const Point2d& pi = points[pid];
		const EdgeMatrix::Row& erow = (*sorted_edges)(pid);							
		float avgdist = 0;
		
		for(EdgeMatrix::Row::const_iterator it = erow.begin();it!=erow.end();++it)
			avgdist += pi.dist(points[it->pid]);
		avgdist /= erow.size();
		return avgdist;
}



void PointCloud::computeLabelMixing()											//Compute per-point label-mix degree (in [0,1]). This tells from how many of the
{																				//point-neighbor-labels (in its triangle-fan) do the point's label differ.
	for(int i=0;i<points.size();++i)				
	{
		const Point2d& pi = points[i];											//Compute label-mixing at point 'pi':
		int lbl_i = point_scalars[i];
		
		vector<int> nn;															//Get R-nearest neighbors, and see how much mixing we've got in there
		const float rad = avgdist;
		int NN = searchR(pi,rad,200,nn);		
		
		float wsum = 0;															//Compute weights for all neighbors
		float diff = 0;
		for(int j=0;j<NN;++j)
		{
		  float r  = pi.dist(points[nn[j]]);
		  float w  = exp(-r*r/rad/rad);  
		  wsum    += w;

		  int lbl_j = point_scalars[nn[j]];
		  if (lbl_j==lbl_i) continue;
		  diff += w;
		}

		label_mix[i] = diff/wsum;
	}
}



void PointCloud::computeFalseNegatives(const Grouping::PointGroup& grp,float range)
{																				//Compute false-negatives w.r.t. all points in 'grp'
	const int NP = points.size();
	vector<float> rerr(NP,1.0e+6);

	for(Grouping::PointGroup::const_iterator it=grp.begin();it!=grp.end();++it)
	{
		int pid = *it;		
		computeFalseNegatives(pid,false);										//Compute FN-error of all points wrt 'pid'
		for(int i=0;i<NP;++i)
		   rerr[i] = std::min(rerr[i],false_negative_error[i]);
	}
	
	if (!range)																	//Auto-normalization
	{
	   for(int i=0;i<NP;++i) range = std::max(range,rerr[i]);
	   cout<<"False negatives max: "<<range<<endl;
	   if (range<1.0e-6) range = 1;
	}

	for(int i=0;i<NP;++i) false_negative_error[i] = std::min(rerr[i]/range,1.0f);
}


void PointCloud::computeFalseNegatives(int pid,bool norm)						//Compute false-negatives w.r.t. point 'pid'
{
	float err_max = 0;
	const DistMatrix::Row& row = (*distmatrix)(pid);							//'row' encodes errors of all points w.r.t. 'pid'
	
	int idx = 0;
	for(DistMatrix::Row::const_iterator it=row.begin();it!=row.end();++it,++idx)
	{
		float val = std::max(*it,0.0f);											//We only want to emphasize false negatives here,
		false_negative_error[idx] = val;										//so we skip false positive points.
		err_max = std::max(err_max,val);
	}
	
	float err_sum = 0;															//Assign relative error for 'pid' (wrt itself..)
	const EdgeMatrix::Row& erow = (*sorted_edges)(pid);							//as being the average error wrt its neighbors
	for(EdgeMatrix::Row::const_iterator it = erow.begin();it!=erow.end();++it)
	{
		int     j = it->pid; 
		err_sum += false_negative_error[j];
	}

	false_negative_error[pid] = err_sum/erow.size();
	
	if (!norm) return;
	
	err_max = std::max(err_max,false_negative_error[pid]);						//Normalize FN-error (if so desired)
	if (err_max<1.0e-6) err_max=1;
	for(int i=0;i<points.size();++i)											//Normalize relative error (so it's in [0,1])
	   false_negative_error[i] /= err_max;
}


