#pragma once


#include <vector>
#include "Point3d.h"
#include "Volume.h"
#include "Vector.h"

#include <unordered_set>
#include <unordered_map>
#include <memory>




class PointSearch;

class FeaturePoints
{
  const std::vector<Point3d>* boundary;               //the input boundary surface (not owned)

public:

  // Using this instead of tuple to prevent major refactoring from FeaturePair
  // Contains the Two or Three (most salient) feature-points for a skeleton point
  struct FeatureTriple
  {
    int first;
    int second;
    int third;

    FeatureTriple(int a, int b, int c) :
      first(a), second(b), third(c) {}

    FeatureTriple() :
      first(-1), second(-1), third(-1) {}

  };

  typedef std::unordered_set<int> ExtFeatures;              //Extended feature-set for one skeleton point
  typedef std::vector<ExtFeatures> SkelExtFeatures;         //All extended feature-sets for all skeleton points
  typedef int ClosestFeature;                               //The guaranteed exact closest feature point for a skeleton point
  typedef std::vector<FeatureTriple> SkelFeatures;            //All strict feature-sets for all skeleton points
  typedef std::vector<ClosestFeature> SkelClosestFeatures;  //All closest one-point feature points for all skeleton points
  typedef std::unordered_map<unsigned, unsigned> Coord2SKP; //Coord key to skeleten point index

  FeaturePoints(const std::vector<Point3d> &boundary);

  ~FeaturePoints();

  void        construct(const std::vector<Point3d> &skeleton,
                        const Volume<Vector> &tangents,
                        const Volume<float> &importance,
                        const Volume<float> &edt);

  void        computeLigatureP(const std::vector<Point3d> &skeleton,
                               const Volume<float> &importance);

  Coord2SKP&   GetOrConstructCoord2SkelPointIndex();

  void refreshPointSearch();

  SkelExtFeatures extFeaturePoints;          //For each skel-point: set of IDs of all its features (extended FT)
  SkelFeatures featurePoints;                //For each skel-point: IDs of the two surf-points which are its best features
  SkelClosestFeatures closestFeaturePoints;  //For each skel-point: ID of surf-point which is closest to it (1-element FT)
  std::vector<Point3d>  fullSkeleton;        //all points of the unsimplified skeleton
  PointSearch* skelSearch;                   //kd-tree for the full (unsimplified) skeleton
  std::unique_ptr<PointSearch> pointSearch;       //kd-tree for input boundary surface
  std::unique_ptr<Coord2SKP> coord2SKPMap;  //Translates coords to skel-point ID's.

  bool useFeatureReflection = false;

protected:

  FeatureTriple     computeTwoFeatures(const ExtFeatures
                                     &fvec,   //Compute featurePoints[] from extFeaturePoints[]
                                     const Point3d &s,
                                     const Volume<float> &edt);

};

