#pragma once

#include <stdlib.h>
#include <string.h>

#ifdef __APPLE__
#include <sys/malloc.h>
#endif

#include <stdio.h>



typedef unsigned char byte;

template< class T > class Image
{
public:

  T **pixels;

protected:
  short width, height;

public:
  /*! Generates an image with the specified name; does not allocate
    internal buffer for the image data. */
  Image(char const * name = "");
  /*! Generates an image of the specified size "w"x"h" and with the specified "name". */
  Image(int w, int h, char const *name = "");
  /*! Copy constructor. */
  Image(const Image<T> &);

  /*! Destructor. */
  virtual ~Image();
  /*! Returns the image width. */
  int getWidth() const
  {
    return width;
  }
  /*! Returns the image height. */
  int getHeight() const
  {
    return height;
  }
  /*! Returns the image size - "w"x"h". */
  int getSize()
  {
    return width * height;
  }
  /*! Allocates internal buffer data of size "w"x"h" for the image and
    initializes the pixel values to "0"; frees old internal buffers,
    if already allocated. */
  void makeImage(int w, int h);
  /*! Destroys all data structures associated with the image; frees
    internal buffers. */
  void destroyImage();
  /*! Returns a pointer to the actual image buffer. */
  T *getPixels()
  {
    return pixels[0];
  }
  /*! Creates an image buffer of size "w"x"h" and initializes
    the data with the buffer pointed by "data". */
  void setPixels(int w, int h, T *data);

  /*! Overloaded access operator */
  T *operator[](short i)
  {
    return pixels[i];
  }
  const T *operator[](short i) const
  {
    return pixels[i];
  }

  T &operator()(short i, short j)
  {
    return pixels[0][i * width + j];
  }
  const T &operator()(short i, short j) const
  {
    return pixels[0][i * width + j];
  }


  /*! Copy operator. */
  Image< T > &operator=(const Image< T > &img);

  /*! Tests if two images are the same in both size and pixel values. */
  int operator==(const Image< T > &img);
  /*! Tests if two images are different in both sizes and pixel values. */
  int operator!=(const Image< T > &img);

  /*! \ingroup point_alg */
  /*! Adds a constant value to the value of image pixels. */
  Image< T >  &operator+=(const T val);
  /*! \ingroup point_alg */
  /*! Subtracts a constant value from the value of image pixels. */
  Image< T > &operator-=(const T val);
  /*! \ingroup point_alg */
  /*! Multiplies with a constant value the value of image pixels. */
  Image< T > &operator*=(const T val);

  /*! \ingroup point_alg */
  /*! Adds an image to the current one. */
  Image< T > &operator+=(const Image< T > &img);
  /*! \ingroup point_alg */
  /*! Multiplies an image with the current one. */
  Image< T > &operator*=(const Image< T > &img);
  /*! \ingroup point_alg */
  /*! Subtracts an image from the current one. */
  Image< T > &operator-=(const Image< T > &img);

  void clearImage(const T val);
  /*! Sets all the pixels in the current image to a zero-bit pattern. */
  void clearImage();
  /*! \ingroup geom_alg */
  /*! Rectifies the image size (width and height) to even value; if necessary,
    duplicates the last row and/or column */
  void makeEvenSized(void);

  void drawLine(int x1, int y1, int x2, int y2, T color);

  // Bellow are some functions intended to be used BUT
  // not used currently for various reasons:

  // Access operator for addressing the data buffer linearly
  //  T &operator()(int i) { return pixels[0][i];}
  // - not used because returns T& and conflicts with passing arguments
  //   to the parser; reimplemented in the derived classes

  // Access operator
  //  T &operator()(int i, int j) {return pixels[0][i*width+j];}
  // - not used because returns T& and conflicts with passing arguments
  //   to the parser; reimplemented in derived classes

  // "+","-", "*" operators: as friends functions
  //  friend Image< T > operator+< T >(const Image< T >& img1, const Image< T >& img2);
  //  friend Image< T > operator*< T >(const Image< T >& img1, const Image< T >& img2);
  //  friend Image< T > operator-< T >(const Image< T >& img1, const Image< T >& img2);
  // - not used because conflict with "Image" container "+" and "-" operators
  //   of the parser AND because they generate memory without a garbage
  //   collection mechanism (see Stroustroup p. 282)

  //  "+", "-", "*" operators as members of the class
  //  Image< T > &operator+(const Image< T > &img);
  //  Image< T > &operator-(const Image< T > &img);
  //  Image< T > &operator*(const Image< T > &img);
  // - not used because they generate memory without a garbage
  //   collection mechanism (see Stroustroup p. 282); can be adapted by
  //   using a pool of buffers of type "Image" (maybe later...)
};

/*  template< class T > Image<T> operator+(const Image<T>&, const Image<T>&); */
/*  template< class T > Image<T> operator-(const Image<T>&, const Image<T>&); */
/*  template< class T > Image<T> operator*(const Image<T>&, const Image<T>&); */









