#ifndef VOLUME_H
#define VOLUME_H

#include <cmath>
#include <vector>
#include "Image.h"
#include "coord3.h"
#include <tuple>
#include <algorithm>
#include <memory>


template< class T >
class SimpleVolume
{
  // replaced vector of images with one continous chunk of data, much better for L2 Caching
  std::unique_ptr<T[]> data;

  int width = 0;
  int height = 0;
  int depth = 0;

public:

  SimpleVolume() {};

  SimpleVolume(int width, int height, int depth) :
    data(new T[width * height * depth]),
    width(width),
    height(height),
    depth(depth)
  {
  }

  SimpleVolume(const SimpleVolume<T> &vol) = default;

  void makeVolume(int width, int height, int depth)
  {
    data.reset(new T[width * height * depth]);
    this->width = width;
    this->height = height;
    this->depth = depth;
  }


  inline T &operator()(short x, short y, short z)
  {
    return data[z * height * width + y * width + x];
  }

  inline const T &operator()(short x, short y, short z) const
  {
    return data[z * height * width + y * width + x];
  }

  inline T &operator()(const coord3s &c)
  {
    return data[c.z * height * width + c.y * width + c.x];
  }

  inline const T &operator()(const coord3s &c) const
  {
    return data[c.z * height * width + c.y * width + c.x];
  }

  int getDepth() const
  {
    return depth;
  }
  int getWidth() const
  {
    return width;
  }
  int getHeight()const
  {
    return height;
  }
  int getSize() const
  {
    return getWidth() * getHeight() * getDepth();
  }
  int getUnpaddedSize(int pad) const
  {
    return (getWidth() - pad) * (getHeight() - pad) * (getDepth() - pad);
  }

  float diagonal() const
  {
    return sqrtf(getWidth() * getWidth() + getHeight() * getHeight() + getDepth() * getDepth());
  }

  void clearVolume(T val)
  {
    for (int i = 0; i < getSize(); ++i)
      data[i] = val;
  }
  void clearVolume()
  {
    for (int i = 0; i < getSize(); ++i)
      data[i] = T();
  }

  static inline bool range(int x, int y, int z,
    int width, int height, int depth)
  {
    return (x > 0 && y > 0 && z > 0 && z < depth && x < width && y < height);
  }
};




template< class T >
class Volume
{

  std::vector< Image< T > > slices;

public:
  typedef struct
  {
    float coord[3];
  } point3;

  typedef std::vector< point3 > POINTS;
  typedef typename Volume<T>::point3 POINT3;

  void bresenham_linie_3D(int x1, int y1, int z1, int x2, int y2, int z2,
                          std::vector<point3> &points);

public:

  Volume() {};

  Volume(int width, int height, int depth) : slices(depth)
  {
    for (int i = 0; i < depth; i++)
      slices[i] = Image<T>(width, height);
  }

  Volume(const Volume<T> &vol) = default;

  void makeVolume(int width, int height, int depth)
  {
    if (depth != (int)slices.size())
    {
      slices.erase(slices.begin(), slices.end());
      for (int i = 0; i < depth; ++i)
        slices.push_back(Image<T>(width, height));
    }
  }

  inline Image<T> &operator[](short z)
  {
    return slices[z];
  }

  inline const Image<T> &operator[](short z) const
  {
    return slices[z];
  }

  inline T &operator()(short x, short y, short z)
  {
    return slices[z].pixels[y][x];
  }

  inline const T &operator()(short x, short y, short z) const
  {
    return slices[z].pixels[y][x];
  }

  inline T &operator()(const coord3s &c)
  {
    return slices[c.z].pixels[c.y][c.x];
  }

  inline const T &operator()(const coord3s &c) const
  {
    return slices[c.z].pixels[c.y][c.x];
  }

  void toVector(std::vector<T> &vec);

  int getDepth() const
  {
    return slices.size();
  }
  int getWidth() const
  {
    return slices[0].getWidth();
  }
  int getHeight()const
  {
    return slices[0].getHeight();
  }
  int getSize() const
  {
    return getWidth() * getHeight() * getDepth();
  }
  int getUnpaddedSize(int pad) const
  {
    return (getWidth() - pad) * (getHeight() - pad) * (getDepth() - pad);
  }

  float diagonal() const
  {
    return sqrtf(getWidth() * getWidth() + getHeight() * getHeight() + getDepth() * getDepth());
  }

  void clearVolume(T val)
  {
    for (int i = 0; i < (int)slices.size(); ++i)
      slices[i].clearImage(val);
  }
  void clearVolume()
  {
    for (int i = 0; i < (int)slices.size(); ++i)
      slices[i].clearImage();
  }

  static inline bool range(int x, int y, int z,
                           int width, int height, int depth)
  {
    return (x > 0 && y > 0 && z > 0 && z < depth && x < width && y < height);
  }

  void getMinMax(T &min, T &max);

  bool readBinaryField(char const *fname, int pad, int &np, int * &points,
                       float * &normals);
  bool readVTKField(char const *fname, int pad = 0);
  bool readAVSField(char const *fname, int pad = 0);
  void exportAVSField(char const *fname);
  void exportVolField(char const *fname, T threshold, byte fg_value, int pad = 0);
  void exportVTKField(char const *fname);

  void grad_order6(Volume<float> &gx, Volume<float> &gy, Volume<float> &gz);
  void grad_order3(Volume<float> &u, Volume<float> &v, Volume<float> &w);
  void scale(T value);
  void downsample(int down = 2);


  std::tuple<coord3s, coord3s> getBounds(coord3s p, coord3s b, short r) const;
  std::tuple<coord3s, coord3s> getBounds(coord3s p, short r) const;

  void addSphere(coord3s coord, short maskValue, float radius);
  void addCube(coord3s coord, short maskValue, short radius);
  T erode(coord3s coord, int r, const Volume<T>& tVol, T threshold, bool inflate) const;
  T dilate(coord3s coord, int r, const Volume<T>& tVol, T threshold) const;
  T median(coord3s coord, int r, const Volume<T>& tVol, T threshold) const;
  T mean(coord3s coord, int r, const Volume<T>& tVol, T threshold) const;
  T trilinear(coord3f coord) const;

  Volume<T> erode(int r, const Volume<T>& tVol, T threshold, bool inflate) const;
  Volume<T> dilate(int r, const Volume<T>& tVol, T threshold) const;
  Volume<T> medianFilter(int r, const Volume<T>& tVol, T threshold) const;
  Volume<T> meanFilter(int r, const Volume<T>& tVol, T threshold) const;
  Volume<T> minFilter(const Volume<T>& other) const;
  Volume<T> maxFilter(const Volume<T>& other) const;
};


#endif


