#pragma once

#include "Volume.h"
#include "Vector.h"
#include "inverseFT.h"
#include "CIsoSurface.h"
#include "coord3.h"
#include "logging.h"

#include <vector>
#include <deque>
#include <set>
#include <math.h>
#include <assert.h>
#include <memory>
#include <array>

class collapseSkel3d
{
public:

  //A coord3s (voxels) plus a floating-point value, used for sorting
  struct Coord3
  {
    coord3s c;
    float  v;
    Coord3() {};
    Coord3(const coord3s &c_): c(c_) {}
    Coord3(const coord3s &c_, float v_): c(c_), v(v_) {}
    Coord3(short x, short y, short z, float v_): c(x, y, z), v(v_) {}
    Coord3(short x, short y, short z): c(x, y, z) {}
  };

  typedef std::vector<Coord3> Interface;
  typedef std::deque<coord3s> NarrowBand; //REMARK: Can use here also deque<coord3s>

  struct coord3hasher
  {
    size_t operator() (const coord3s &p)
    const         //distribute available bits over size_t
    {
      return p.z * 1000000 + p.y * 1000 +
             p.x;        //return (p.z << 20) ^ (p.y << 10 ) ^ (p.x);
    }
  };

  struct index_cmp_min
  {
    Volume<float> &data;
    index_cmp_min(Volume<float> &d) : data(d) {};
    bool operator()(const Coord3 &a, const Coord3 &b) const
    {
      return data(a.c) < data(b.c);
    }
  };

  struct index_cmp_minv
  {
    index_cmp_minv() {};
    bool operator()(const Coord3 &a, const Coord3 &b) const
    {
      return a.v < b.v;
    }
  };


  struct index_cmp_max
  {
    Volume<float> &data;
    index_cmp_max(Volume<float> &d) : data(d) {};
    bool operator()(const coord3s &a, const coord3s &b) const
    {
      return data(a) > data(b);
    }
  };

  struct index_cmp_minb
  {
    Volume<byte> &thin_img;
    Volume<float>  &edt;
    Volume<float>  &dens;

    index_cmp_minb(Volume<byte> &_thin_img, Volume<float> &_edt,
                   Volume<float> &_dens) :
      thin_img(_thin_img), edt(_edt), dens(_dens) {};

    bool operator()(const coord3s &a, const coord3s &b) const
    {
      int na = 0, nb = 0;

      for (int z = a.z - 1; z <= a.z + 1; z++)
        for (int y = a.y - 1; y <= a.y + 1; y++)
          for (int x = a.x - 1; x <= a.x + 1; x++)
            if (thin_img(x, y, z)) na++;

      for (int z = b.z - 1; z <= b.z + 1; z++)
        for (int y = b.y - 1; y <= b.y + 1; y++)
          for (int x = b.x - 1; x <= b.x + 1; x++)
            if (thin_img(x, y, z)) nb++;

      return na > nb;

      float curv1 = (float) thin_img[a.z][a.y][a.x + 1] + (float) thin_img[a.z][a.y][a.x - 1] +
                    (float) thin_img[a.z][a.y + 1][a.x] + (float) thin_img[a.z][a.y - 1][a.x] +
                    (float) thin_img[a.z + 1][a.y][a.x] + (float) thin_img[a.z - 1][a.y][a.x];
      //- 6.0f*thin_img[a.z][a.y][a.x];

      float curv2 = (float) thin_img[b.z][b.y][b.x + 1] + (float) thin_img[b.z][b.y][b.x - 1] +
                    (float) thin_img[b.z][b.y + 1][b.x] + (float) thin_img[b.z][b.y - 1][b.x] +
                    (float) thin_img[b.z + 1][b.y][b.x] + (float) thin_img[b.z - 1][b.y][b.x];
      //-6.0f*thin_img[b.z][b.y][b.x];

      return curv1 > curv2;

      if (edt[a.z][a.y][a.x] < edt[b.z][b.y][b.x]) return true;
      else if (edt[a.z][a.y][a.x] > edt[b.z][b.y][b.x]) return false;
      return dens[a.z][a.y][a.x] < dens[b.z][b.y][b.x];
    }
  };

  struct index_cmp_min_div
  {
    Volume<float> &u, &v, &w;

    index_cmp_min_div(Volume<float> &_u, Volume<float> &_v, Volume<float> &_w) :
      u(_u), v(_v), w(_w) {};

    bool operator()(const coord3s &a, const coord3s &b) const
    {

      float curv1 = u[a.z][a.y][a.x + 1] - u[a.z][a.y][a.x - 1] + v[a.z][a.y + 1][a.x]
                    - v[a.z][a.y - 1][a.x] + w[a.z + 1][a.y][a.x] - w[a.z - 1][a.y][a.x];
      float curv2 = u[b.z][b.y][b.x + 1] - u[b.z][b.y][b.x - 1] + v[b.z][b.y + 1][b.x]
                    - v[b.z][b.y - 1][b.x] + w[b.z + 1][b.y][b.x] - w[b.z - 1][b.y][b.x];
      return curv1 > curv2;
    }
  };


  collapseSkel3d(bool advect_vel, bool force_monotonic);
  virtual~        collapseSkel3d();
  int         init(Volume<byte> &data, int thr, int &bordervoxels, bool smoothEdt = false, bool enableImportanceBoosting = false);
  void          init_iteration(float dt)
  {
    sim_iter = 0;
    curr_dst = 0.0f;
  }
  int         collapse_iteration(float dincr);
  void          simplify_skel(Volume<float> &lambda, float thr, float incr);
  void          simplify_skel_filter(Volume<float> &lambda, float thr,
                                     float incr);
  void reset();

  //Getters
  Volume<byte>     &getThinImage()
  {
    return thin_img;
  }
  Volume<byte> const &getThinImage() const
  {
    return thin_img;
  }
  Volume<float>    &getImportance()
  {
    return max_dens;
  }
  Volume<float> const &getImportance() const
  {
    return max_dens;
  }
  void setImportance(Volume<float>& imp)
  {
    max_dens = imp;
  }
  void setThinImage(Volume<byte>& value)
  {
    thin_img = value;
  }


  Volume<float>    &getDensity(float &totdens)
  {
    totdens = tdens;
    return (sim_iter & 1) ? dens : dens1;
  }
  Volume<float>    &getEDT(float &max_edt)
  {
    max_edt = max_dst;
    return edt;
  }
  Volume<float>    &getEDT()
  {
    return edt;
  }
  void    setEDT(Volume<float> value)
  {
    edt = value;
  }
  Volume<float> const &getEDT() const
  {
    return edt;
  }
  Volume<int>      &getFT()
  {
    return ft;
  }
  Volume<int> const &getFT() const
  {
    return ft;
  }
  const NarrowBand   &getIPoints() const
  {
    return queue;
  }
  CIsoSurface<byte>  &getSurface()
  {
    return surf;
  }
  CIsoSurface<byte> const &getSurface() const
  {
    return surf;
  }
  Volume<Vector>   &getV()
  {
    return v;
  }
  void setV(Volume<Vector>& new_v)
  {
    v = new_v;
  }
  Volume<Vector> const &getV() const
  {
    return v;
  }
  LogData        &getLog()
  {
    return log;
  }
  int getLastForegroundCount() const
  {
    return lastForegroundCount;
  }
  float getMaxDst()
  {
    return max_dst;
  }

  void set_ready(bool isReady);

  bool IsReady();

  //Postprocessing methods:
  void          assignImpLoops();
  void          fillGaps(float thr);
  void          thinSimplify(float thresh1, float thresh2, bool curve);
  void          detectBoundary(float thr, bool curve);
  void          detectSSBoundary(float thr);

  bool            isSimple(const coord3s &) const;
  bool            isSimple_tab(const coord3s &c) const;
  bool          isEndPoint(const coord3s &, bool curve) const;

  void smoothEdt();


protected:

  void getCC(int x, int y, int z, int &c_obj, int &c_back);

  bool isRemovable(int x, int y, int z, Volume<float> &dens);

  void  advect_conservative(Volume<float> &dens, Volume<float> &dens_prev);

  float transport_dens_diffusion(float weight, Volume<float> &dens,
                                 Volume<float> &dens_prev,
                                 const Interface &ipoints);

  void reaction(const std::vector<coord3s> &ipoints, Volume<float> &dens);

  float transport_dens_advection(float weight, Volume<float> &dens,
                                 Volume<float> &dens_prev,
                                 const Interface &ipoints);

  float transport_dens_advection_unconstrained(float weight,
      std::vector<coord3s> &out_coords, Volume<float> &tmp_dens, Volume<float> &dens,
      Volume<float> &dens_prev,
      std::vector<coord3s> &ipoints);

  float transport_dens_advection_project(float weight,
                                         std::vector<coord3s> &out_coords, Volume<float> &tmp_dens, Volume<float> &dens,
                                         Volume<float> &dens_prev,
                                         std::vector<coord3s> &ipoints);

  float transport_dens_advection_fwd(float weight, Volume<float> &dens,
                                     Volume<float> &dens_prev, Volume<float> &tmp_dens,
                                     std::vector<coord3s> &ipoints);

  float transport_dens_advection_back(float weight, Volume<float> &dens,
                                      Volume<float> &dens_prev, Volume<float> &tmp_dens,
                                      std::vector<coord3s> &new_ipoints,
                                      std::vector<coord3s> &ipoints);

  void transport_vel_advection(float weight, Volume<float> &dens,
                               Volume<Vector> &v,
                               std::vector<coord3s> &ipoints);

  void transport_vel_diffusion(float weight, const Interface &ipoints);

  void potential_new_interface(std::vector<coord3s> &ipoints,
                               std::vector<coord3s> &new_ipoints);
  int  initSimpleTable(const char*);
  void computeIFT(Volume<int> &ft);

private:
  bool                enableImportanceBoosting = false;
  Volume<float>       dens, dens1;          //this (and previous) density fields
  Volume<byte>        thin_img;           //
  Volume<float>       max_dens;           //
  Volume<float>       edt;              //Euclidean DT of initial surface
  Volume<Vector>      v;                //velocity field
  CIsoSurface<byte>   surf;             //initial surface (for rendering purposes)
  Volume<int>       ft;               //FT of initial surface
  std::vector<InverseFT>  ift;              //inverse FT
  NarrowBand        queue, queue1;
  index_cmp_minb      icmp;
  float(*diff_vecs)[3];
  std::unique_ptr<float[]> diff_weights;         //diffusion weights (precomputed)
  std::unique_ptr<float[]> diff_vecs_holder;
  int           r;
  int           sim_iter;
  float           tdens, curr_dst, max_dst;
  bool            advect_vel;
  float cs_importance_boost;      //value used during advection to boost importance of CS points
  bool force_monotonic;        //whether to force density to be monotonic on SS or not
  LogData         log;
  int lastForegroundCount;
  bool forceReady = false;
};

