#pragma once

#include <vector>
#include <utility>
#include <stdexcept>
#include <algorithm>

template<class Key, class Value, template<class...>class Storage=std::vector>
class flat_map
{
public:
  using value_type = std::pair<Key, Value>;

  using storage_t = Storage<value_type>;
  storage_t storage;

  using key_type = Key;
  using mapped_type = Value;


  // boilerplate:
  // iterators
  using iterator=typename storage_t::iterator;
  using const_iterator=typename storage_t::const_iterator;
  using reverse_iterator=typename storage_t::reverse_iterator;
  using const_reverse_iterator=typename storage_t::const_reverse_iterator;

  iterator begin()
  {
    return storage.begin();
  }
  const_iterator begin() const
  {
    return storage.cbegin();
  }
  const_iterator cbegin() const
  {
    return storage.cbegin();
  }
  reverse_iterator rbegin()
  {
    return storage.rbegin();
  }
  const_reverse_iterator rbegin() const
  {
    return storage.crbegin();
  }
  const_reverse_iterator crbegin() const
  {
    return storage.crbegin();
  }

  iterator end()
  {
    return storage.end();
  }
  const_iterator end() const
  {
    return storage.cend();
  }
  const_iterator cend() const
  {
    return storage.cend();
  }
  reverse_iterator rend()
  {
    return storage.rend();
  }
  const_reverse_iterator rend() const
  {
    return storage.crend();
  }
  const_reverse_iterator crend() const
  {
    return storage.crend();
  }


  //capacity
  size_t max_size() const
  {
    return storage.max_size();
  }
  size_t size() const
  {
    return storage.size();
  }
  bool empty() const
  {
    return storage.empty();
  }

  //element access
  Value& operator[](const key_type& k)
  {
    auto it = find(k);
    if (it != end()) return it->v;
    storage.emplace_back( k, Value{} );
    return storage.back().v;
  }

protected: // C++14, but you can just inject the lambda at point of use in 11:
  auto key_match( const key_type& k )
  {
    return [&k](value_type const& kv)
    {
      return kv.first == k;
    };
  }

public:
  iterator find(const key_type& k)
  {
    return std::find_if( begin(), end(), key_match(k) );
  }

  const_iterator find(const key_type& k) const
  {
    return const_cast<flat_map*>(this)->find(k);
  }

  size_t count (const key_type& k) const
  {
    return find(k) == end() ? 0 : 1;
  }

  // iterator-less query functions:
  mapped_type& at(const key_type& k)
  {
    auto it = find(k);
    if (it==end()) throw std::out_of_range("");
    return it->second;
  }

  const mapped_type& at(const key_type& k) const
  {
    return const_cast<flat_map*>(this)->at(k);
  }


  //modifiers
  size_t erase(const key_type& k)
  {
    auto it = std::remove_if(
      storage.begin(), storage.end(), key_match(k)
    );
    if (it == storage.end()) return 0;
    storage.erase( it, storage.end() );
    return 1;
  }
  // classic erase, for iterating:
  iterator  erase (const_iterator position)
  {
    return storage.erase(position);
  }
  iterator  erase (const_iterator first, const_iterator last)
  {
    return storage.erase(first, last);
  }

  std::pair<iterator,bool> insert (const value_type& val)
  {
    auto it = find(val.first);
    if (it != end()) return {it, false};
    storage.emplace_back(val);
    return {--(storage.end()), true};
  }
  template <class P>
  std::pair<iterator,bool> insert (P&& val)
  {
    auto it = find(val.first);
    if (it != end()) return {it, false};
    storage.emplace_back(std::forward<P>(val));
    return {--(storage.end()), true};
  }

  void clear()
  {
    storage.clear();
  }
};
