#ifndef LEVEL_SET_H
#define LEVEL_SET_H

#include <map>
#include "Volume.h"

typedef enum { FARAWAY = 0, ACCEPTED = 1, TRIAL = 2} STAT;
typedef float real;

#define TIME_MAX 1E+5
#define TIME_THR 50.0
#define ALPHA 0.2
#define EPS 1.0E-6

class coord
{
  int pos[3];
public:
  coord(int _x, int _y, int _z)
  {
    pos[0] = _x;
    pos[1] = _y;
    pos[2] = _z;
  }
  bool operator==(coord &c) const
  {
    return pos[0] == c.pos[0] && pos[1] == c.pos[1] && pos[2] == c.pos[2];
  }
  int operator[](int idx)
  {
    return pos[idx];
  }
};

struct _exist
{
  coord n;
  _exist(coord _n) : n(_n) {};
  bool operator()(coord node) const
  {
    return n == node;
  }
};

template<class COMP>
class Status
{
  real time;
  STAT flag;

public:
  Status(real _time = TIME_MAX, STAT _flag = FARAWAY)
  {
    if (COMP()(TIME_MAX, -TIME_MAX)) _time *= -1;
    time = _time;
    flag = _flag;
  }
  real &Time()
  {
    return time;
  }
  STAT &Flag()
  {
    return flag;
  }
};

typedef void (*DISPLAY_FUNC)(void);
typedef void (*WRITE_FUNC)(void);

//typedef multimap<real, coord, greater<real> > Map;

template< class T, class COMP >
class levelSet
{
  typedef std::multimap<real, coord, COMP> Map;
  typedef typename Map::iterator myIterator;

protected:
  Map map;
  std::vector< myIterator > pointers;

  std::vector< Status< COMP > > status;
  Volume< T > &img;
  int width, height, depth;
  DISPLAY_FUNC df;
  WRITE_FUNC wf;

public:

  levelSet(Volume< T > &im) : img(im)
  {
    width = img.getWidth();
    height = img.getHeight();
    depth = img.getDepth();
    pointers.resize(width * height * depth);
    status.resize(width * height * depth);
    df = NULL;
    wf = NULL;
  }
  virtual ~levelSet() {};

  // map functions
  void map_insert(int x, int y, int z, real time)
  {
    pointers[z * width * height + y * width + x] =
      map.insert(std::make_pair<real, coord>(std::move(time), coord(x, y, z)));
  }
  std::pair<real, coord> map_pop()
  {
    myIterator it = map.begin();
    std::pair<real, coord> res = (*it);
    map.erase(it);
    return res;
  }
  void map_update(int x, int y, int z, real newtime)
  {
    map.erase(pointers[z * width * height + y * width + x]);
    map_insert(x, y, z, newtime);
  }
  std::pair<real, coord> map_visit(int x, int y, int z)
  {
    return (*pointers[z * width * height + y * width + x]);
  }
  int map_size()
  {
    return map.size();
  }
  bool map_empty()
  {
    return map.size() == 0;
  }
  inline real &Tm(int x, int y, int z)
  {
    return status[z * width * height + y * width + x].Time();
  }
  inline STAT &FL(int x, int y, int z)
  {
    return status[z * width * height + y * width + x].Flag();
  }
  //display function
  void setDisplayFunction(DISPLAY_FUNC _df = NULL)
  {
    df = _df;
  }

  //write function
  void setWriteFunction(WRITE_FUNC _wf = NULL)
  {
    wf = _wf;
  }

  //level set functions
  virtual void Init(int x, int y, int z);
  virtual void Init(float thr);
  virtual real F(int x, int y, int z);

  real eikonal(float x, float f, float T1, float T2,
               float T3, float T4, float T5, float T6);
  real updateT(float tmin, int i, int j, int k);


  virtual float fastMarching();

  void output(Volume<float> &img, float max_time);
};

#endif







