#pragma once
#include <set>
#include <map>
#include <vector>
#include <boost/container/static_vector.hpp>
#include <unordered_map>

#include "Point3d.h"
#include "Volume.h"
#include "FeaturePoints.h"
#include "flat_map.h"

namespace surface
{

// Models the connectivity of a surface. Note that this does NOT mean
// this is a mesh - the surface can be a voxel-surface, for example.
// Also adds various operations that pertain to this graph.
class Graph

{

public:

  Graph();
  ~Graph();
  void      construct(const std::vector<Point3d> &boundary);
  int       size() const
  {
    return nodes.size();
  }

  class Node                  //A sampling-point (node) of the surface whose connectivity we represent
  {
  public:

    //typedef std::map<Node*, float>
    //Neighbors; //neighbors of this node in the graph, with corresponding edge-weights
    template <std::size_t size>
    struct static_vector_N
    {
      template <typename Value>
      using vector_type=boost::container::static_vector<Value, size>;
    };
    using Neighbors = flat_map<Node*, float, static_vector_N<26>::vector_type>;

    Node(int id_, unsigned int key_): id(id_), key(key_) {}
    void      addNeighbor(Node* n, float w)
    {
      neighbors.insert(std::make_pair(n, w));
      n->neighbors.insert(std::make_pair(this, w));
    }

    Neighbors
    neighbors;          //all boundary-neighbors of this, with values for their edge-weights
    int       id;             //location in boundary[] of this node
    unsigned int  key;            //unique id for this node on the boundary
  };

  //A curved path consisting of a set of neighbor-voxels in the graph
  class Path : public std::vector<Node*>
  {
  public:
    Node*     midPoint();
    float     length;
  };

  const Point3d  &nodePosition(int node_id)
  const //Utility to get 3D coordinates of a given node (passed by node-ID)
  {
    return (*boundary)[node_id];
  }

  const Point3d  &nodePosition(const Node* n)
  const //Utility to get 3D coordinates of a given node
  {
    return (*boundary)[n->id];
  }

  //Utility to get a Node by the ID of its corresponding boundary-point
  Node*     node(int node_id) const
  {
    return nid2nodes.find(node_id)->second;
  }

  typedef std::unordered_map<unsigned int, Node*> Nodes; //all boundary nodes, hashed by coord-key

  void      shortestPath(int pid1, int pid2, Path &) const;
  //Given two graph nodes, computes and returns shortest-path between them
  void      shortestPath(const FeaturePoints &, int skelpoint, Path &) const;
  //Given a set of EFT points for a skeleton point, computes and returns
  //best approximation of the shortest-path between the 'true' FT points
  //(which are an unknown subset of the EFT points)


  void      geoFront(int pid1, int pid2, Path &path, std::vector<float> &dist);
  void      geoLoci(const FeaturePoints &fp, int skelpoint, float thr,
                    std::vector<float> &dist);
  void      detectSaddles(std::vector<float> &dist);
  void      updateLength(Path &p) const;    //Update the 'length' field for a given path
  void      reset();

  Nodes     nodes;            //all boundary nodes, hashed by coord-key

//protected:

  std::unordered_map<int, Node*> nid2nodes;    //allows finding Nodes by point-index rather than coord-key
  const std::vector<Point3d>* boundary;   //boundary surface that this represents
};

}
