#include "FeaturePoints.h"
#include "PointSearch.h"
#include "VolumeHelpers.h"
#include "utils.h"
#include <unordered_map>
#include <cassert>

using namespace std;



FeaturePoints::FeaturePoints(const vector<Point3d> &boundary_): skelSearch(0),
  boundary(&boundary_),
   pointSearch(new PointSearch(*boundary, 1000))
{
}


FeaturePoints::~FeaturePoints()
{
  delete skelSearch;
}

vector<int> indexes;
vector<float> distances;

void FeaturePoints::construct(const vector<Point3d> &skeleton,
                              const Volume<Vector> &tangents,
                              const Volume<float> &importance,
                              const Volume<float> &edt)
{
  typedef std::unordered_map<unsigned int, ExtFeatures*> Coord2FPs;
  Coord2FPs vol2FP;
  int NS = skeleton.size();

  //Resize featurePoints and extFeaturePoints to the size of our skeleton
  featurePoints.resize(NS);
  extFeaturePoints.resize(NS);
  closestFeaturePoints.resize(NS);
  fullSkeleton = skeleton; //Cache the full skeleton for later use

  cout << "Start computing EFT..." << endl;

  //1. Compute extended feature-sets for all skel-points:
  for (int i = 0; i < NS; ++i)                    
  {
    const Point3d &s = skeleton[i];
    ExtFeatures &eft = extFeaturePoints[i];         //Collects all unique FP's for skeleton point 's'

    //Compute the extended FT of 's' (see Reniers' PhD)
    for (int x = s.x - 1; x <= s.x + 1; ++x)
      for (int y = s.y - 1; y <= s.y + 1; ++y)
        for (int z = s.z - 1; z <= s.z + 1; ++z)
        {

          unsigned int key = generateKey(coord3s(x, y, z));
          //'key' tells if we've computed the FP-set of this location
          //if so, reuse it, since it's expensive to recompute it
          ExtFeatures* fv = 0;
          Coord2FPs::const_iterator f = vol2FP.find(key);

          //No, we didn't compute the FP-set: compute it now
          if (f == vol2FP.end())
          {
            Point3d P(x, y, z);
            //Find closest surface point to current location
            int closest = pointSearch->closest(P);
            //Save closest-FP for this skeleton point
            closestFeaturePoints[i] = closest;        

            fv = new ExtFeatures();           //Create a new set of feature-points for location (x,y,z)
            fv->insert(closest);              //Add the first feature-point to this set

            //v:  velocity-field at skeleton-point location (normalized)
            Vector v = importance(x, y, z) > 0 ?
              tangents(x, y, z) :
              tangents(s.x, s.y, s.z);

            if (useFeatureReflection && v.norm2() > 1.0e-4)   //Cannot do anything if we don't have a good tangent-direction
            {
              Point3d B1 = (*boundary)[closest];      //B1 is the first feature-point of P, that's for sure.
              Point3d f1 = B1 - P;                    //f1: feature-vector corresponding to B1
              //(will happen if (x,y,z) is outside the surf-skeleton, for instance)

              Point3d tan = Point3d(v.u, v.v, v.w);
              Point3d f2 = tan * (f1.dot(tan) * 2) - f1;    //f2: reflection of f1 w.r.t. vector v (v = bisector of angle (f1,f2))
              Point3d B2 = P + f2;                          //B2: estimation of 2nd feature point

              fv->insert(pointSearch->closest(B2));         //Find closest surface-point to estimated 2nd feature-point B2...
              //...and add it to the feature-set too
            }


            float dist = (*boundary)[closest].dist(P);
            pointSearch->closest(dist, P, indexes, distances);
            for (int i = 0, IS = indexes.size(); i < IS; ++i)
              fv->insert(indexes[i]);

            //Record that we computed the feature-set for location (x,y,z)
            vol2FP.insert(make_pair(key, fv));        
          }
          else fv = f->second;                //Yes, FP-set for current location known; reuse it

          eft.insert(fv->begin(), fv->end());         //Append the FP-set of (x,y,z) to FP-set of 's'
        }

    featurePoints[i] = computeTwoFeatures(eft,s,edt);     //2. Compute strict feature-point-pair frome extended FP-set
  }


  for (Coord2FPs::iterator it = vol2FP.begin(), ie = vol2FP.end(); it != ie; ++it)
    delete it->second;                    //Done with the per-voxel feature-points

  cout << "End computing EFT." << endl;

  delete skelSearch;                      //5. Rebuild the kd-tree for the skeleton points

  skelSearch = (skeleton.size()) ? new PointSearch(skeleton, 1) : 0;
}


FeaturePoints::FeatureTriple
FeaturePoints::computeTwoFeatures(const ExtFeatures &fvec,    //Compute two-feature-set from extended FP-set as those two feature points
                                  const Point3d &s,
                                  const Volume<float> &edt)   //which maximize the angle w.r.t. the corresponding skeleton point
{
  int F = fvec.size();
  int i = 0;

  assert(F > 0);


  vector<Point3d> nv(F);                    //Precompute normalized feature-vectors
  for (ExtFeatures::const_iterator it = fvec.begin(); it != fvec.end(); ++it, ++i)
  {
    nv[i] = (*boundary)[*it] - s;
    nv[i].normalize();
    nv[i].value = *it;
  }

  if (F < 2)
  {
    // If EDT is smaller of equal to one then there can only be one feature point
    if (/*edt(s.x, s.y, s.z) <= 1.01f &&*/ F == 1)
      return FeatureTriple(nv[0].value, nv[0].value, nv[0].value);

    cout << "Skeleton point " << s << " does not have enough features: (" << F
      << "),  edt: " << edt(s.x, s.y, s.z) << endl;
  }

  float dp_min = 2;
  int i_min(-1), j_min(-1);
  for (int i = 0; i < F; ++i)
  {
    const Point3d &v1 = nv[i];
    for (int j = i + 1; j < F; ++j)
    {
      const Point3d &v2 = nv[j];
      float dp = v1.dot(v2);
      if (dp < dp_min)
      {
        dp_min = dp;
        i_min  = v1.value;
        j_min = v2.value;
      }
    }
  }

  // Find third feature by maximizing the area..
  float area_max = 0.f;
  int k_max = -1;

  const Point3d &v1 = (*boundary)[i_min];
  const Point3d &v2 = (*boundary)[j_min];
  for (int k = 0; k < F; ++k)
  {
    const Point3d &v3 = nv[k];

    float new_area = AreaTriangleSquared(v1, v2, v3);
    if (new_area > area_max)
    {
      area_max = new_area;
      k_max = v3.value;
    }
  }

  assert(i_min != -1);
  assert(j_min != -1);

  //Save strict FP-triple in final location
  return FeatureTriple(i_min, j_min, k_max);
}


void FeaturePoints::computeLigatureP(const vector<Point3d> &skeleton,
                                     const Volume<float> &importance)
{
  vector<Point3d> &bound = (*const_cast<vector<Point3d>*>(boundary));

  int  NS = skeleton.size();
  int  NB = bound.size();

  for (int i = 0; i < NB; ++i)
    bound[i].value = 0;

  float max_val = 0;
  for (int i = 0; i < NS; ++i)
  {
    const Point3d &s = skeleton[i];
    float imp = importance(s.x, s.y, s.z);

    int pid = closestFeaturePoints[i];

    float &val = bound[pid].value;
    val += imp;
    if (max_val < val) max_val = val;
  }

  cout << "LP: max #skelpoints/boundarypoint: " << max_val << endl;
  cout << "#surfpoints " << NB << " #skelpoints " << NS << endl;

  for (int i = 0; i < NB; ++i)
  {
    bound[i].value /= 1.0e-3;
    if (bound[i].value > 1) bound[i].value = 1;
  }
}

FeaturePoints::Coord2SKP& FeaturePoints::GetOrConstructCoord2SkelPointIndex()
{
  if (coord2SKPMap)
    return *coord2SKPMap;

  coord2SKPMap.reset(new Coord2SKP(fullSkeleton.size()));
  Coord2SKP& map = *coord2SKPMap;

  for (size_t i = 0; i < fullSkeleton.size(); i++)
  {
    auto& p = fullSkeleton[i];
    coord3s coord(p.x, p.y, p.z);
    unsigned key = generateKey(coord);
    map[key] = i;
  }

  return map;
}

void FeaturePoints::refreshPointSearch()
{
  pointSearch.reset(new PointSearch(*boundary, 1000));
}



