#include "NoiseCreator.h"
#include "utils.h"

#include<algorithm>


NoiseCreator::NoiseCreator(Volume<byte>& input, 
  surface::Graph& graph,
  std::shared_ptr<SkeletonModel> inputModel) :
input(input),
graph(graph),
inputModel(inputModel)
{
  CalculateNormalMap();
}


Point3d NoiseCreator::getRandomSphericalDir()
{
  std::normal_distribution<double> distribution;

  double x = distribution(generator);
  double y = distribution(generator);
  double z = distribution(generator);

  return Point3d(x, y, z);
}

Point3d NoiseCreator::getRandomTangentDir(Point3d normal)
{
  Point3d randomDir = getRandomSphericalDir();
  Point3d tangent = normal.cross(randomDir);
  tangent.normalize();

  return tangent;
}

std::vector<coord3s> getViableDirections(coord3s loc, coord3s dir)
{
  std::vector<coord3s> directions;

  for (int y = -1; y <= 1; y++)
  for (int x = -1; x <= 1; x++)
  {
    if (x == 0 && y == 0)
      continue;

    coord3s cur(x, y, 0);
    if (cur.dot(dir) > 0)
      directions.push_back(cur);
  }
  return directions;
}

void NoiseCreator::AddRandomStripe(NoisePickType noiseType, NoiseShape shape, coord3s start, float radius, int height, int length)
{
  AddSingleLineSurface(noiseType, shape, start, radius, height);

  if (length == 1) 
    return;

  std::random_device rd;
  std::mt19937 gen(rd());

  int key = generateKey(start);
  if (graph.nodes.find(key) == graph.nodes.end())
    return;

  auto node = graph.nodes[key];
  Point3d startP = graph.boundary->at(node->id);
  auto& neighbours = node->neighbors;

  if (startP.x != start.x || startP.y != start.y || startP.z != start.z)
    return;

#if 0
  std::uniform_int_distribution<> dis(0, neighbours.size() - 1);
  int randIndex = dis(gen);

  auto firstNeighbour(neighbours.begin());
  std::advance(firstNeighbour, randIndex);

  int boundaryId = firstNeighbour->first->id;
  Point3d neighbour = graph.boundary->at(boundaryId);
  Point3d dir = neighbour - Point3d(start.x, start.y, start.z);
  dir.normalize();
#endif

  Point3d& startNormal = normalsMap[start];
  Point3d dir = getRandomTangentDir(startNormal);

  coord3s iter = start;
  coord3s prevIter = start;
  std::vector<Point3d> filteredNeighbours;
  for (int i = 0; i < length - 1; i++)
  {
    auto node = graph.nodes[generateKey(iter)];
    auto& neighbours = node->neighbors;
    
    // Project current dir on local plane
    // Force smoothness by averaging normals (prevents early breaks)
    Point3d& curNormal = normalsMap[iter];
    Point3d& prevNormal = normalsMap[prevIter];
    Point3d normal = (curNormal + prevNormal) * 0.5f;
    dir = dir.projectPlane(normal);
    dir.normalize();

    for (auto& neigh : neighbours)
    {
      Point3d p = graph.boundary->at(neigh.first->id);
      Point3d newDir = p - Point3d(iter.x, iter.y, iter.z);
      newDir.normalize();
      if (newDir.dot(dir) >= 0.2f)
        filteredNeighbours.push_back(p);
    }

    if (filteredNeighbours.size() == 0)
      break;

    std::uniform_int_distribution<> dis(0, filteredNeighbours.size() - 1);
    Point3d newIter = filteredNeighbours[dis(gen)];

    prevIter = iter;
    iter = coord3s(newIter.x, newIter.y, newIter.z); 
    AddSingleLineSurface(noiseType, shape, iter, radius, height);
    filteredNeighbours.clear();
  }

}

void NoiseCreator::AddSingle(NoisePickType noiseType, NoiseShape shape, coord3s point, float radius)
{
  short maskValue = noiseType == NoisePickType::Convex ? 255 : 0;

  switch (shape)
  {
  case NoiseShape::Ball:
    input.addSphere(point, maskValue, radius);
    break;
  case NoiseShape::Cube:
    input.addCube(point, maskValue, radius);
    break;
  case NoiseShape::Gaussian: break;
  default: break;
  }

}


void NoiseCreator::AddSingleLine(NoisePickType noiseType, NoiseShape shape, coord3s point, float radius, Point3d* normal, int length)
{
  if (!normal || length == 1)
    return AddSingle(noiseType, shape, point, radius);

  int sign = noiseType == NoisePickType::Convex ? 1 : -1;
  Point3d start(point.x, point.y, point.z);
  for (int i = 0; i < length; i++)
  {
    Point3d newPoint = start + *normal * i * sign;
    AddSingle(noiseType, shape, coord3s(round(newPoint.x), round(newPoint.y), round(newPoint.z)), radius);
  }
}

void NoiseCreator::AddSingleLineSurface(NoisePickType noiseType, NoiseShape shape, coord3s point, float radius, int height)
{
  assert(normalsMap.find(point) != normalsMap.end());
  Point3d& normal = normalsMap[point];
  AddSingleLine(noiseType, shape, point, radius, &normal, height);
}

void NoiseCreator::GeneratePoints(double probability, NoisePickType noiseType, NoiseShape shape, float radius, int height, int length, int seed)
{
  std::bernoulli_distribution distribution(probability);
  generator.seed(seed);

  for (auto& node : *graph.boundary)
  {
    coord3s key(round(node.x), round(node.y), round(node.z));
    if (normalsMap.find(key) == normalsMap.end())
      continue;

    //Point3d& normal = normalsMap[key];

    if (distribution(generator))
      AddRandomStripe(noiseType, shape, key, radius, height, length);
  }
}

void NoiseCreator::CalculateNormalMap()
{
  auto& points = inputModel->GetThinPoints();
  auto& normals = inputModel->GetThinNormals();

  for (size_t i = 0; i < points.size(); i++)
  {
    Point3d& point = points[i];
    coord3s key(round(point.x), round(point.y), round(point.z));
    normalsMap[key] = normals[i];

  }
}
