#include "PointSearch.h"
#include <iostream>
#include <algorithm>

using namespace std;

std::mutex PointSearch::s_mtx;

PointSearch::PointSearch(const vector<Point3d> &points, int Nmax_): Nmax(Nmax_)
{
  std::unique_lock<std::mutex> lck (s_mtx);
  nn_idx       = new ANNidx[Nmax];
  nn_distances = new ANNdist[Nmax];
  //Local buffer keeping all grid point coordinates. Used by the ANN engine.
  data_pts     = annAllocPts(points.size(), 3);

  //Copy grid vertices in an ANN-compatible structure
  for (size_t i = 0; i < points.size(); ++i)
  {
    data_pts[i][0] = points[i].x;
    data_pts[i][1] = points[i].y;
    data_pts[i][2] = points[i].z;
  }

  //Create the KD-tree used by ANN to search close points
  kdt = new ANNkd_tree(data_pts, points.size(), 3); 
}


int PointSearch::closest(const Point3d &p)
{ 
  std::unique_lock<std::mutex> lck (s_mtx);
  //Search the closest point in the grid to the point 'p'.
  ANNcoord query_pt[3];
  //1. Pass the search point to ANN
  query_pt[0] = p.x;
  query_pt[1] = p.y;
  query_pt[2] = p.z;

  //2. Search the N closest points to 'p' using ANN
  kdt->annkSearch(query_pt, 1, nn_idx, nn_distances); 

  return nn_idx[0];
}


void PointSearch::closest(int N, const Point3d &p, vector<int> &indexes,
                          vector<float> &distances)
{
  std::unique_lock<std::mutex> lck (s_mtx);
  //Search the N closest points in the grid to the point 'p'.
  ANNcoord query_pt[3];
  //1. Pass the search point to ANN
  query_pt[0] = p.x;
  query_pt[1] = p.y;
  query_pt[2] = p.z;

  //2. Make sure we have enough space to return N search results
  indexes.resize(N);
  distances.resize(N);

  //3. Search the N closest points to 'p' using ANN
  kdt->annkSearch(query_pt, N, nn_idx, nn_distances);

  //4. Pass results back to caller
  for (int i = 0; i < N; ++i)
  {
    //4.1. Index of i-th closest point
    indexes[i]   = nn_idx[i];
    //4.2. Distance to i-th closest point. Note that ANN returns
    //     squared distances to the search point.
    distances[i] = sqrt(nn_distances[i]);
  }
}


void PointSearch::closest(float R, const Point3d &p, vector<int> &indexes,
                          vector<float> &distances)
{
  std::unique_lock<std::mutex> lck (s_mtx);
  //Search the closest points in the grid closer to the point 'p' than R
  ANNcoord query_pt[3];
  query_pt[0] = p.x;                  //1. Pass the search point to ANN
  query_pt[1] = p.y;
  query_pt[2] = p.z;

  int NP = kdt->annkFRSearch(query_pt, R * R, Nmax, nn_idx,
                             nn_distances); //2. Search the points within R to 'p' using ANN

  int NfoundPts = min(NP, Nmax);

  indexes.resize(
    NfoundPts);              //3. Make sure we have enough space to return N search results
  distances.resize(NfoundPts);


  for (int i = 0; i < NfoundPts; ++i)       //4. Pass results back to caller
  {
    indexes[i]   = nn_idx[i];             //4.1. Index of i-th closest point
    distances[i] = sqrt(
                     nn_distances[i]);       //4.2. Distance to i-th closest point. Note that ANN returns
  }                           //     squared distances to the search point.
}


PointSearch::~PointSearch()
{
  std::unique_lock<std::mutex> lck (s_mtx);
  delete kdt;                     //Delete all ANN-related data structures
  delete[] nn_distances;
  delete[] nn_idx;
  annDeallocPts(data_pts);
}



