#include "SaliencyMetricEngine.h"
#include <SkeletonReconstructor.h>
#include <utils.h>
#include <limits>
#include <Eigen/StdVector>
#include <algorithm>
#include <numeric>
using namespace Eigen;

SaliencyMetricEngine::SaliencyMetricEngine(FeaturePoints& featurePoints, Volume<float>& edt, Volume<float>& importance):
  featurePoints(featurePoints),
  edt(edt),
  importance(importance)
{
}

Volume<float> SaliencyMetricEngine::Compute(float impThreshold)
{
  Volume<float> saliency(edt.getWidth(), edt.getHeight(), edt.getDepth());

  for (int z = 0; z < edt.getDepth(); z++)
    for (int y = 0; y < edt.getHeight(); y++)
      for (int x = 0; x < edt.getWidth(); x++)
      {
        float imp = importance(x, y, z);
        float dist = edt(x, y, z);
        if (imp < impThreshold) continue;

        saliency(x, y, z) = dist > 0 ? imp / dist : 0;
      }

  return saliency;
}

coord3s SaliencyMetricEngine::FindTangentNext(size_t index, Volume<Vector>& velocity, surface::Graph & graph, float impThreshold, int distFallover)
{
  Point3d& p = featurePoints.fullSkeleton[index];
  unsigned int x = p.x, y = p.y, z = p.z;

  float dist = edt(x, y, z);
  Point3d orthoCenter = GetOrthoCenter(graph, index);
  Point3d diff = p - orthoCenter;
  Vector tangent(diff.x, diff.y, diff.z);

  if (dist < distFallover)
    tangent = velocity(x, y, z);

  coord3s prev = FindVelocityCoord(coord3s(x, y, z), impThreshold, tangent);
  return prev;
}


coord3s SaliencyMetricEngine::FindTangentNextHighDerivative(Volume<float>* derivative, size_t index, FeaturePoints::Coord2SKP& map,
  Volume<Vector>& velocity, surface::Graph & graph, float impThreshold, float derThreshold, int distFallover)
{
  Point3d& p = featurePoints.fullSkeleton[index];
  unsigned int x = p.x, y = p.y, z = p.z;

  float dist = edt(x, y, z);
  Point3d orthoCenter = GetOrthoCenter(graph, index);
  Point3d diff = p - orthoCenter;
  Vector tangent(diff.x, diff.y, diff.z);

  if (dist < distFallover)
    tangent = velocity(x, y, z);

  coord3s prev = FindVelocityCoordHighDerivative(derivative, map, graph, velocity,
    coord3s(x, y, z), impThreshold, tangent, derThreshold);

  return prev;
}

coord3s SaliencyMetricEngine::FindVelocityCoord(coord3s point, float impThreshold,  Vector velocity)
{
  Vector3f bestDir(velocity.u, velocity.v, velocity.w);
  bestDir.normalize();
  float maxEdt = 0;
  float maxDot = 0;
  coord3s maxCoord(-1, -1, -1);

  for (int z = -1; z <= 1; z++)
    for (int y = -1; y <= 1; y++)
      for (int x = -1; x <= 1; x++)
      {
        if (x == 0 && y == 0 && z == 0)
          continue;

        coord3s coord(point.x + x, point.y + y, point.z + z);
        Vector3f curDir(x, y, z);
        curDir.normalize();

        float dist = edt(coord);
        float dot = curDir.dot(bestDir);

        if (dot > maxDot && importance(coord) > impThreshold)
        {
          maxEdt = dist;
          maxCoord = coord;
          maxDot = dot;
        }
      }

  return maxCoord;
}

coord3s SaliencyMetricEngine::FindVelocityCoordHighDerivative(Volume<float>* derivative, FeaturePoints::Coord2SKP& map,
  surface::Graph& graph, Volume<Vector>& velocityField, coord3s point, float impThreshold, Vector velocity, float derThreshold)
{
  Vector3f bestDir(velocity.u, velocity.v, velocity.w);
  bestDir.normalize();
  float maxEdt = 0;
  float maxDot = 0;
  coord3s maxCoord(-1, -1, -1);

  for (int z = -1; z <= 1; z++)
    for (int y = -1; y <= 1; y++)
      for (int x = -1; x <= 1; x++)
      {
        if (x == 0 && y == 0 && z == 0)
          continue;

        coord3s coord(point.x + x, point.y + y, point.z + z);
        Vector3f curDir(x, y, z);
        curDir.normalize();

        float dist = edt(coord);
        float der = derivative
          ? (*derivative)(coord)
          : GetSingleDerrivative(map[generateKey(coord)], graph, velocityField, impThreshold, 1.8f);

        float dot = curDir.dot(bestDir);

        if (dot > maxDot && importance(coord) > impThreshold && der > derThreshold)
        {
          maxEdt = dist;
          maxCoord = coord;
          maxDot = dot;
        }
      }

  return maxCoord;
}

std::pair<coord3s, coord3s> SaliencyMetricEngine::GetWidestFeaturePoints(surface::Graph& graph, size_t fpIndex)
{
  auto& fp = featurePoints.featurePoints[fpIndex];

  Point3d a = graph.nodePosition(fp.first);
  Point3d b = graph.nodePosition(fp.second);

  return std::make_pair(coord3s(a.x, a.y, a.z), coord3s(b.x, b.y, b.z));
}

Point3d SaliencyMetricEngine::GetOrthoCenter(surface::Graph & graph, size_t fpIndex)
{
  auto& fp = featurePoints.featurePoints[fpIndex];

  Point3d a = graph.nodePosition(fp.first);
  Point3d b = graph.nodePosition(fp.second);
  Point3d c = graph.nodePosition(fp.third);

  return InCenterTriangleOrtho(a, b, c);
}


coord3s SaliencyMetricEngine::TraceBackwards(size_t index, surface::Graph& graph, Volume<Vector>& velocity,
  float impThreshold, int steps)
{
  FeaturePoints::Coord2SKP& map = featurePoints.GetOrConstructCoord2SkelPointIndex();
  Point3d& p = featurePoints.fullSkeleton[index];
  unsigned int x = p.x, y = p.y, z = p.z;
  auto& features = featurePoints.featurePoints[index];

  size_t currentIndex = index;
  coord3s currentCoord = coord3s(x, y, z);
  int i;
  for (i = 0; i < steps; i++)
  {
    Point3d orthoCenter = GetOrthoCenter(graph, currentIndex);
    Point3d diff = orthoCenter - p;
    Vector tangent(diff.x, diff.y, diff.z);
    currentCoord = FindVelocityCoord(currentCoord, impThreshold, tangent);
    if (currentCoord.x == -1)
      break;

    currentIndex = map[generateKey(currentCoord)];
  }

  return i == 0? coord3s(-1, -1, -1) : currentCoord;
}

coord3s SaliencyMetricEngine::TraceBackwardsDirect(size_t index, surface::Graph & graph, Volume<Vector>& velocity, int steps)
{
  FeaturePoints::Coord2SKP& map = featurePoints.GetOrConstructCoord2SkelPointIndex();
  Point3d& p = featurePoints.fullSkeleton[index];
  unsigned int x = p.x, y = p.y, z = p.z;
  auto& features = featurePoints.featurePoints[index];

  size_t currentIndex = index;
  coord3s currentCoord = coord3s(x, y, z);
  int i;
  for (i = 0; i < steps; i++)
  {
    Vector tangent = velocity(x, y, z);
    currentCoord = FindVelocityCoord(currentCoord, 0.0f, tangent);
    if (currentCoord.x == -1)
      break;

    currentIndex = map[generateKey(currentCoord)];
  }

  return i == 0 ? coord3s(-1, -1, -1) : currentCoord;
}

float SaliencyMetricEngine::GetSingleDerrivative(size_t index, surface::Graph& graph, Volume<Vector>& velocity, 
  float impThreshold, float salThreshold)
{
  Point3d& p = featurePoints.fullSkeleton[index];
  unsigned int x = p.x, y = p.y, z = p.z;
  auto& features = featurePoints.featurePoints[index];

  float dist = edt(x, y, z);
  float imp = importance(x, y, z);
  if (dist < 0.1f || imp <= impThreshold)
    return -1;

  coord3s prev = TraceBackwards(index, graph, velocity, impThreshold, 3);

  float measure = std::numeric_limits<float>::max();
  float prevEdt = prev.x != -1 ? edt(prev) : 0.0f;

  if (prev.x != -1 && abs((dist - prevEdt)) > 0.f)
    measure = abs((imp - importance(prev))) / abs((dist - edt(prev)));

  if (abs((dist - prevEdt)) < 0.01f)
  {
    float impDiff = abs(imp - importance(prev));
    if (dist < 5.0f && impDiff > 0.1f)
      return impDiff;
    else
      return 1.99f;
  }

  // In some cases the tangent vector is not correct, 
  // in these cases we check if the ratio of the importance and the EDT
  if (prev.x != -1 && imp / dist < salThreshold)
    return 0.0f;

  return measure;

}



Volume<float> SaliencyMetricEngine::ComputeDerivativeGradient(Volume<Vector>& velocity, surface::Graph& graph)
{
  Volume<float> saliency(edt.getWidth(), edt.getHeight(), edt.getDepth());

  float boost = edt.diagonal() / 2;


  Volume<Vector> gradientEdt = BuildGradientEdtVelocity(graph);
  for (size_t i = 0; i < featurePoints.fullSkeleton.size(); ++i)
  {
    Point3d& p = featurePoints.fullSkeleton[i];
    unsigned int x = p.x, y = p.y, z = p.z;
    float imp = importance(x, y, z);
    float dist = edt(x, y, z);
    if (dist < 0.1f)
    {
      saliency(x, y, z) = imp;
      continue;
    }

    if (velocity(x, y, z).norm2() < 1.0f && gradientEdt(x,y,z).norm2() > 0.05f)
      saliency(x, y, z) = imp;
    else
      saliency(x, y, z) = imp + boost;
  }

  return saliency;
}

Volume<float> SaliencyMetricEngine::ComputeDerivativeDirect(Volume<Vector>& velocity, surface::Graph& graph, float derThreshold)
{
  Volume<float> saliency(edt.getWidth(), edt.getHeight(), edt.getDepth());

  float boost = edt.diagonal() / 2;

  for (size_t i = 0; i < featurePoints.fullSkeleton.size(); ++i)
  {
    Point3d& p = featurePoints.fullSkeleton[i];
    unsigned int x = p.x, y = p.y, z = p.z;
    float imp = importance(x, y, z);
    float dist = edt(x, y, z);
    if (dist < 0.1f)
    {
      saliency(x, y, z) = imp;
      continue;
    }

    if (velocity(x, y, z).norm2() < 0.01f)
    {
      saliency(x, y, z) = imp + boost;
      continue;
    }



    coord3s prev = TraceBackwardsDirect(i, graph, velocity, 3);

    float measure = std::numeric_limits<float>::max();
    float prevEdt = prev.x != -1 ? edt(prev) : 0.0f;

    if (prev.x != -1 && abs((dist - prevEdt)) > 0.f)
      measure = abs((imp - importance(prev))) / abs((dist - edt(prev)));

    if (prev.x != -1 && abs((dist - prevEdt)) < 0.01f)
    {
      float impDiff = abs(imp - importance(prev));
      if (dist < 5.0f && impDiff > 0.1f)
        measure = impDiff;
      else
        measure = 1.99f;
    }

    saliency(x, y, z) = imp;
    if (measure > derThreshold)
      saliency(x, y, z) += boost;
  }


  return saliency;
}

Volume<float> SaliencyMetricEngine::ComputeDerivative(Volume<Vector>& velocity, surface::Graph& graph, float impThreshold, float derThreshold)
{
  Volume<float> saliency(edt.getWidth(), edt.getHeight(), edt.getDepth());

  float boost = edt.diagonal() / 2;

  for (size_t i = 0; i < featurePoints.fullSkeleton.size(); ++i)
  {
    Point3d& p = featurePoints.fullSkeleton[i];
    unsigned int x = p.x, y = p.y, z = p.z;
    float imp = importance(x, y, z);
    float measure = GetSingleDerrivative(i, graph, velocity, impThreshold, 1.8f);


    saliency(x, y, z) = imp;
    if (measure > derThreshold)
      saliency(x, y, z) += boost;
  }


  return saliency;
}

Volume<float> SaliencyMetricEngine::ComputeDerivativeCombiMeasure(Volume<Vector>& velocity, surface::Graph& graph, float impThreshold, float salThreshold)
{
  Volume<float> saliency(edt.getWidth(), edt.getHeight(), edt.getDepth());

  float boost = edt.diagonal() / 2;

  for (size_t i = 0; i < featurePoints.fullSkeleton.size(); ++i)
  {
    Point3d& p = featurePoints.fullSkeleton[i];
    unsigned int x = p.x, y = p.y, z = p.z;
    float measure = GetSingleDerrivative(i, graph, velocity, impThreshold, salThreshold);

    saliency(x, y, z) = measure;
  }

  return saliency;
}

Volume<float> SaliencyMetricEngine::ComputeReverseEDT(std::shared_ptr<SkeletonModel> skeletonModel, float impThreshold)
{
  Volume<float> saliency(edt.getWidth(), edt.getHeight(), edt.getDepth());
  Volume<byte> castedSkeletonPoints(edt.getWidth(), edt.getHeight(), edt.getDepth());

  SkeletonReconstructor skeletonReconstructor(skeletonModel->GetSkelPoints(), edt);
  SimpleVolume<coord3s> inverseFeatures;

  Volume<byte> reconstruction = skeletonReconstructor.Reconstruct(inverseFeatures);
  std::vector<Point3d> thinPoints = createThinSet(reconstruction);

  for (auto& point : thinPoints)
  {
    coord3s location(point.x, point.y, point.z);
    coord3s featureLocation = inverseFeatures(location);
    castedSkeletonPoints(featureLocation) = 1;
  }
  
  // the boost is set to the upper bound of the EDT.
  float boost = edt.diagonal() / 2;
  for (int z = 0; z < edt.getDepth(); z++)
    for (int y = 0; y < edt.getHeight(); y++)
      for (int x = 0; x < edt.getWidth(); x++)
      {
        float dist = edt(x, y, z);
        float imp = importance(x, y, z);
        if (dist < 0.1f || imp <= impThreshold)
          continue;

        saliency(x, y, z) = imp;
        if (castedSkeletonPoints(x, y, z))
          saliency(x, y, z) += boost;
      }

  return saliency;
}

float SaliencyMetricEngine::DistbetweenFeaturePoints(surface::AStar& aStarEngine, size_t index)
{
  auto& features = featurePoints.featurePoints[index];
  auto path = aStarEngine.shortestPath(features.first, features.second);
  return path ? path->length : 0;
}


Volume<float> SaliencyMetricEngine::ComputeGeodesic(surface::Graph& graph)
{
  Volume<float> saliency(edt.getWidth(), edt.getHeight(), edt.getDepth());
  surface::AStar aStarEngine(graph);

  saliency.clearVolume(0);
      
  for (size_t i = 0; i < featurePoints.fullSkeleton.size(); ++i)
  {
    Point3d& p = featurePoints.fullSkeleton[i];
    float dist = edt(p.x, p.y, p.z);
    float imp = DistbetweenFeaturePoints(aStarEngine, i);
    saliency(p.x, p.y, p.z) = dist > 0 ? imp / dist : 0;
  }

  return saliency;
}

Volume<float> SaliencyMetricEngine::ComputeGeodesicRaw(surface::Graph& graph, bool useEstimate)
{
  Volume<float> geodesics(edt.getWidth(), edt.getHeight(), edt.getDepth());
  surface::AStar aStarEngine(graph);

  geodesics.clearVolume(0);
  int chunkSize = 10000;
  int chunks = ceil((double)featurePoints.fullSkeleton.size() / chunkSize);
  int leftover = featurePoints.fullSkeleton.size() % chunkSize;
  leftover = leftover == 0 ? chunkSize : leftover;

  for (int j = 0; j < chunks; j++)
  {
    int imax = j < chunks - 1 ? chunkSize * (j+1) : chunkSize * j + leftover;
#pragma omp parallel for
    for (int i = j * chunkSize; i < imax; ++i)
    {
      Point3d& p = featurePoints.fullSkeleton[i];
      float dist = edt(p.x, p.y, p.z);
      float imp;

      if (useEstimate)
      {
        auto pair = GetWidestFeaturePoints(graph, i);
        imp = estimateShortestPathLength(pair.first, pair.second, 1.0f);
      }
      else
      {
        imp = DistbetweenFeaturePoints(aStarEngine, i);
      }

      geodesics(p.x, p.y, p.z) = dist > 0 ? imp : 0;
    }

     std::cout << "computed " << imax  << "/ " << featurePoints.fullSkeleton.size() << " points." << std::endl;
  }

  return geodesics;
}

Volume<float> SaliencyMetricEngine::ComputeEstimatedEuclidean(float preThreshold)
{
  Volume<float> eucl(edt.getWidth(), edt.getHeight(), edt.getDepth());

  for (int z = 0; z < edt.getDepth(); z++)
    for (int y = 0; y < edt.getHeight(); y++)
      for (int x = 0; x < edt.getWidth(); x++)
      {
        float dist = edt(x, y, z);
        if (dist <= 1.0f) continue;

        float imp = estimateFeatureLength(coord3s(x, y, z));
        eucl(x, y, z) = imp > preThreshold ? imp : 0;
      }

  return eucl;
}


Volume<Vector> SaliencyMetricEngine::BuildFeatureVelocity(surface::Graph& graph)
{
  Volume<Vector> velocity(edt.getWidth(), edt.getHeight(), edt.getDepth());
  velocity.clearVolume(Vector(0, 0, 0));

  for (size_t i = 0; i < featurePoints.fullSkeleton.size(); ++i)
  {
    Point3d& p = featurePoints.fullSkeleton[i];
    unsigned int x = p.x, y = p.y, z = p.z;

    float dist = edt(x, y, z);
    Point3d orthoCenter = GetOrthoCenter(graph, i);
    Point3d diff = p - orthoCenter;
    Vector tangent(diff.x, diff.y, diff.z);

    velocity(x, y, z) = tangent;
  }


  return velocity;
}

Volume<Vector> SaliencyMetricEngine::BuildGradientEdtVelocity(surface::Graph& graph)
{
  Volume<Vector> velocity(edt.getWidth(), edt.getHeight(), edt.getDepth());
  velocity.clearVolume(Vector(0, 0, 0));

  for (size_t i = 0; i < featurePoints.fullSkeleton.size(); ++i)
  {
    Point3d& p = featurePoints.fullSkeleton[i];
    unsigned int x = p.x, y = p.y, z = p.z;

    float dtx = (edt(x + 1, y, z) - edt(x - 1, y, z)) / 2;
    float dty = (edt(x, y + 1, z) - edt(x, y - 1, z)) / 2;
    float dtz = (edt(x, y, z + 1) - edt(x, y, z - 1)) / 2;

    Vector tangent(dtx, dty, dtz);

    velocity(x, y, z) = tangent;
  }

  return velocity;
}

Volume<Vector> SaliencyMetricEngine::BuildGradientImpVelocity(surface::Graph& graph, Volume<float>& imp)
{
  Volume<Vector> velocity(edt.getWidth(), edt.getHeight(), edt.getDepth());
  velocity.clearVolume(Vector(0, 0, 0));

  for (size_t i = 0; i < featurePoints.fullSkeleton.size(); ++i)
  {
    Point3d& p = featurePoints.fullSkeleton[i];
    unsigned int x = p.x, y = p.y, z = p.z;

    float dtx = imp(x + 1, y, z) - imp(x - 1, y, z);
    float dty = imp(x, y + 1, z) - imp(x, y - 1, z);
    float dtz = imp(x, y, z + 1) - imp(x, y, z - 1);

    Vector tangent(dtx, dty, dtz);

    velocity(x, y, z) = tangent;
  }

  return velocity;
}

float SaliencyMetricEngine::estimateShortestPathLength(coord3s origin, coord3s target, float accuracy)
{
  coord3s dir = target - origin;
  float dLength = sqrtf((target - origin).lengthSquared());
  int steps = std::max((int)(dLength * accuracy), 1);

  float prevEdt = edt(origin);
  coord3s prevStep = origin;
  float lengthEstimate = 0.0f;

  // estimate length by path integral estimate of the EDT
  for (int i = 0; i < steps; i++)
  {
    coord3s cur = origin + coord3s((i + 1) * dir.x / steps, (i + 1) * dir.y / steps,
      (i + 1) * dir.z / steps);

    float curEdt = edt(cur);
    float curDistSq = (cur - prevStep).lengthSquared();
    float diffEdt = curEdt - prevEdt;
    lengthEstimate += sqrtf(diffEdt * diffEdt + curDistSq);

    prevEdt = curEdt;
    prevStep = cur;
  }

  // See geo_distances.m, where the average error factor comes from
  const float averageErrorFactor = 1.113f;
  return lengthEstimate * averageErrorFactor;
}


float SaliencyMetricEngine::estimateFeatureLength(coord3s p)
{
  const float factor = sqrtf(2) * 2;
  unsigned int x = p.x, y = p.y, z = p.z;

  float dtx = (edt(x + 1, y, z) - edt(x - 1, y, z)) / 2;
  float dty = (edt(x, y + 1, z) - edt(x, y - 1, z)) / 2;
  float dtz = (edt(x, y, z + 1) - edt(x, y, z - 1)) / 2;

  float rate = 1.f - dtx*dtx - dty*dty - dtz * dtz + 0.0001f;
  float rep = sqrt(rate);

  return rate > 0.f ? edt(p) * rep * factor : 0.f;
}

float SaliencyMetricEngine::TraceVelocityStreamLine(Volume<float>* derivative, size_t index, Volume<Vector>& velocity, surface::Graph & graph,
  float impThreshold, FeaturePoints::Coord2SKP& map, std::vector<coord3s>* outPoints)
{
  const float derThreshold = 1.0f;

  Point3d& p = featurePoints.fullSkeleton[index];
  unsigned int x = p.x, y = p.y, z = p.z;
  float imp = importance(x, y, z);

  coord3s next = coord3s(x, y, z);
  float maxImportance = imp;
  int max = importance.diagonal();
  const float lowerBound = 2.0f;

  for (int i = 0; i < max; i++)
  {
    size_t nextIndex = map[generateKey(next)];
    float nextImportanceThreshold = std::max(impThreshold, maxImportance - lowerBound);

#if USE_TANGENT_DIRECT
    coord3s newNext = FindTangentNext(nextIndex, velocity, graph, nextImportanceThreshold);
#else
    coord3s newNext = FindTangentNextHighDerivative(derivative, nextIndex, map,
      velocity, graph, impThreshold, derThreshold);
#endif

    if (outPoints)
      outPoints->push_back(next);

    // If we cannot find a tangent vector then we are at an endpoint
    if (newNext.x == -1)
      break;

    size_t newNextIndex = map[generateKey(newNext)];
    float newImportance = importance(newNext);
    float newEdt = edt(newNext);
    //float measure = newImportance / newEdt;

    float measure = derivative
      ? (*derivative)(newNext)
      : GetSingleDerrivative(newNextIndex, graph, velocity, impThreshold, 1.8f);

    // If importance no longer increases, we are in the vicinity of the global root.
    // Or the tangent direction is seriously wrong...
    if (newImportance < nextImportanceThreshold)
      break;

    // Also break if skeleton point is not salient.
    if (measure < 0.1f)
      break;

    maxImportance = std::max(newImportance, maxImportance);
    next = newNext;
  }

  return maxImportance;

}

Volume<float> SaliencyMetricEngine::ComputeStreamlines(Volume<Vector>& velocity, surface::Graph & graph, float impThreshold)
{
  Volume<float> saliency(edt.getWidth(), edt.getHeight(), edt.getDepth());

  const float boost = edt.diagonal() / 2;
  const float salFallbackMeasure = 1.8f;

  FeaturePoints::Coord2SKP& map = featurePoints.GetOrConstructCoord2SkelPointIndex();
  Volume<float> derivative = ComputeDerivativeCombiMeasure(velocity, graph, 2.0f, salFallbackMeasure);

  for (size_t i = 0; i < featurePoints.fullSkeleton.size(); ++i)
  {
    Point3d& p = featurePoints.fullSkeleton[i];
    unsigned int x = p.x, y = p.y, z = p.z;
    float imp = importance(x, y, z);

    saliency(x, y, z) = imp;

    if (imp < impThreshold)
      continue;

    // From here we know the feature is salient, we trace down until we find something non-salient
    if (derivative(x, y, z) < 0.1f)
      saliency(x, y, z) = imp;
    else
      saliency(x, y, z) = TraceVelocityStreamLine(&derivative, i, velocity, graph, impThreshold, map, nullptr);
  }

  return saliency;
}


struct ImpPoint
{
  coord3s p;
  float imp;
  int orgIndex;
};


coord3s TraceMonotonicHistory(SimpleVolume<coord3s>& history, coord3s coord, int steps)
{
  coord3s iter = coord;
  for (int i = 0; i < steps; i++)
  {
    iter = history(iter);
    if (iter.x == -1)
      break;
  }

  return iter;
}


float SaliencyMetricEngine::TraceSingleMonotonic(size_t index, surface::Graph & graph, Volume<Vector>& velocity,
  float impThreshold, FeaturePoints::Coord2SKP& map, const Volume<float>& saliency, Volume<float>& angleMetric,
  SimpleVolume<coord3s>& history)
{
  Point3d& p = featurePoints.fullSkeleton[index];
  int i = p.z, j = p.y, k = p.x;

  Point3d orthoCenter = GetOrthoCenter(graph, index);
  Point3d diff = p - orthoCenter;
  Vector3f tangent(diff.x, diff.y, diff.z);
  float tangentLength = tangent.squaredNorm();
  tangent.normalize(); 

  float maxSal = impThreshold;
  coord3s maxCoord(-1, -1, -1);
  Vector3f maxDir;

  for (int m = i - 1; m <= i + 1; m++)
    for (int n = j - 1; n <= j + 1; n++)
      for (int p = k - 1; p <= k + 1; p++)
      {
        float imp = importance(p, n, m);
        float sal = saliency(p, n, m);

        if (imp < impThreshold)
          continue;


        Vector3f curDir(p - k, n - j, m - i);
        curDir.normalize();

#if USE_FLOW_METRIC
        // points with a wrong flow angle cannot contribute importance
        if (tangentLength > 1.f && tangent.dot(curDir) < 0.0f)
         continue;

        if (angleMetric(k, j, i) < 1.0f && angleMetric(k, j, i) != 0.0f)
          continue;
#endif

        // Non-salient points cannot contribute importance
        size_t nbIndex = map[generateKey(coord3s(p, n, m))];
        float measure = GetSingleDerrivative(nbIndex, graph, velocity, impThreshold, 1.8f);
        if (measure < 0.6f)
          continue;

        if (sal > maxSal)
        {
          maxSal = sal;
          maxDir = curDir;
          maxCoord = coord3s(p, n, m);
        }
      }

  if (maxCoord.x != -1)
  {
    float dotAngle = tangent.dot(maxDir);

    //angleMetric(k, j, i) += angleMetric(maxCoord) + 1.f - dotAngle;
    history(k, j, i) = maxCoord;

    coord3s curTarget = TraceMonotonicHistory(history, coord3s(k, j, i), 5);
    if (curTarget.x != -1)
    {
      Vector3f curDir = Vector3f(curTarget.x, curTarget.y, curTarget.z) -
        Vector3f(k, j, i);
      curDir.normalize();
      float dotAngle = tangent.dot(curDir);

      angleMetric(k, j, i) = abs( (importance(curTarget) - impThreshold) / (edt(curTarget) - edt(k, j, i))) + 0.001f;
    }
  }

  return maxSal;
}


Volume<float> SaliencyMetricEngine::ComputeGlobalMonotonic(Volume<Vector>& velocity, surface::Graph & graph, float impThreshold)
{
  Volume<float> saliency(edt.getWidth(), edt.getHeight(), edt.getDepth());
  Volume<float> angleMetric(edt.getWidth(), edt.getHeight(), edt.getDepth());
  SimpleVolume<coord3s> history(edt.getWidth(), edt.getHeight(), edt.getDepth());

  history.clearVolume(coord3s(-1, -1, -1));

  FeaturePoints::Coord2SKP& map = featurePoints.GetOrConstructCoord2SkelPointIndex();

  // Sort all points based on importance
  std::vector<size_t> sortedIndices(featurePoints.fullSkeleton.size());
  std::iota(sortedIndices.begin(), sortedIndices.end(), 0);
  sort(sortedIndices.begin(), sortedIndices.end(),
    [this](size_t a, size_t b) -> bool
  {
    Point3d pa = featurePoints.fullSkeleton[a];
    Point3d pb = featurePoints.fullSkeleton[b];
    return importance(pa.x, pa.y, pa.z) > importance(pb.x, pb.y, pb.z);
  });

  for (size_t i = 0; i < sortedIndices.size(); ++i)
  {
    size_t index = sortedIndices[i];
    Point3d& p = featurePoints.fullSkeleton[index];
    unsigned int x = p.x, y = p.y, z = p.z;
    float imp = importance(x, y, z);

    // By default a point retains its importance even if not salient.
    saliency(x, y, z) = imp;

    if (imp < impThreshold)
      continue;

    float measure = GetSingleDerrivative(index, graph, velocity, impThreshold, 1.8f);
    if (measure < 0.6f)
      continue;

    // From here we know the point is Salient
    saliency(x, y, z) = TraceSingleMonotonic(index, graph, velocity, imp, map, saliency, 
      angleMetric, history);
    
  }

  return saliency;
}
