
#include "Volume.h"
#include <string.h>
#include <stdio.h>
#include <vector>
#include <iostream>
#include <math.h>
#include <algorithm>
#include <assert.h>
#include <stdint.h>
#include <tuple>
#include <tuple>
#include <level_set.h>
#include <array>
#include "utils.h"

using namespace std;



template< class T >
void Volume<T>::getMinMax(T &min, T &max)
{
  int width = getWidth(), height = getHeight(), depth = getDepth();
  T gval;

  min = slices[0][0][0];
  max = min;

  for (int i = 0; i < depth; i++)
    for (int j = 0; j < width * height; j++)
    {
      gval = slices[i][0][j];
      if (gval < min) min = gval;
      if (gval > max) max = gval;
    }
}


template< class T >
void Volume<T>::toVector(vector<T> &vec)
{
  int width = getWidth(), height = getHeight(), depth = getDepth();
  int len = 0;

  vec.resize(width * height * depth);

  for (int i = 0; i < depth; i++)
    for (int j = 0; j < height; j++)
      for (int k = 0; k < width; k++)
        vec[len++] = slices[i][j][k];
}




#define PUT_PIXEL(a) { \
point3 p; p.coord[0]=a[0]; p.coord[1]=a[1]; p.coord[2]=a[2]; points.push_back(p); }

template< class T>
void Volume<T>::bresenham_linie_3D(int x1, int y1, int z1, int x2, int y2,
                                   int z2,
                                   vector<point3> &points)
{
  int i, dx, dy, dz, l, m, n, x_inc, y_inc, z_inc,
      err_1, err_2, dx2, dy2, dz2;
  int pixel[3];

  pixel[0] = x1;
  pixel[1] = y1;
  pixel[2] = z1;
  dx = x2 - x1;
  dy = y2 - y1;
  dz = z2 - z1;
  x_inc = (dx < 0) ? -1 : 1;
  l = abs(dx);
  y_inc = (dy < 0) ? -1 : 1;
  m = abs(dy);
  z_inc = (dz < 0) ? -1 : 1;
  n = abs(dz);
  dx2 = l << 1;
  dy2 = m << 1;
  dz2 = n << 1;

  if ((l >= m) && (l >= n))
  {
    err_1 = dy2 - l;
    err_2 = dz2 - l;
    for (i = 0; i < l; i++)
    {
      PUT_PIXEL(pixel);
      if (err_1 > 0)
      {
        pixel[1] += y_inc;
        err_1 -= dx2;
      }
      if (err_2 > 0)
      {
        pixel[2] += z_inc;
        err_2 -= dx2;
      }
      err_1 += dy2;
      err_2 += dz2;
      pixel[0] += x_inc;
    }
  }
  else if ((m >= l) && (m >= n))
  {
    err_1 = dx2 - m;
    err_2 = dz2 - m;
    for (i = 0; i < m; i++)
    {
      PUT_PIXEL(pixel);
      if (err_1 > 0)
      {
        pixel[0] += x_inc;
        err_1 -= dy2;
      }
      if (err_2 > 0)
      {
        pixel[2] += z_inc;
        err_2 -= dy2;
      }
      err_1 += dx2;
      err_2 += dz2;
      pixel[1] += y_inc;
    }
  }
  else
  {
    err_1 = dy2 - n;
    err_2 = dx2 - n;
    for (i = 0; i < n; i++)
    {
      PUT_PIXEL(pixel);
      if (err_1 > 0)
      {
        pixel[1] += y_inc;
        err_1 -= dz2;
      }
      if (err_2 > 0)
      {
        pixel[0] += x_inc;
        err_2 -= dz2;
      }
      err_1 += dy2;
      err_2 += dx2;
      pixel[2] += z_inc;
    }
  }
  PUT_PIXEL(pixel);
}



template< class T >
bool Volume<T>::readAVSField(char const *fname, int pad)
{
  FILE *pfile = fopen(fname, "r");
  if (!pfile)
  {
    return false;
  }

  char buffer[128];
  int dx = 0, dy = 0, dz = 0, dummy, width;

  while (!feof(pfile))
  {
    if (! fgets(buffer, 128, pfile))
    {
      fprintf(stderr, "error reading file\n");
      return false;
    }
    if (buffer[0] == '#') continue;             //Skip comments
    sscanf(buffer, "ndim=%d\n", &dx);
    assert(dx == 3);
    if (fscanf(pfile, "dim1=%d\n", &dx) == EOF)
    {
      fprintf(stderr, "error reading dx");
      return false;
    }
    assert(dx > 0);
    if (fscanf(pfile, "dim2=%d\n", &dy) == EOF)
    {
      fprintf(stderr, "error reading dy");
      return false;
    }
    assert(dy > 0);
    if (fscanf(pfile, "dim3=%d\n", &dz) == EOF)
    {
      fprintf(stderr, "error reading dz");
      return false;
    }
    assert(dz > 0);
    width = dx;

    printf("readAVSField: loading volume of dimensions: %d %d %d.\n", width, dy,
           dz);

    makeVolume(width + 2 * pad, dy + 2 * pad,
               dz + 2 * pad); //Allocate volume, pad it for safety with a small border
    if (fscanf(pfile, "nspace=%d\n", &dummy) == EOF)
    {
      fprintf(stderr, "error reading file\n");
      return false;
    }
    if (fscanf(pfile, "veclen=%d\n", &dummy) == EOF)
    {
      fprintf(stderr, "error reading file\n");
      return false;
    }
    if (fscanf(pfile, "data=%s\n", buffer) == EOF)
    {
      fprintf(stderr, "error reading file\n");
      return false;
    }

    if (!strcmp(buffer, "byte"))
    {
      if (! fgets(buffer, 128, pfile))
      {
        fprintf(stderr, "error reading file\n");
        return false;
      }
      if (! fgets(buffer, 128, pfile))
      {
        fprintf(stderr, "error reading file\n");
        return false;
      }
      if (! fgets(buffer, 128, pfile))
      {
        fprintf(stderr, "error reading file\n");
        return false;
      }
      
      byte* buffer = new byte[dx * dy * sizeof(byte)];
      for (int i = 0; i < dz; ++i)
      {
        int read = fread(buffer, sizeof(byte), dx * dy, pfile);
        const byte* buf = buffer;
        assert(read == dx * dy);
        Image<T> &slice = slices[i + pad];
        for (int k = 0; k < dy; ++k)
        {
          T* row = slice[k + pad] + pad;
          for (T* rend = row + width; row != rend;)
            *row++ = (T)(*buf++);
        }
      }
      delete[] buffer;
    }
    return true;
  }

  return true;
}

template< class T >
bool Volume<T>::readVTKField(char const *fname, int pad)
{
  FILE *pfile = fopen(fname, "rb");
  if (!pfile)
  {
    return false;
  }

  char buffer[128];
  int dx = 0, dy = 0, dz = 0, dummy1, dummy2, dummy3;

  while (!feof(pfile))
  {
    if (! fgets(buffer, 128, pfile))
    {
      fprintf(stderr, "error reading file\n");
      return false;
    }
    if (buffer[0] == '#') continue;
    break;
  }

  if (strcmp(buffer, "BINARY\n"))
  {
    printf("readVTKField: cannot load asci files.\n");
    return false;
  }

  if (! fgets(buffer, 128, pfile))
  {
    fprintf(stderr, "error reading file\n");
    return false;
  }
  if (strcmp(buffer, "DATASET STRUCTURED_POINTS\n"))
  {
    printf("readVTKField: can load only VTK structured points.\n");
    return false;
  }
  if (! fgets(buffer, 128, pfile))
  {
    fprintf(stderr, "error reading DIMENSIONS\n");
    return false;
  }
  sscanf(buffer, "DIMENSIONS %d %d %d\n", &dz, &dx, &dy);
  assert(dx > 0 && dy > 0 && dz > 0);

  if (! fgets(buffer, 128, pfile))
  {
    fprintf(stderr, "error reading ORIGIN\n");
    return false;
  }
  sscanf(buffer, "ORIGIN %d %d %d\n", &dummy1, &dummy2, &dummy3);

  if (! fgets(buffer, 128, pfile))
  {
    fprintf(stderr, "error reading SPACING\n");
    return false;
  }
  sscanf(buffer, "SPACING %d %d %d\n", &dummy1, &dummy2, &dummy3);

  if (! fgets(buffer, 128, pfile))
  {
    fprintf(stderr, "error reading POINT_DATA\n");
    return 0;
  }
  sscanf(buffer, "POINT_DATA %d\n", &dummy1);

  if (! fgets(buffer, 128, pfile))
  {
    fprintf(stderr, "error reading file\n");
    return 0;
  }
  if (! fgets(buffer, 128, pfile))
  {
    fprintf(stderr, "error reading file\n");
    return 0;
  }

  makeVolume(dx + 2 * pad, dy + 2 * pad, dz + 2 * pad);

  printf("readVTKField: loading volume of dimensions: %d %d %d.\n", dx, dy, dz);

  Image<byte> buf(dx, dy);

  for (int i = 0; i < dz; i++)
  {
    int read = fread(buf[0], sizeof(byte), dx * dy, pfile);
    assert(read == dx * dy);

    Image<T> &slice = this->slices[i + pad];

    for (int k = 0; k < dy; k++)
      for (int j = 0; j < dx; j++)
      {
        int val = buf[k][j];
        if (val) val = (val << 1) - 1;
        slice[k + pad][j + pad] = (T)val;
      }
  }

  return true;
}

template<>
bool Volume<float>::readVTKField(char const *fname, int pad)
{
  FILE *pfile = fopen(fname, "rb");
  if (!pfile)
  {
    return false;
  }

  char buffer[128];
  int dx = 0, dy = 0, dz = 0, dummy1, dummy2, dummy3;

  while (!feof(pfile))
  {
    if (!fgets(buffer, 128, pfile))
    {
      fprintf(stderr, "error reading file\n");
      return false;
    }
    if (buffer[0] == '#') continue;
    break;
  }

  if (strcmp(buffer, "BINARY\n"))
  {
    printf("readVTKField: cannot load asci files.\n");
    return false;
  }

  if (!fgets(buffer, 128, pfile))
  {
    fprintf(stderr, "error reading file\n");
    return false;
  }
  if (strcmp(buffer, "DATASET STRUCTURED_POINTS\n"))
  {
    printf("readVTKField: can load only VTK structured points.\n");
    return false;
  }
  if (!fgets(buffer, 128, pfile))
  {
    fprintf(stderr, "error reading DIMENSIONS\n");
    return false;
  }
  sscanf(buffer, "DIMENSIONS %d %d %d\n", &dz, &dx, &dy);
  assert(dx > 0 && dy > 0 && dz > 0);

  if (!fgets(buffer, 128, pfile))
  {
    fprintf(stderr, "error reading ORIGIN\n");
    return false;
  }
  sscanf(buffer, "ORIGIN %d %d %d\n", &dummy1, &dummy2, &dummy3);

  if (!fgets(buffer, 128, pfile))
  {
    fprintf(stderr, "error reading SPACING\n");
    return false;
  }
  sscanf(buffer, "SPACING %d %d %d\n", &dummy1, &dummy2, &dummy3);

  if (!fgets(buffer, 128, pfile))
  {
    fprintf(stderr, "error reading POINT_DATA\n");
    return 0;
  }
  sscanf(buffer, "POINT_DATA %d\n", &dummy1);

  if (!fgets(buffer, 128, pfile))
  {
    fprintf(stderr, "error reading file\n");
    return 0;
  }
  if (!fgets(buffer, 128, pfile))
  {
    fprintf(stderr, "error reading file\n");
    return 0;
  }

  makeVolume(dx + 2 * pad, dy + 2 * pad, dz + 2 * pad);

  printf("readVTKField: loading volume of dimensions: %d %d %d.\n", dx, dy, dz);

  Image<float> buf(dx, dy);

  for (int i = 0; i < dz; i++)
  {
    int read = fread(buf[0], sizeof(float), dx * dy, pfile);
    assert(read == dx * dy);

    Image<float> &slice = this->slices[i + pad];

    for (int k = 0; k < dy; k++)
      for (int j = 0; j < dx; j++)
      {
        float val = buf[k][j];
        slice[k + pad][j + pad] = (float)val;
      }
  }

  return true;
}

template< class T >
bool Volume<T>::readBinaryField(char const *fname, int pad, int &np, int * &points, float * &normals)
{
  FILE *pfile = fopen(fname, "rb");
  if (!pfile)
  {
    return false;
  }

  int dx = 0, dy = 0, dz = 0;

  unsigned char compressed = fgetc(pfile);

  assert(!compressed);

  uint64_t shape[3];
  {
    size_t nread = fread(&shape, 3 * sizeof(uint64_t), 1, pfile);
    if (nread != 3 * sizeof(uint64_t))
      fprintf(stderr, "error getting size from file\n");
  }
  dz = (int)shape[0];
  dy = (int)shape[1];
  dx = (int)shape[2];

  assert(dx > 0 && dy > 0 && dz > 0);

  printf("readBinaryField: loading volume of dimensions: %d %d %d.\n", dx, dy,
         dz);

  makeVolume(dx + 2 * pad, dy + 2 * pad, dz + 2 * pad);

  Image<byte> buf(dx, dy);

  for (int i = 0; i < dz; i++)
  {
    int read = fread(buf[0], sizeof(byte), dx * dy, pfile);
    assert(read == dx * dy);

    Image<T> &slice = this->slices[i + pad];

    for (int k = 0; k < dy; k++)
      for (int j = 0; j < dx; j++)
      {
        int val = buf[k][j];
        if (val) val = 255;
        slice[k + pad][j + pad] = (T)val;
      }
  }

  np = 0;
  {
    size_t nread = fread(&np, sizeof(int), 1, pfile);
    if (nread != sizeof(int))
      fprintf(stderr, "error getting number of points from file\n");
  }
  points = new int[3 * np];
  normals = new float[3 * np];

  {
    size_t nread = fread(points, 3 * sizeof(int), np, pfile);
    if (nread != 3 * sizeof(int) * np)
      fprintf(stderr, "error getting points from file\n");
  }
  {
    size_t nread = fread(normals, 3 * sizeof(float), np, pfile);
    if (nread != 3 * sizeof(float) * np)
      fprintf(stderr, "error getting normals from file\n");
  }
  fclose(pfile);

  return true;
}

template< class T >
void Volume<T>::exportAVSField(char const *fname)
{
  FILE *pfile = fopen(fname, "w");

  fprintf(pfile, "# AVS field file %s\n#\n", fname);
  fprintf(pfile, "%s", "ndim=3\n");
  fprintf(pfile, "dim1=%d\n", getWidth());
  fprintf(pfile, "dim2=%d\n", getHeight());
  fprintf(pfile, "dim3=%d\n", getDepth());
  fprintf(pfile, "%s", "nspace=3\n");
  fprintf(pfile, "%s", "veclen=1\n");
  //fprintf(pfile, "%s", "data=byte\n");
  fprintf(pfile, "%s", "data=float\n");
  fprintf(pfile, "%s", "field=uniform\n");
  fprintf(pfile, "min_ext=%f %f %f\n", 0.0f, 0.0f, 0.0f);
  fprintf(pfile, "max_ext=%f %f %f\n",
          (float)getWidth(), (float)getHeight(), (float)getDepth());

  fputc(0x0c, pfile);
  fputc(0x0c, pfile);

  //Image<byte> img(getWidth(), getHeight());

  Image<float> slice(getWidth(), getHeight());
  char *odata = (char*)slice.getPixels();

  for (int i = 0; i < getDepth(); i++)
  {
    /*
    for(int j=0;j<getWidth()*getHeight();j++)  {
      img[0][j] = (byte)(255.0f*slices[i][0][j]);
    }

    fwrite(img.getPixels(), sizeof(byte), img.getSize(), pfile);
    */

    char *idata = (char*)slices[i].getPixels();

    //swap bytes (endianness issue)
    for (int j = 0; j < getWidth()*getHeight(); j++)
    {
      odata[j * 4 + 0] = idata[j * 4 + 3];
      odata[j * 4 + 1] = idata[j * 4 + 2];
      odata[j * 4 + 2] = idata[j * 4 + 1];
      odata[j * 4 + 3] = idata[j * 4 + 0];
    }

    fwrite(slice.getPixels(), sizeof(float), slice.getSize(), pfile);

  }

  fclose(pfile);
}

template< class T >
void Volume<T>::exportVolField(char const *fname, T threshold, byte fg_value, int pad)
{
  FILE *pfile = fopen(fname, "w");

  fprintf(pfile, "X: %d\n", getWidth() - 2 * pad);
  fprintf(pfile, "Y: %d\n", getHeight() - 2 * pad);
  fprintf(pfile, "Z: %d\n", getDepth() - 2 * pad);
  fprintf(pfile, "Version: 2\n");
  fprintf(pfile, "Voxel-Size: 1\n");
  fprintf(pfile, "Alpha-Color: 0\n");
  fprintf(pfile, "Int-Endian: 0123\n");
  fprintf(pfile, "Voxel-Endian: 0\n");
  fprintf(pfile, "Res-X: 1.000000\n");
  fprintf(pfile, "Res-Y: 1.000000\n");
  fprintf(pfile, "Res-Z: 1.000000\n");
  fputc(0x2E, pfile);
  fputc(0x0A, pfile);

  Image<byte> slice(getWidth() - 2 * pad, getHeight() - 2 * pad);
  for (int i = pad; i < getDepth() - pad; i++)
  {
    for (int j = pad; j < getHeight() - pad; j++)
    {
      for (int k = pad; k < getWidth() - pad; k++)
      {
        T val =  (slices[i][j][k] >= threshold) ? fg_value : 0;
        slice[j - pad][k - pad] = (byte)val;
      }
    }
    fwrite(slice.getPixels(), sizeof(byte), slice.getSize(), pfile);
  }

  fclose(pfile);
}

// Hack needed for msvc compatibility
int removeCarriageReturn(char* buf, int length)
{

  if (length > 3 && buf[length - 3] == '\r')
  {
    buf[length - 3] = '\n';
    buf[length - 2] = '\0';
    return length - 1;
  }

  return length;
}

template<class T>
void Volume<T>::exportVTKField(char const * fname)
{
  FILE *pfile = fopen(fname, "wb");
  if (!pfile)
  {
    return;
  }

  char buffer[128];
  int dx = getWidth(), dy = getHeight(), dz = getDepth();
  int dummy1, dummy2, dummy3;

  char binaryOp[] = "BINARY\n";
  int binarySize = removeCarriageReturn(binaryOp, sizeof(binaryOp));
  fwrite(binaryOp, sizeof(char), binarySize - 1, pfile);

  char sPointsOp[] = "DATASET STRUCTURED_POINTS\n";
  int sPointsSize = removeCarriageReturn(sPointsOp, sizeof(sPointsOp));
  fwrite(sPointsOp, sizeof(char), sPointsSize - 1, pfile);

  int dimBufSize = sprintf(buffer, "DIMENSIONS %d %d %d\n", dz, dx, dy);
  dimBufSize = removeCarriageReturn(buffer, dimBufSize);
  fwrite(buffer, sizeof(char), dimBufSize, pfile);

  int originBufSize = sprintf(buffer, "ORIGIN %d %d %d\n", dz/2, dx/2, dy/2);
  originBufSize = removeCarriageReturn(buffer, originBufSize);
  fwrite(buffer, sizeof(char), originBufSize, pfile);

  int spacingBufSize = sprintf(buffer, "SPACING %d %d %d\n", 1, 1, 1);
  spacingBufSize = removeCarriageReturn(buffer, spacingBufSize);
  fwrite(buffer, sizeof(char), spacingBufSize, pfile);

  int pointBufSize = sprintf(buffer, "POINT_DATA %d\n", dx * dy * dz);
  pointBufSize = removeCarriageReturn(buffer, pointBufSize);
  fwrite(buffer, sizeof(char), pointBufSize, pfile);

  char scalarsOp[] = "SCALARS voxel_data unsigned_char\n";
  int scalarsSize = removeCarriageReturn(scalarsOp, sizeof(scalarsOp));
  fwrite(scalarsOp, sizeof(char), scalarsSize - 1, pfile);

  char lookupOp[] = "LOOKUP_TABLE default\n";
  int lookupOpSize = removeCarriageReturn(lookupOp, sizeof(lookupOp));
  fwrite(lookupOp, sizeof(char), lookupOpSize - 1, pfile);

  Image<byte> buf(dx, dy);

  for (int i = 0; i < dz; i++)
  {
    Image<T> &slice = this->slices[i];

    for (int k = 0; k < dy; k++)
      for (int j = 0; j < dx; j++)
      {
        byte val = slice[k][j];
        if (val) val = (val + 1) >> 1;
        buf[k][j] = val;
      }

    fwrite(buf[0], sizeof(byte), dx * dy, pfile);
  }

  fclose(pfile);
}

template<>
void Volume<float>::exportVTKField(char const * fname)
{
  FILE *pfile = fopen(fname, "wb");
  if (!pfile)
  {
    return;
  }

  char buffer[128];
  int dx = getWidth(), dy = getHeight(), dz = getDepth();
  int dummy1, dummy2, dummy3;

  char binaryOp[] = "BINARY\n";
  int binarySize = removeCarriageReturn(binaryOp, sizeof(binaryOp));
  fwrite(binaryOp, sizeof(char), binarySize - 1, pfile);

  char sPointsOp[] = "DATASET STRUCTURED_POINTS\n";
  int sPointsSize = removeCarriageReturn(sPointsOp, sizeof(sPointsOp));
  fwrite(sPointsOp, sizeof(char), sPointsSize - 1, pfile);

  int dimBufSize = sprintf(buffer, "DIMENSIONS %d %d %d\n", dz, dx, dy);
  dimBufSize = removeCarriageReturn(buffer, dimBufSize);
  fwrite(buffer, sizeof(char), dimBufSize, pfile);

  int originBufSize = sprintf(buffer, "ORIGIN %d %d %d\n", dz / 2, dx / 2, dy / 2);
  originBufSize = removeCarriageReturn(buffer, originBufSize);
  fwrite(buffer, sizeof(char), originBufSize, pfile);

  int spacingBufSize = sprintf(buffer, "SPACING %d %d %d\n", 1, 1, 1);
  spacingBufSize = removeCarriageReturn(buffer, spacingBufSize);
  fwrite(buffer, sizeof(char), spacingBufSize, pfile);

  int pointBufSize = sprintf(buffer, "POINT_DATA %d\n", dx * dy * dz);
  pointBufSize = removeCarriageReturn(buffer, pointBufSize);
  fwrite(buffer, sizeof(char), pointBufSize, pfile);

  char scalarsOp[] = "SCALARS voxel_data float\n";
  int scalarsSize = removeCarriageReturn(scalarsOp, sizeof(scalarsOp));
  fwrite(scalarsOp, sizeof(char), scalarsSize - 1, pfile);

  char lookupOp[] = "LOOKUP_TABLE default\n";
  int lookupOpSize = removeCarriageReturn(lookupOp, sizeof(lookupOp));
  fwrite(lookupOp, sizeof(char), lookupOpSize - 1, pfile);

  Image<float> buf(dx, dy);

  for (int i = 0; i < dz; i++)
  {
    Image<float> &slice = this->slices[i];

    for (int k = 0; k < dy; k++)
      for (int j = 0; j < dx; j++)
      {
        float val = slice[k][j];
        buf[k][j] = val;
      }

    fwrite(buf[0], sizeof(float), dx * dy, pfile);
  }

  fclose(pfile);
}

template <class T>
std::tuple<coord3s, coord3s> Volume<T>::getBounds(coord3s p, coord3s b, short r) const
{
  coord3s topleft(max(p.x - r, 0), max(p.y - r, 0), max(p.z - r, 0));
  coord3s botright(min((short)(p.x + r), b.x), min((short)(p.y + r), b.y), min((short)(p.z + r), b.z));
  return std::make_tuple(topleft, botright);
}

template <class T>
std::tuple<coord3s, coord3s> Volume<T>::getBounds(coord3s p, short r) const
{
  coord3s b(getWidth() - 1, getHeight() - 1, getDepth() - 1);
  coord3s topleft(max(p.x - r, 0), max(p.y - r, 0), max(p.z - r, 0));
  coord3s botright(min((short)(p.x + r), b.x), min((short)(p.y + r), b.y), min((short)(p.z + r), b.z));
  return std::make_tuple(topleft, botright);
}


template <class T>
void Volume<T>::addSphere(coord3s coord, short maskValue, float radius)
{
  coord3s topleft, botright;
  std::tie(topleft, botright) = getBounds(coord, static_cast<int>(ceil(radius)));

  for (int k = topleft.z; k <= botright.z; k++)
    for (int j = topleft.y; j <= botright.y; j++)
      for (int i = topleft.x; i <= botright.x; i++)
        if ((coord - coord3s(i, j, k)).lengthSquared() < radius * radius)
          operator()(i, j, k) = maskValue;
}

template <class T>
void Volume<T>::addCube(coord3s coord, short maskValue, short radius)
{
  coord3s topleft, botright;
  std::tie(topleft, botright) = getBounds(coord, radius);

  for (int k = topleft.z; k <= botright.z; k++)
    for (int j = topleft.y; j <= botright.y; j++)
      for (int i = topleft.x; i <= botright.x; i++)
        operator()(i, j, k) = maskValue;

}


template <class T>
T Volume<T>::erode(coord3s coord, int radius, const Volume<T>& tVol, T t, bool inflate) const
{
  coord3s topleft, botright;
  std::tie(topleft, botright) = getBounds(coord, radius);
  T minValue = std::numeric_limits<T>::max();

  int falloverCount = 0;

  for (int k = topleft.z; k <= botright.z; k++)
    for (int j = topleft.y; j <= botright.y; j++)
      for (int i = topleft.x; i <= botright.x; i++)
      {
        if (tVol(i, j, k) > (T)t)
          minValue = min(minValue, operator()(i, j, k));
        else if (operator()(i, j, k) <= 0)
          falloverCount++;
      }

  if (inflate && !falloverCount)
    minValue += radius;

  return minValue;
}

template <class T>
T Volume<T>::dilate(coord3s coord, int radius, const Volume<T>& tVol, T t) const
{
  coord3s topleft, botright;
  std::tie(topleft, botright) = getBounds(coord, radius);
  T maxValue = std::numeric_limits<T>::min();


  for (int k = topleft.z; k <= botright.z; k++)
    for (int j = topleft.y; j <= botright.y; j++)
      for (int i = topleft.x; i <= botright.x; i++)
        if(tVol(i, j, k) > (T)t)
          maxValue = max(maxValue, operator()(i, j, k));

  return maxValue;
}

template <class T>
Volume<T> Volume<T>::erode( int r, const Volume<T>& tVol, T t, bool inflate) const
{
  Volume<T> out(getWidth(), getHeight(), getDepth());

  for (int i = 0; i < getDepth(); i++)
    for (int j = 0; j < getHeight(); j++)
      for (int k = 0; k < getWidth(); k++)
        if (tVol(k, j, i) >(T)t)
          out(k, j, i) = erode(coord3s(k, j, i), r, tVol, t, inflate);
        else
          out(k, j, i) = operator()(k, j, i);

  return out;
}

template <class T>
Volume<T> Volume<T>::dilate(int r, const Volume<T>& tVol, T t) const
{
  Volume<T> out(getWidth(), getHeight(), getDepth());

  for (int i = 0; i < getDepth(); i++)
    for (int j = 0; j < getHeight(); j++)
      for (int k = 0; k < getWidth(); k++)
        if (tVol(k, j, i) >(T)t)
          out(k, j, i) = dilate(coord3s(k, j, i), r, tVol, t);
        else
          out(k, j, i) = operator()(k, j, i);

  return out;
}

template<class T>
T Volume<T>::median(coord3s coord, int r, const Volume<T>& tVol, T threshold) const
{
  if(operator()(coord) < threshold)
    return operator()(coord);

  coord3s topleft, botright;
  std::tie(topleft, botright) = getBounds(coord, r);
  std::vector<float> values;
  
  for (int k = topleft.z; k <= botright.z; k++)
    for (int j = topleft.y; j <= botright.y; j++)
      for (int i = topleft.x; i <= botright.x; i++)
      {
        float value = operator()(i, j, k);
        if (tVol(i, j, k) > threshold)
          values.push_back(value);
      }

  if (values.size() == 1)
    return operator()(coord);

  std::nth_element(values.begin(), values.begin() + values.size() / 2 - 1, values.end());
  return values[values.size() / 2 - 1];
}

template<class T>
T Volume<T>::mean(coord3s coord, int r, const Volume<T>& tVol, T threshold) const
{
  if (tVol(coord) < threshold)
    return operator()(coord);

  coord3s topleft, botright;
  std::tie(topleft, botright) = getBounds(coord, r);
  std::vector<float> values;
  
  float sum = 0.0f;
  int count = 0;

  for (int k = topleft.z; k <= botright.z; k++)
    for (int j = topleft.y; j <= botright.y; j++)
      for (int i = topleft.x; i <= botright.x; i++)
      {
        float value = operator()(i, j, k);
        if (tVol(i, j, k) > threshold)
        {
          count++;
          sum += value;
        }
      }

  return sum / count;
}


template<class T>
T Volume<T>::trilinear(coord3f coord) const
{
  coord3f tl(floor(coord.x), floor(coord.y), floor(coord.z));
  coord3f br(ceil(coord.x), ceil(coord.y), ceil(coord.z));

  std::array<float, 4> face0 =
  { (float) operator()(tl.x, tl.y, tl.z), (float) operator()(br.x, tl.y, tl.z),
    (float) operator()(tl.x, br.y, tl.z), (float) operator()(br.x, br.y, tl.z) };

  std::array<float, 4> face1 =
  { (float) operator()(tl.x, tl.y, br.z), (float) operator()(br.x, tl.y, br.z),
    (float) operator()(tl.x, br.y, br.z), (float) operator()(br.x, br.y, br.z) };

  return Trilinear<float>(tl, br, coord, face0, face1);
}

template<class T>
Volume<T> Volume<T>::medianFilter(int r, const Volume<T>& tVol, T t) const
{
  Volume<T> out(getWidth(), getHeight(), getDepth());

  for (int i = 0; i < getDepth(); i++)
    for (int j = 0; j < getHeight(); j++)
      for (int k = 0; k < getWidth(); k++)
        if (tVol(k, j, i) >(T)t)
          out(k, j, i) = median(coord3s(k, j, i), r, tVol, t);
        else
          out(k, j, i) = operator()(k, j, i);

  return out;
}

template<class T>
Volume<T> Volume<T>::meanFilter(int r, const Volume<T>& tVol, T t) const
{
  Volume<T> out(getWidth(), getHeight(), getDepth());

  for (int i = 0; i < getDepth(); i++)
    for (int j = 0; j < getHeight(); j++)
      for (int k = 0; k < getWidth(); k++)
        if (tVol(k, j, i) >(T)t)
          out(k, j, i) = mean(coord3s(k, j, i), r, tVol, t);
        else
          out(k, j, i) = operator()(k, j, i);

  return out;
}

template<class T>
Volume<T> Volume<T>::minFilter(const Volume<T>& other) const
{
  Volume<T> out(getWidth(), getHeight(), getDepth());

  for (int i = 0; i < getDepth(); i++)
    for (int j = 0; j < getHeight(); j++)
      for (int k = 0; k < getWidth(); k++)
        out(k, j, i) = std::min(operator()(k, j, i), other(k, j, i));

  return out;
}

template<class T>
Volume<T> Volume<T>::maxFilter(const Volume<T>& other) const
{
  Volume<T> out(getWidth(), getHeight(), getDepth());

  for (int i = 0; i < getDepth(); i++)
    for (int j = 0; j < getHeight(); j++)
      for (int k = 0; k < getWidth(); k++)
        out(k, j, i) = std::max(operator()(k, j, i), other(k, j, i));

  return out;
}


template class Volume<float>;
template class Volume<byte>;
template class Volume<char>;
template class Volume<int>;
template class SimpleVolume<coord3s>;
