#include "cuts/CutSet.h"

#include "SkeletonModel.h"
#include "PointSearch.h"

#include <cstdio>
#include <utility>

namespace cuts
{

CutSet::CutSet(FeaturePoints &fp, SkeletonModel &skeletonModel)
:
  r_fp(fp),
  r_skeletonModel(skeletonModel)
{
}

CutSet::CutSet(const CutSet &other)
:
  r_fp(other.r_fp),
  r_skeletonModel(other.r_skeletonModel)
{
  for (auto const &idx_cut_pair: other.d_cuts)
  this->d_cuts.insert(
        std::make_pair(
          idx_cut_pair.first,
          std::make_unique<Cut>(*idx_cut_pair.second)));
}

CutSet::CutSet(CutSet &&other)
:
  r_fp(other.r_fp),
  r_skeletonModel(other.r_skeletonModel),
  d_cuts(std::move(other.d_cuts))
{
}

void CutSet::compute(CutCreator &cutCreator)
{
  //for every point of the full skeleton, run the cutcreator
  auto &r_filteredSkelPoints(r_skeletonModel.GetSkelPoints());
  std::cout << "start computing " << r_filteredSkelPoints.size() <<" cuts\n";
  auto startTime = std::chrono::duration_cast<std::chrono::duration<double, std::chrono::seconds::period>>
      (std::chrono::steady_clock::now().time_since_epoch()).count();
  d_cuts.clear();
  #pragma omp parallel for schedule(guided)
  for (int_fast64_t filteredSkelPoint_idx = 0;
       filteredSkelPoint_idx < r_filteredSkelPoints.size();
       ++filteredSkelPoint_idx)
  {
    //get associated Point3D
    Point3d &skelPoint(r_filteredSkelPoints.at(filteredSkelPoint_idx));
    //convert to index in the full skeleton
    size_t skelpoint_idx = r_fp.skelSearch->closest(skelPoint);
//    std::printf("compute %lu of %lu : %lu\n",
//                filteredSkelPoint_idx+1,
//                r_filteredSkelPoints.size(),
//                skelpoint_idx);
    auto up_cut = cutCreator(skelpoint_idx);
    if (up_cut->up_path)
      d_cuts[skelpoint_idx] = std::move(up_cut);
  }
  auto execTime = std::chrono::duration_cast<std::chrono::duration<double, std::chrono::seconds::period>>(std::chrono::steady_clock::now().time_since_epoch()).count() - startTime;
  std::printf("found %lu paths for %lu skeleton points in %.2f seconds\n",
              d_cuts.size(), r_filteredSkelPoints.size(), execTime);
}

}
