#include "cuts/ShortestPathCutCreator.h"
#include "cuts/Cut.h"
#include "collapseSkel3d.h"
#include "PointSearch.h"

#include <iostream>

namespace cuts
{

ShortestPathCutCreator::ShortestPathCutCreator(
  const FeaturePoints &fp,
  const surface::Graph &sufaceGraph,
  const collapseSkel3d &skel)
:
  CutCreator(fp, sufaceGraph, skel),
  d_aStar(sufaceGraph)
{

}

std::unique_ptr<Cut> ShortestPathCutCreator::
operator()(size_t skelpoint_idx)
{
  auto cut = std::unique_ptr<Cut>(new Cut);

  //Step size along a voxel ray (in voxel sizes)
  const float step_size = 0.5;

  //1. Construct shortest geo-path between feature points
  const FeaturePoints::FeatureTriple &feature_triple = r_fp.featurePoints[skelpoint_idx];

  // cut->up_path.reset(new surface::Graph::Path);
  // r_surfaceGraph.shortestPath(feature_pair.first, feature_pair.second, *cut->up_path);
  cut->up_path = d_aStar.shortestPath(feature_triple.first, feature_triple.second);
  //2. Find midpoint of this geo-path
  surface::Graph::Node* geo_midpoint = cut->up_path->midPoint();

  //3. Find coordinates of skeleton point for which we work
  Point3d s = r_fp.fullSkeleton[skelpoint_idx];

  //4. Find position of geo-midpoint on the input surface
  const Point3d &m = r_surfaceGraph.nodePosition(geo_midpoint->id);

  //5. Find direction vector from geo-midpoint to its skeleton point
  Point3d dir = s - m;
  dir.normalize();
  dir *= step_size;

  //6. Walk, along 'dir', from the skeleton point until we exit
  //   the shape. For this, we use a simple/probably not optimal
  //   algorithm that walks along the ray with steps smaller than one voxel
  //   (so we're sure we visit each voxel along the ray).

  Volume<float> const &r_edt(r_skel.getEDT());
  int max_steps = (int) ceilf(r_edt.diagonal() / step_size);
  for (int i = 0; i < max_steps; ++i)             
  {
    float dt = r_edt((short) s.x, (short) s.y, (short) s.z);
    if (dt < 1.0f) break;
    s += dir;
  }

  //7. Find boundary-point closest to the exit point. This just
  //   to make 100% sure that the exit point _is_ a boundary point.
  int boundary_id = r_fp.pointSearch->closest(s);

#ifndef NDEBUG
  //8. Find node (on boundary graph) corresponding to exit point
  std::unordered_map<int, surface::Graph::Node*>::const_iterator it = r_surfaceGraph.nid2nodes.find(boundary_id);

  // Normally, this node should exist. If not, we have an error
  if (it == r_surfaceGraph.nid2nodes.end())
  {
    // in the Graph class.
    std::cout << "Error: Cut::construct: exit point not on boundary" << std::endl;
    throw;
  }
#endif

  //10. Try next to create a closed cut by connecting the start
  //    and end points of geo_path with the exit point, using geodesics
  surface::Graph::Node* path_start = (*cut->up_path)[0];
  surface::Graph::Node* path_end   = (*cut->up_path)[cut->up_path->size() - 1];

  //    The first geo-piece goes from the path_end to the exit point...
  //    The second geo-piece goes from exit point to path_start.
  //    Take care remove duplicate start and endpoints of the three paths.
  // std::unique_ptr<surface::Graph::Path> path1(new surface::Graph::Path),
  //                                       path2(new surface::Graph::Path);
  // r_surfaceGraph.shortestPath(path_end->id, boundary_id, *path1);
  // r_surfaceGraph.shortestPath(boundary_id, path_start->id, *path2);
  auto path1 = d_aStar.shortestPath(path_end->id, boundary_id);
  auto path2 = d_aStar.shortestPath(boundary_id, path_start->id);

  // std::cout <<  "path start id: " << path_start->id
  //           << " path end   id: " << path_end->id
  //           << " boundary   id: " << boundary_id << "\n";

  // std::cout <<  "p0: " << cut->up_path->size()
  //           << ", p1: " << path1->size()
  //           << ", p2: " << path2->size() << "\n";

  if (path1->size())
  {
    cut->up_path->insert(
        cut->up_path->end(), ++path1->begin(), path1->end());
  }
  if (path2->size() > 1)
  {
    path2->resize(path2->size() - 1);
    cut->up_path->insert(
        cut->up_path->end(), ++path2->begin(), path2->end());
  }

  r_surfaceGraph.updateLength(*cut->up_path);
  return std::move(cut);
}

}