#include "cuts/TransformLaplacianSmooth.h"
#include "surface/AStar.h"

#include <PointSearch.h>

#include <vector>
#include <algorithm>
#include <iterator>
#include <unordered_map>

namespace cuts
{

TransformLaplacianSmooth::TransformLaplacianSmooth(
    const FeaturePoints &fp,
    const surface::Graph &sufaceGraph,
    const size_t NI,
    const size_t NS,
    const float weight,
    const size_t maxPass)
:
  r_fp(fp),
  r_sufaceGraph(sufaceGraph),
  d_NI(NI),
  d_NS(NS),
  d_maxPass(maxPass),
  d_weight(weight)
{

}

Cut TransformLaplacianSmooth::operator()(Cut cut) const
{
  //do nothing if there's no path
  if (!cut.up_path) return cut;
  surface::Graph::Path &r_path = *cut.up_path;
  std::vector<Point3d> P;
  P.reserve(r_path.size());
  //2. Smooth cut until no more voxels can be removed from it:
  for (size_t pass = 0; pass != d_maxPass; ++pass)
  {
    //count how many voxels in size the path has reduced
    size_t old_size = r_path.size();
    //2.1. Do a number NI of shrinking+backprojection iterations:
    for (size_t iter = 0; iter != d_NI; ++iter)
    {
      //1. Copy cut-positions to a floating-point vector, so we
      //   can next do the smoothing on these positions
      P.clear();
      for (auto node_ptr : r_path)
         P.emplace_back(r_sufaceGraph.nodePosition(node_ptr));
      const size_t old_size = r_path.size();
      //2.2. Do a number NS of smoothing iterations:
      //shift by old_size to make use of modulo operator
      for (size_t iS = old_size; iS != old_size*(d_NS+1); ++iS)
      {
        Point3d L;
        L += P[ (iS-1) % old_size ] - P[ iS % old_size ];
        L += P[ (iS+1) % old_size ] - P[ iS % old_size ];
        L /= 2;
        P[ iS % old_size ] += L * d_weight;
      }
      //2.3. Reproject smoothed curve to voxel-surface, taking care
      //     to avoid duplicating in sequence voxels or inserting holes
      r_path.clear();
      //2.7. Add new back-projected voxel to path
      std::transform(P.begin(), P.end(), std::back_inserter(r_path),
            [&](Point3d p)->surface::Graph::Node*
            {
              //2.4. Find closest boundary-voxel to the smoothed position P[i]
              auto node_id = r_fp.pointSearch->closest(p);
              //2.5. Find graph node corresponding to above boundary voxel
              auto node_p = r_sufaceGraph.node(node_id);
              return node_p;
            }
          );
      //2.6. Skip duplicate neighbors in the path
      auto r_path_unique_end = std::unique(r_path.begin(), r_path.end());
      //fix first/last element
      if ((r_path_unique_end - r_path.begin() > 2) &&
          (*(r_path_unique_end - 1) == r_path[0]))
        --r_path_unique_end;
      r_path.erase(r_path_unique_end, r_path.end());
    }
    //No points removed during the last NI iterations? Then we're done
    if (r_path.size() == old_size) break;
  }
  //path stuff together if gaps have appeared:
//  std::cout << "checking continuuity for path "
//            << cut.d_skelpoint_idx << ", "
//            << "with length "
//            << r_path.size() << "\n";
  bool path_changed;
  if (r_path.size() > 1)
  do
  {
    path_changed = false;
    for (size_t i = 0, sz = r_path.size(); i != sz; ++i)
    {
      auto crt = r_path[i];
      auto nxt = r_path[(i + 1) % sz];

      auto it = crt->neighbors.find(nxt);
      if (it == crt->neighbors.end())
      {
        auto astar = surface::AStar(r_sufaceGraph);
        auto shortestpath = astar.shortestPath(crt, nxt);
//        std::cout << "inserting stuff:\n"
//                  << r_path.size() << ",\n"
//                  << &*(r_path.end()) << ",\n"
//                  << &*(r_path.begin() + i + 1) << ",\n"
//                  << shortestpath->size() << ",\n"
//                  << &*(shortestpath->begin()+1) << ",\n"
//                  << &*(shortestpath->end()-1)
//                  << "\n";
        r_path.insert(r_path.begin() + i + 1,
                      shortestpath->begin()+1,
                      shortestpath->end()-1);
        path_changed = true;
        break;
      }
    }
  } while ( path_changed );
  r_sufaceGraph.updateLength(r_path);
  return cut;
}

}
