#include "ima/ImaSkel.h"

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include <time.h>
#include <math.h>
#include <algorithm>

namespace IMA
{
  const int GAMMA = 2;
  const int FOREGROUND = 1;
  const int BACKGROUND = 0;
  const int SKEL = 255;

#define sqr(x) ((x)*(x))
#define sqrlength(i, j, k) ((i)*(i)+(j)*(j)+(k)*(k))

  static int			gamma_val = GAMMA;



  /*************** SKELETONIZATION *****************/
  void doubleScan(int ft[], int lim, int ftline[],
    int dtline[], int ss[], int tt[])
  {
    int q = 0, j, w;
    ss[0] = tt[0] = 0;
    for (j = 1; j < lim; j++) {
      while (q >= 0 &&
        (j - ss[q])*(j + ss[q] - 2 * tt[q]) < dtline[ss[q]] - dtline[j]) {
        q--;
      }
      if (q < 0) {
        q = 0;
        ss[0] = j;
      }
      else {
        w = 1 +
          ((j + ss[q])*(j - ss[q]) + dtline[j] - dtline[ss[q]]) / (2 * (j - ss[q]));
        if (w < lim) {
          q++;
          ss[q] = j;
          tt[q] = w;
        }
      }
    }
    for (j = lim - 1; j >= 0; j--) {
      ft[j] = ftline[ss[q]] * lim + ss[q]; /* encoding */
      if (j == tt[q]) q--;
    }
  }

  IntVolume *featTrans(ByteVolume* boundary)
  {
    /* interpretation (x,y,z) in boundary IFF boundary[x][y][z] == 0
    first phase: construction of feature transform in the x-direction
    Vectors (x, y, z) are encoded as integers by
    encode(x, y, z) = z + zdim * (y + ydim * x) */
    int x, y, z, xy, right, left;
    int xdim = boundary->xdim;
    int ydim = boundary->ydim;
    int zdim = boundary->zdim;
    IntVolume *ftx, *ftxy, *ftxyz;
    int INFTY = 1 + (int)sqrt(sqr(xdim) + sqr(ydim) + sqr(zdim));
    int LIM = std::max(std::max(xdim, ydim), zdim);
    int *ftline, *dtline, *ss, *tt;

    ftline = ivector(0, LIM - 1);
    dtline = ivector(0, LIM - 1);
    ss = ivector(0, LIM - 1);
    tt = ivector(0, LIM - 1);

    /* The pure algorithm require a nonempty boundary; the encoding
    * requires all coordinates nonnegative. We therefore extend the
    * boundary with points with the plane x = xdim - 1 + INFTY.
    * Conditions: (xdim-1)^2 + ... + (zdim-1)^2 < INFTY^2
    * and (xdim-1+INFTY) * ydim * zdim <= Integer.MAX_VALUE */
    ftx = IntVolume_New(ydim, zdim, xdim);
    for (y = 0; y < ydim; y++) {
      for (z = 0; z < zdim; z++) {
        dtline[xdim - 1] = right =
          (boundary->data[xdim - 1][y][z] == 0 ? 0 : INFTY);
        /* INFTY simulates a boundary outside the image */
        for (x = xdim - 2; x >= 0; x--) {
          dtline[x] = right =
            (boundary->data[x][y][z] == 0 ? 0 : right + 1);
        }
        ftx->data[y][z][0] = left = dtline[0];
        for (x = 1; x < xdim; x++) {
          right = dtline[x];
          ftx->data[y][z][x] = left =
            (x - left <= right ? left : x + right);
        }
      }
    }

    /* second phase: construction of feature transform in the xy-direction
    * based on feature transform in the x-direction */
    ftxy = IntVolume_New(zdim, xdim, ydim);
    for (z = 0; z < zdim; z++) {
      for (x = 0; x < xdim; x++) {
        for (y = 0; y < ydim; y++) {
          ftline[y] = xy = ftx->data[y][z][x];
          dtline[y] = sqr(xy - x);
        }
        doubleScan(ftxy->data[z][x], ydim, ftline,
          dtline, ss, tt);
      }
    }
    IntVolume_Delete(ftx);

    /* third phase: construction of feature transform in the xyz-direction
    * based on feature transform in the xy-direction */
    ftxyz = IntVolume_New(xdim, ydim, zdim);
    for (x = 0; x < xdim; x++) {
      for (y = 0; y < ydim; y++) {
        for (z = 0; z < zdim; z++) {
          ftline[z] = xy = ftxy->data[z][x][y];
          dtline[z] = sqr(xy / ydim - x) + sqr(xy % ydim - y);
        }
        doubleScan(ftxyz->data[x][y], zdim, ftline,
          dtline, ss, tt);
      }
    }
    IntVolume_Delete(ftxy);
    free_ivector(ftline, 0, LIM - 1);
    free_ivector(dtline, 0, LIM - 1);
    free_ivector(ss, 0, LIM - 1);
    free_ivector(tt, 0, LIM - 1);

    return ftxyz;
  }

  inline double euclidnorm(int x, int y, int z)
  {
    return sqrt(x * x + y * y + z*z);
  }


  void compare(IMA_BYTE *xskel, IMA_BYTE *pskel,
    int x, int y, int z, int p, int q, int r,
    int xf, int pf, int ydim, int zdim)
  {
    /* compute feature transform vectors vf = (xf, yf, zf), uf = (pf, qf, rf) */
    int yf, zf, qf, rf, dif, crit;
    zf = xf % zdim; xf = xf / zdim;
    yf = xf % ydim; xf = xf / ydim;
    rf = pf % zdim; pf = pf / zdim;
    qf = pf % ydim; pf = pf / ydim;
    dif = sqr(xf - pf) + sqr(yf - qf) + sqr(zf - rf);

    if (dif > 1 && dif >
      (gamma_val > 0 ? gamma_val * gamma_val /* Constant pruning */
        : gamma_val < 0 ?         /* Linear pruning */
        (sqr(x - xf + p - pf) + sqr(y - yf + q - qf) + sqr(z - zf + r - rf)) * gamma_val * gamma_val
        : /* Square root pruning */
        euclidnorm(x - xf + p - pf, y - yf + q - qf, z - zf + r - rf)
        + 2 * ((x - p)*(xf - pf) + (y - q)*(yf - qf) + (z - r)*(zf - rf)) + 1.5)
      )
      /*	 C_d=1.5 for d=3; C_d=1 for d=2;  */
    {
      /* the point xskel is in the skeleton
      iff ||m-uf|| <= ||m-vf|| iff crit >= 0, where */
      crit = (xf - pf)*(xf + pf - x - p) + (yf - qf)*(yf + qf - y - q) + (zf - rf)*(zf + rf - z - r);
      if (crit >= 0) *xskel = SKEL;
      if (crit <= 0) *pskel = SKEL;
    }
  }


  void skeleton(IntVolume *ft, ByteVolume *skel)
  {
    int xdim = ft->xdim;
    int ydim = ft->ydim;
    int zdim = ft->zdim;
    int x, y, z;

    for (x = 1; x < xdim; x++)
      for (y = 0; y < ydim; y++)
        for (z = 0; z < zdim; z++)
          if (skel->data[x][y][z] == FOREGROUND || skel->data[x - 1][y][z] == FOREGROUND)
            compare(&skel->data[x][y][z], &skel->data[x - 1][y][z],
              x, y, z, x - 1, y, z,
              ft->data[x][y][z], ft->data[x - 1][y][z], ydim, zdim);

    for (x = 0; x < xdim; x++) {
      for (y = 1; y < ydim; y++)
        for (z = 0; z < zdim; z++)
          if (skel->data[x][y][z] == FOREGROUND || skel->data[x][y - 1][z] == FOREGROUND)
            compare(&skel->data[x][y][z], &skel->data[x][y - 1][z],
              x, y, z, x, y - 1, z,
              ft->data[x][y][z], ft->data[x][y - 1][z], ydim, zdim);
      for (y = 0; y < ydim; y++)
        for (z = 1; z < zdim; z++)
          if (skel->data[x][y][z] == FOREGROUND || skel->data[x][y][z - 1] == FOREGROUND)
            compare(&skel->data[x][y][z], &skel->data[x][y][z - 1],
              x, y, z, x, y, z - 1,
              ft->data[x][y][z], ft->data[x][y][z - 1], ydim, zdim);
    }
  }





}

template <class T>
Volume<T> ComputeImaSkeleton(Volume<byte>& input, T dummy)
{

  int xdim = input.getWidth();
  int ydim = input.getHeight();
  int zdim = input.getDepth();


  Volume<T> output(xdim, ydim, zdim);
  IMA::ByteVolume* indata = IMA::ByteVolume_New(xdim, ydim, zdim);

  for (int x = 0; x < xdim; x++)
    for (int y = 0; y < ydim; y++)
      for (int z = 0; z < zdim; z++)
        indata->data[x][y][z] = input(x, y, z);

  byte min, max;
  ByteVolume_MinMax(indata, &min, &max);
  if (max > IMA::FOREGROUND)
  {
    //  Clamp all input values to FOREGROUND
    ByteVolume_Clamp(indata, IMA::FOREGROUND);
  }

  // Construct the feature transform 
  IMA::IntVolume* ft = IMA::featTrans(indata);

  // Construct the skeleton from the feature transform into 'indata'
  IMA::skeleton(ft, indata);

  // Put skeleton points to FOREGROUND, and non-skeleton points to BACKGROUND
  for (int x = 0; x<xdim; x++)
    for (int y = 0; y<ydim; y++)
      for (int z = 0; z<zdim; z++)
      {
        if (indata->data[x][y][z] == IMA::FOREGROUND) 
          output(x,y,z) = IMA::BACKGROUND;
        else if (indata->data[x][y][z] == IMA::SKEL) 
          output(x, y, z) = IMA::FOREGROUND;
      }

  
  IMA::ByteVolume_Delete(indata);
  IMA::IntVolume_Delete(ft);

  return output;
}

Volume<byte> ComputeImaSkeletonByte(Volume<byte>& input)
{
  return ComputeImaSkeleton(input, (byte)0);
}

Volume<float> ComputeImaSkeletonFloat(Volume<byte>& input)
{
  return ComputeImaSkeleton(input, (float)0);
}
