#include "surface/AStar.h"

#include "Point3d.h"

#include <algorithm>
#include <map>
#include <unordered_map>
#include <unordered_set>

#include <cstdio>

namespace surface
{

AStar::AStar(const Graph &graph)
:
  r_graph(graph)
{
}


static float heuristic_dist_estimate(const Point3d &current_pos, const Point3d &target_pos)
{
  //ordinary eulcidean distance is a consistent heuristic
  return current_pos.dist(target_pos);
}


std::unique_ptr<Graph::Path> AStar::shortestPath(
    int origin_boundary_idx,
    int target_boundary_idx) const
{
  Graph::Node *origin_node = r_graph.nid2nodes.at(origin_boundary_idx);
  Graph::Node *target_node = r_graph.nid2nodes.at(target_boundary_idx);
  return shortestPath(origin_node, target_node);
}

std::unique_ptr<Graph::Path> AStar::shortestPath(
    Graph::Node *origin_node,
    Graph::Node *target_node) const
{
  //following http://en.wikipedia.org/wiki/A*_search_algorithm

  //the set of already evaluated nodes
  std::unordered_set<Graph::Node*> closed_set;
  //The set of tentative nodes to be evaluated, initially containing the start node
  std::multimap<double, Graph::Node*> open_set;

  struct Node_data
  {
    //store the optimal previous node needed to reach the node
    Graph::Node* came_from;
    double dist_to_origin;
    double heuristic_dist;
    //obviously invalidated once removed from the open set
    std::multimap<double, Graph::Node*>::iterator open_set_it;
  };
  std::unordered_map<Graph::Node*, Node_data> node_data_lookup;


  //cache target node and pos
  const Point3d &target_pos = (*r_graph.boundary)[target_node->id];

  //queue the origin to be processed
  {
    Node_data *origin_data = &node_data_lookup[origin_node];
    origin_data->came_from = nullptr;
    origin_data->dist_to_origin = 0;
    const Point3d &origin_pos = (*r_graph.boundary)[origin_node->id];
    origin_data->heuristic_dist = heuristic_dist_estimate(origin_pos, target_pos);
    origin_data->open_set_it = 
        open_set.insert(std::make_pair(origin_data->heuristic_dist, origin_node));
  }

  bool succes = false;
  while (! open_set.empty())
  {
    //pop from queue
    Graph::Node* current_node = open_set.begin()->second;
    if (current_node == target_node)
    {
      succes = true;
      break;
    }
    Node_data *current_data = &node_data_lookup[current_node];
    open_set.erase(open_set.begin());
    closed_set.insert(current_node);
    // std::printf("node: %5d, neighbors: %2lu\n", current_node->id, current_node->neighbors.size());
    //iterate over neighbors
    for (auto neighbor_it = current_node->neighbors.begin();
        neighbor_it != current_node->neighbors.end();
        ++neighbor_it)
    {
      Graph::Node* neighbor_node = neighbor_it->first;
      float        neighbor_dist = neighbor_it->second;
      // std::printf("neighbor: %5d\n", neighbor_it->first->id);
      //case: already handled
      if (closed_set.count(neighbor_node))
        continue;

      auto neighbor_data_it = node_data_lookup.find(neighbor_node);
      //case: already in queue
      if (neighbor_data_it != node_data_lookup.end())
      {
        Node_data *neighbor_data = &(neighbor_data_it->second);
        //should we update the node?
        double update_dist = current_data->dist_to_origin + neighbor_dist;
        if (neighbor_data->dist_to_origin > update_dist)
        {
          neighbor_data->dist_to_origin = update_dist;
          neighbor_data->came_from = current_node;
          //remove and reinsert from queue
          open_set.erase(neighbor_data->open_set_it);
          neighbor_data->open_set_it =
              open_set.insert(std::make_pair(neighbor_data->heuristic_dist + update_dist, neighbor_node));
        }
      //case: not yet in queue, make new one
      } else {
        Node_data *neighbor_data = &node_data_lookup[neighbor_node];
        neighbor_data->came_from = current_node;
        neighbor_data->dist_to_origin = current_data->dist_to_origin + neighbor_dist;
        const Point3d &neighbor_pos = (*r_graph.boundary)[neighbor_node->id];
        neighbor_data->heuristic_dist = heuristic_dist_estimate(neighbor_pos, target_pos);
        neighbor_data->open_set_it = 
            open_set.insert(
                std::make_pair(
                    neighbor_data->heuristic_dist + neighbor_data->dist_to_origin,
                    neighbor_node));
      }
    }
  }
  if (! succes)
  {
    //throw
    std::printf("error: no path\n");
    std::unique_ptr<Graph::Path> up_path;
    return up_path;
  }

  //reconstruct path
  std::unique_ptr<Graph::Path> up_path(new Graph::Path);
  {
    Graph::Node *current_node = target_node;
    Node_data *current_data = &node_data_lookup[current_node];
    up_path->push_back(current_node);
    while (current_data->came_from)
    {
      current_node = current_data->came_from;
      current_data = &node_data_lookup[current_node];
      up_path->push_back(current_node);
    }
  }
  //reverse the path to go from origin to target
  std::reverse(up_path->begin(), up_path->end());
  up_path->length = node_data_lookup[target_node].dist_to_origin;
  // std::printf("origin: %5d, target: %5d, length %f\n", origin_boundary_idx, target_boundary_idx, up_path->length);
  // for (auto it = up_path->begin(); it != up_path->end(); ++it)
    // std::printf("%5d, ", (*it)->id);
  // std::printf("\n");
  return std::move(up_path);
}


}
