#include "surface/PlaneTrace.h"

#include "surface/AStar.h"

#include <Eigen/Core>
#include <cmath>

#include <unordered_set>
#include <queue>
#include <cassert>

namespace surface
{

PlaneTrace::PlaneTrace(
  const surface::Graph  &graph,
  const std::vector<Point3d> &r_normals)
  :
  r_graph(graph),
  r_normals(r_normals)
{
}

namespace {
Eigen::Vector3d node2pos(const std::vector<Point3d> &boundary, const Graph::Node *node)
{
  Point3d p3d = boundary.at(node->id);
  return Eigen::Vector3d( p3d.x, p3d.y, p3d.z);
}

template <class settype>
static std::unique_ptr<Graph> subGraphFromSet(const surface::Graph &base_graph, settype &set)
{
  std::unique_ptr<Graph> up_graph(new Graph);
  //copy whatever we can
  up_graph->boundary = base_graph.boundary;

  //add nodes to the new graph
  for (auto base_node : set)
  {
    size_t id = base_node->id;
    size_t key = base_node->key;
    Graph::Node* p_new_node = new Graph::Node(id, key);
    //up_graph now owns the new node:
    up_graph->nodes.insert(std::make_pair(key, p_new_node));
    up_graph->nid2nodes.insert(std::make_pair(id, p_new_node));
  }
  //add the right edges
  for (auto base_node : set)
  {
    //find corresponding new node
    Graph::Node* p_new_node = up_graph->nid2nodes.at(base_node->id);
    for (auto base_neighbor_pair : base_node->neighbors)
    {
      //is this link still valid?
      if (set.count(base_neighbor_pair.first))
      {
        //find corresponding new node
        Graph::Node* p_new_neighbor_node = up_graph->nid2nodes.at(base_neighbor_pair.first->id);
        //the new edge has the same weight
        p_new_node->addNeighbor(p_new_neighbor_node, base_neighbor_pair.second);
      }
    }
  }
  return std::move(up_graph);
}



struct MakeCorridorGraph_ret
{
#if _MSC_VER && !__INTEL_COMPILER
  // Explicit constructors, required to circumvent a visual compiler bug.
  MakeCorridorGraph_ret(
    std::unique_ptr<Graph>&& up_corridor_graph, 
    Graph::Node *p_origin_node,
    Graph::Node *p_target_node) :
    up_corridor_graph(std::move(up_corridor_graph)),
    p_origin_node(p_origin_node),
    p_target_node(p_target_node)
  {
  }

  MakeCorridorGraph_ret(MakeCorridorGraph_ret&& r) :
    up_corridor_graph(std::move(r.up_corridor_graph)),
    p_origin_node(r.p_origin_node),
    p_target_node(r.p_target_node)
  {
  }
#endif

  std::unique_ptr<Graph> up_corridor_graph;
  Graph::Node *p_origin_node;
  Graph::Node *p_target_node;
};

MakeCorridorGraph_ret makeCorridorGraph(
  std::unique_ptr<Graph> graph,
  Graph::Node *seed_node,
  PlaneTrace::Plane &corridor_cut_plane)
{
  //nodes that have been added to the queue, together with a bool that's true
  //if the node's distance from the plane is nonnegative
  //with respect to the plane
  std::unordered_map<surface::Graph::Node*, bool> nodes;
  //nodes already discovered, but not processed:
  std::queue<surface::Graph::Node*> node_queue;
  //node discovery
  //queue the seed
  {
    assert(graph->nid2nodes.at(seed_node->id) == seed_node);
    node_queue.push(seed_node);
    auto eigen_seed_pos = node2pos(*(graph->boundary), seed_node);
    bool current_plane_dist_nonnegative = corridor_cut_plane.signedDistance(eigen_seed_pos) >= 0;
    nodes.emplace(seed_node, current_plane_dist_nonnegative);
  }

  //explore
  while (!node_queue.empty())
  {
    //pop front
    surface::Graph::Node* current_node = node_queue.front();
    node_queue.pop();
    //process neighbors of current node
    for (auto const& neighbor_pair : current_node->neighbors)
    {
      auto &neighbor_node = neighbor_pair.first;
      auto eigen_neighbor_pos = node2pos(*(graph->boundary), neighbor_node);
      double neighbor_signed_plane_dist = corridor_cut_plane.signedDistance(eigen_neighbor_pos);
      bool neighbor_plane_dist_nonnegative = neighbor_signed_plane_dist >= 0;
      double neighbor_abs_plane_dist = std::abs(neighbor_signed_plane_dist);
      if (neighbor_abs_plane_dist <= std::sqrt(3.0))
      {
        //try to add to queue
        //pair with iterator to el in set and bool that's true if a new element was inserted
        auto node_set_ret = nodes.emplace(neighbor_node, neighbor_plane_dist_nonnegative);
        if (node_set_ret.second)
        {
          node_queue.push(neighbor_node);
        }
      }
    }
  }

  //remove all links to the 'other' side, unless it's the seed node
  for (auto const& current_pair: nodes)
  {
    auto &current_node = current_pair.first;
    if (current_node != seed_node)
    {
      bool  current_plane_dist_nonnegative = current_pair.second;
      for (auto neighbor_pair_it = current_node->neighbors.begin();
           neighbor_pair_it != current_node->neighbors.end();)
      {
        auto &neighbor_pair = *neighbor_pair_it;
        Graph::Node* neighbor_node = neighbor_pair.first;
        if (nodes.find(neighbor_node) == nodes.end())
        {
          ++neighbor_pair_it;
          continue;
        }
        bool neighbor_plane_dist_nonnegative = nodes.at(neighbor_node);
        //check if it's on the other side
        if (neighbor_plane_dist_nonnegative != current_plane_dist_nonnegative)
        {
          //remove edge
          neighbor_pair_it = current_node->neighbors.erase(neighbor_pair_it);
        } else {
          ++neighbor_pair_it;
        }
      }
    }
  }

  //replace the seed node for one half of the plane with a copy
  Graph::Node* clone_node;
  {
    //clone the seed node
    //id is important but choose unlikely hashkey for storage
    //this is probably bad style, but the graph was never designed
    //to have 2 nodes share a position...
    //clone will not be added to nid2nodes!
    clone_node = new Graph::Node(seed_node->id, -1);
    graph->nodes.insert(std::make_pair(-1, clone_node));
    auto old_neighbors = seed_node->neighbors;
    //check neighbours of seed
    for (auto const& neighbor_pair : old_neighbors)
    {
      auto neighbor_node = neighbor_pair.first;
      if (neighbor_node->neighbors.find(seed_node) == neighbor_node->neighbors.end())
      {
        //the node is on the other side
        auto neighbor_dist = old_neighbors.at(neighbor_node);
        seed_node->neighbors.erase(neighbor_node);
        clone_node->addNeighbor(neighbor_node, neighbor_dist);
      }
    }

  }
#if _MSC_VER && !__INTEL_COMPILER
  return MakeCorridorGraph_ret(std::move(graph), seed_node, clone_node);
#else
  return {.up_corridor_graph = std::move(graph), .p_origin_node=seed_node, .p_target_node = clone_node};
#endif
}

std::unique_ptr<Graph::Path> subgraph_path_to_graph_path(
    std::unique_ptr<Graph::Path> up_path,
    const Graph &graph)
{
  for (auto &node : *up_path)
  {
    node = graph.nid2nodes.at(node->id);
  }
  //remove duplicate last node
  up_path->pop_back();
  return up_path;
}
}

//starting in seed_node, create a new graph that is a subgraph of graph,
//with all the nodes that are within distance distance_thr from the plane,
//and reachable from seed_node with a path that always stays within thr from plane.
std::unique_ptr<Graph> makeCutBand(
  const surface::Graph &input_graph,
  double distance_thr,
  const surface::Graph::Node *seed_node,
  const PlaneTrace::Plane &plane)
{
  //build a set of all the nodes that should be in the subgraph
  //nodes to be added to subgraph:
  std::unordered_set<const surface::Graph::Node*> nodes;
  //nodes already discovered, but not processed:
  std::queue<const surface::Graph::Node*> node_queue;
  //node discovery
  //queue the seed
  node_queue.push(seed_node);
  nodes.insert(seed_node);
  while (!node_queue.empty())
  {
    //pop front
    const surface::Graph::Node* current_node = node_queue.front();
    node_queue.pop();
    //add neighbors of current node to queue
    for (auto neighbor_pair : current_node->neighbors)
    {
      Graph::Node* neighbor_node = neighbor_pair.first;
      Eigen::Vector3d eigen_neighbor_pos = node2pos(*input_graph.boundary, neighbor_node);
      if (plane.absDistance(eigen_neighbor_pos) < distance_thr)
      {
        //pair with iterator to el in set and bool that's true if a new element was inserted
        auto node_set_ret = nodes.insert(neighbor_node);
        if (node_set_ret.second)
        {
          node_queue.push(neighbor_node);
        }
      }
    }
  }

  //construct and return new graph
  return subGraphFromSet(input_graph, nodes);
}

std::unique_ptr<Graph::Path> PlaneTrace::trace(
  int seed_boundary_index,
  const Plane &plane,
  double bandDeviation)
{
  Graph::Node *seed_node = r_graph.nid2nodes.at(seed_boundary_index);
  std::unique_ptr<Graph> up_cutBand(makeCutBand(
    r_graph,
    bandDeviation,
    seed_node,
    plane));

  //the corridor cut plane should be perpendicular to both the surface
  //as well as to the plane which cuts the shape
  Graph::Node *new_seed_node = up_cutBand->nid2nodes.at(seed_boundary_index);
  Eigen::Vector3d seed_normal   = node2pos(r_normals, new_seed_node);
  Eigen::Vector3d seed_position = node2pos(*r_graph.boundary, new_seed_node);
  Eigen::Vector3d corridor_cut_plane_normal = seed_normal.cross(plane.normal());
  corridor_cut_plane_normal.normalize();
  Plane corridor_cut_plane(
    corridor_cut_plane_normal,
    seed_position);
  auto makeCorridorGraph_ret = makeCorridorGraph(
    std::move(up_cutBand),
    new_seed_node,
    corridor_cut_plane);
  //compute shortest path on new graph
  auto up_path = AStar(*(makeCorridorGraph_ret.up_corridor_graph)).shortestPath(
    makeCorridorGraph_ret.p_origin_node,
    makeCorridorGraph_ret.p_target_node);
  if (!up_path) return up_path;

  //translate new path back to old graph
  up_path = subgraph_path_to_graph_path(std::move(up_path), r_graph);
//  std::cout << "path lenght: " << up_path->length << "\n";
  return up_path;
}

}
