#pragma once

#include <iostream>
#include <fstream>
#include <sstream>

#include "SkeletonModel.h"
#include "surface/Graph.h"
#include "NoiseCreator.h"
#include "SaliencyMetricEngine.h"
#include "VolumeHelpers.h"
#include "SkeletonReconstructor.h"

inline std::vector<float> CreateImportanceIndex(Volume<float> &imp, int size)
{
  std::vector<float> allImportancePoints;
  std::vector<float> sortedImportancePoints;


  std::ofstream logfile("histogram_importance.csv", std::ios::out);

  imp.toVector(allImportancePoints);
  auto newEnd = std::remove_if(allImportancePoints.begin(),
    allImportancePoints.end(), [](float a) {return a == 0.0f; });
  allImportancePoints.erase(newEnd, allImportancePoints.end());
  std::sort(allImportancePoints.begin(), allImportancePoints.end());

  for (int i = 0; i < allImportancePoints.size(); i++)
    logfile << allImportancePoints[i] << std::endl;


  sortedImportancePoints.reserve(size + 1);
  float stepSize = 1.0f / size;
  for (float i = 0.0f; i <= 1.01f; i += stepSize)
  {
    int index = static_cast<size_t>(i * (allImportancePoints.size() - 1));
    sortedImportancePoints.push_back(allImportancePoints[index] + std::numeric_limits<float>::epsilon());
  }

  return sortedImportancePoints;
}


inline int volumeDiff(Volume<byte>& a, Volume<byte>& b)
{
  int error = 0;

  // Cast to bool is required as skeleton points may contain scalar data which
  // is still to be reinterpreted as binary.
  for (int z = 0; z < a.getDepth(); z++)
    for (int y = 0; y < a.getHeight(); y++)
      for (int x = 0; x < a.getWidth(); x++)
      {
        bool va = static_cast<bool>(a(x, y, z));
        bool vb = static_cast<bool>(b(x, y, z));

        if (static_cast<bool>(a(x, y, z)) != static_cast<bool>(b(x, y, z)))
          error++;
      }

  return error;
}

inline int volumeLeftDiff(Volume<byte>& a, Volume<byte>& b)
{
  int error = 0;

  // Cast to bool is required as skeleton points may contain scalar data which
  // is still to be reinterpreted as binary.
  for (int z = 0; z < a.getDepth(); z++)
    for (int y = 0; y < a.getHeight(); y++)
      for (int x = 0; x < a.getWidth(); x++)
      {
        bool va = static_cast<bool>(a(x, y, z));
        bool vb = static_cast<bool>(b(x, y, z));

        if (static_cast<bool>(a(x, y, z)) &&  !static_cast<bool>(b(x, y, z)))
          error++;
      }

  return error;
}


template <class T>
int getForegroundPoints(Volume<T>& a)
{
  int count = 0;

  for (int z = 0; z < a.getDepth(); z++)
    for (int y = 0; y < a.getHeight(); y++)
      for (int x = 0; x < a.getWidth(); x++)
        if (a(x, y, z) > 0)
          count++;

  return count;
}

template <class Ta, class Tb>
int getUniqueForegroundPoints(Volume<Ta>& a, Volume<Tb>& b)
{
  int count = 0;

  for (int z = 0; z < a.getDepth(); z++)
    for (int y = 0; y < a.getHeight(); y++)
      for (int x = 0; x < a.getWidth(); x++)
        if (a(x, y, z) > 0 || b(x, y, z) > 0)
          count++;

  return count;
}

inline SaliencyMetricEngine constructSaliencyEngine(collapseSkel3d& skel, FeaturePoints& featurePoints)
{
  SaliencyMetricEngine saliencyMetricEngine(featurePoints, skel.getEDT(), skel.getImportance());

  return saliencyMetricEngine;
}

inline Volume<byte> toByteVolume(Volume<float>& imp, float t = 0)
{
  Volume<byte> ret(imp.getWidth(), imp.getHeight(), imp.getDepth());

  Volume<float> impCopy = imp;
  FilterLargestComponent(impCopy, t);

  for (int x = 0; x < imp.getWidth(); x++)
    for (int y = 0; y < imp.getHeight(); y++)
      for (int z = 0; z < imp.getDepth(); z++)
      {
        float importance = impCopy(x, y, z);
        if (importance > t)
          ret(x, y, z) = 128;
      }

  return ret;
}

inline Volume<float> getFilteredImportance(Volume<float>& imp, float t)
{
  Volume<float> impCopy = imp;
  FilterLargestComponent(impCopy, t);
  return impCopy;
}


inline Volume<byte> Reconstruct(const std::vector<Point3d>& skelPoints, const Volume<float> &edt,
  ReconstructionSmoothingType smoothingType, int radius,
  const Volume<float>* imp = nullptr, float threshold = 0, surface::Graph* graph = nullptr,
  FeaturePoints* featurePoints = nullptr)
{
  SkeletonReconstructor reconstructor(skelPoints, edt);
  return reconstructor.Reconstruct(smoothingType, radius, imp, threshold, graph, featurePoints);
}

void ExportReconstructionGen(ReconstructionSmoothingType method, std::string filename, collapseSkel3d& skel, std::shared_ptr<SkeletonModel> skeletonModel,
  Volume<float>& saliency, float imp,
  int kernel,
  bool constrainSearch = false,
  surface::Graph* graph = nullptr,
  FeaturePoints* featurePoints = nullptr);

void ExportReconstructionGenLog(std::ofstream & logstream, std::string input, Volume<byte>& original, 
  ReconstructionSmoothingType method, std::string filename, collapseSkel3d & skel,
  std::shared_ptr<SkeletonModel> skeletonModel, Volume<float>& saliency, float imp, 
  int kernel,
  bool constrainSearch = false,
  surface::Graph* graph = nullptr,
  FeaturePoints* featurePoints = nullptr);



inline float EstimateFeatureLength(Volume<float>& edt, coord3s p)
{
  const float factor = sqrtf(2) * 2;
  unsigned int x = p.x, y = p.y, z = p.z;

  float dtx = (edt(x + 1, y, z) - edt(x - 1, y, z)) / 2;
  float dty = (edt(x, y + 1, z) - edt(x, y - 1, z)) / 2;
  float dtz = (edt(x, y, z + 1) - edt(x, y, z - 1)) / 2;

  float rate = 1.f - dtx*dtx - dty*dty - dtz * dtz + 0.0001f;
  float rep = sqrt(rate);

  return rate > 0.f ? edt(p) * rep * factor : 0.f;
}

inline Volume<float> ComputeEstimatedEuclidean(Volume<float>& edt, float preThreshold)
{
  Volume<float> eucl(edt.getWidth(), edt.getHeight(), edt.getDepth());

  for (int z = 0; z < edt.getDepth(); z++)
    for (int y = 0; y < edt.getHeight(); y++)
      for (int x = 0; x < edt.getWidth(); x++)
      {
        float dist = edt(x, y, z);
        if (dist <= 1.0f) continue;

        float imp = EstimateFeatureLength(edt, coord3s(x, y, z));
        eucl(x, y, z) = imp > preThreshold ? imp : 0;
      }

  return eucl;
}


inline Volume<byte> GetNoisyModel(Volume<byte> data, float incidence, float radius, float size, float length)
{
  collapseSkel3d skel(false, true);
  int borderVoxels;

  std::cout << "Computing Skeleton...." << std::endl;
  skel.init(data, 0, borderVoxels, false);
  Volume<byte> oldThin = skel.getThinImage();

  std::shared_ptr<SkeletonModel> inputModel(new SkeletonModel(skel, 0.0f));
  surface::Graph graph;
  graph.construct(inputModel->GetThinPoints());

  // add noise
  NoiseCreator noiseCreator(data, graph, inputModel);
  noiseCreator.GeneratePoints(incidence, NoisePickType::Convex, NoiseShape::Ball, radius, size, length);

  // return data with added noise
  return data;
}


inline Volume<float> GetEdt(Volume<byte> data)
{
  collapseSkel3d skel(false, true);
  int borderVoxels;

  std::cout << "Computing Skeleton...." << std::endl;
  skel.init(data, 0, borderVoxels, false);

  return skel.getEDT();
}


inline Volume<byte> ComputeOrgSkeleton(Volume<byte> data, bool addNoise)
{
  collapseSkel3d skel(false, true);
  int borderVoxels;

  std::cout << "Computing Skeleton...." << std::endl;
  skel.init(data, 0, borderVoxels, false);


  // add noise, then reset skeleton
  if (addNoise)
  {
    std::shared_ptr<SkeletonModel> inputModel(new SkeletonModel(skel, 0.0f));

    surface::Graph graph;
    graph.construct(inputModel->GetThinPoints());

    NoiseCreator noiseCreator(data, graph, inputModel);
    noiseCreator.GeneratePoints(0.020, NoisePickType::Convex, NoiseShape::Ball, 3.0f);
    skel.reset();
    skel.init(data, 0, borderVoxels, false);
  }

  Volume<float> importance = ComputeEstimatedEuclidean(skel.getEDT(), 2.0 * sqrt(2));

  return toByteVolume(importance, sqrtf(2) * 2);
}

inline void ExportReconstruction(std::string filename, collapseSkel3d& skel, std::shared_ptr<SkeletonModel> skeletonModel,
  Volume<float>& saliency, float imp)
{
  SkeletonReconstructor reconstructor(skeletonModel->GetSkelPoints(), skel.getEDT());
  Volume<byte> reconstruction = reconstructor.Reconstruct(ReconstructionSmoothingType::None, 0, &saliency, imp);
  reconstruction.exportVTKField(filename.c_str());
}

inline Volume<byte> ComputeSkeleton(Volume<byte> data, bool addNoise)
{
  collapseSkel3d skel(false, true);
  int borderVoxels;

  std::cout << "Computing Skeleton...." << std::endl;
  skel.init(data, 0, borderVoxels, false);


  // add noise, then reset skeleton
  if (addNoise)
  {
    std::shared_ptr<SkeletonModel> inputModel(new SkeletonModel(skel, 0.0f));

    surface::Graph graph;
    graph.construct(inputModel->GetThinPoints());

    NoiseCreator noiseCreator(data, graph, inputModel);
    noiseCreator.GeneratePoints(0.020, NoisePickType::Convex, NoiseShape::Ball, 3.0f);
    skel.reset();
    skel.init(data, 0, borderVoxels, false);
  }

  while (skel.collapse_iteration(1.0f));

  Volume<float> importance = skel.getImportance();

  return toByteVolume(importance);
}




enum class VelocityMethod
{
  GradientImportance = 0,
  GradientImportanceDirect,
  GradientEdt,
  Tpami,
  Feature,
};



inline const char* GetMethodName(VelocityMethod method)
{
  switch (method)
  {
  case VelocityMethod::GradientImportanceDirect:
    return "GradientImportanceDirect";
  case VelocityMethod::GradientImportance:
    return "GradientImportance";
  case VelocityMethod::GradientEdt:
    return "GradientEdt";
  case VelocityMethod::Tpami:
    return "Tpami";
  default:
    return "Feature";
  }
}

inline Volume<Vector> VelocityForMethod(VelocityMethod method, collapseSkel3d& skel,
  SaliencyMetricEngine& saliencyEngine, surface::Graph& graph, Volume<float>& importance)
{

  switch (method)
  {
  case VelocityMethod::GradientImportanceDirect:
  case VelocityMethod::GradientImportance:
    return saliencyEngine.BuildGradientImpVelocity(graph, importance);
  case VelocityMethod::GradientEdt:
    return saliencyEngine.BuildGradientEdtVelocity(graph);
  case VelocityMethod::Tpami:
    return skel.getV();
  default:
    return saliencyEngine.BuildFeatureVelocity(graph);
  }


}

inline Volume<float> getGeoDesicImportance(surface::Graph& graph, SaliencyMetricEngine& saliencyEngine, bool useEstimate, std::string filename)
{
  std::string filename_cache = filename.substr(0, filename.find_last_of(".")) + "_geocache.vtk";
  Volume<float> geoImportance;

  // Try to find the cached vtk file, otherwise compute and save it
  if (!geoImportance.readVTKField(filename_cache.c_str(), 0))
  {
    geoImportance = saliencyEngine.ComputeGeodesicRaw(graph, useEstimate);
    geoImportance.exportVTKField(filename_cache.c_str());
  }

  return geoImportance;
}

bool ExportVolumeToObj(std::string filename, Volume<byte>& volume);