#include <FeaturePoints.h>
#include <collapseSkel3d.h>
#include <memory>
#include <iostream>
#include <tuple>
#include <vector>
#include <boost/filesystem.hpp>

#include <iostream>
#include <fstream>
#include <sstream>

#include <SkeletonModel.h>
#include <surface/Graph.h>
#include <NoiseCreator.h>
#include <TestHelpers.h>
#include <SkeletonReconstructor.h>
#include "Stopwatch.h"

using namespace std;


int TestNoise(std::string filename, std::string outPath, float incidence, float radius, float size, float length, std::string postfix)
{
  Volume<byte> data;
  boost::filesystem::path path(filename);

  if (!data.readVTKField(filename.c_str(), 15))
  {
    std::cerr << "Loading " << filename << " failed" << std::endl;
    return false;
  }
  FilterLargestComponent(data, (byte)0);

  Volume<byte> original = data;
  Volume<byte> noisy = GetNoisyModel(data, incidence, radius, size, length);

  std::string outNoisyFilename = outPath + "Noisy_" + path.stem().string() + "_" + postfix + ".vtk";
  noisy.exportVTKField(outNoisyFilename.c_str());

}


int TestCompletePipeline(std::string filename, bool estimateGoedesic, float x, float glob_x, std::string outPath)
{
  Volume<byte> data;
  boost::filesystem::path path(filename);

  if (!data.readVTKField(filename.c_str(), 10))
  {
    std::cerr << "Loading " << filename << " failed" << std::endl;
    return false;
  }
  FilterLargestComponent(data, (byte) 0);

  Volume<byte> original = data;
  //Volume<byte> noisy = GetNoisyModel(data, 0.012, 2.1f, 4, 4); //the rest
  //Volume<byte> noisy = GetNoisyModel(data, 0.014, 2.1f, 4, 3); //bunny
  //Volume<byte> noisy = GetNoisyModel(data, 0.014, 2.1f, 3, 6);
  Volume<byte> noisy = GetNoisyModel(data, 0.014, 2.1f, 3, 1);
 // Volume<byte> noisy = original;
  Volume<byte> orgSkeleton = ComputeOrgSkeleton(original, false);
  

  collapseSkel3d skel(false, true);
  int borderVoxels;

  std::cout << "Computing Skeleton...." << std::endl;
  skel.init(noisy, 0, borderVoxels, false);
  Volume<byte> oldThin = skel.getThinImage();

  std::shared_ptr<SkeletonModel> inputModel(new SkeletonModel(skel, 0.0f));
  surface::Graph graph;
  graph.construct(inputModel->GetThinPoints());

  float noiseDiff = volumeDiff(original, noisy);
  cout << " noise diff " << noiseDiff << std::endl;

  //export noisy model
  skel.getSurface().ExportToObj(outPath + path.stem().string() + "_Export_Noisy" + ".obj");

  Stopwatch timer;
  timer.start("Complete");
  timer.start("Adapted IMA");
  Volume<float> newImportance = ComputeEstimatedEuclidean(skel.getEDT(), 2.0 * sqrt(2));
  timer.stop("Adapted IMA");

  skel.setImportance(newImportance);

  skel.set_ready(true);
  skel.getThinImage().clearVolume();


  timer.start("FeaturePoints");
  std::shared_ptr<SkeletonModel> skeletonModel(new SkeletonModel(skel, 0.0f));

  FeaturePoints featurePoints(inputModel->GetThinPoints());
  featurePoints.construct(skeletonModel->GetSkelPoints(), skel.getV(), skel.getImportance(), skel.getEDT());
  SaliencyMetricEngine saliencyEngine = constructSaliencyEngine(skel, featurePoints);
  timer.stop("FeaturePoints");

  Volume<Vector> newV = saliencyEngine.BuildGradientEdtVelocity(graph);
  skel.setV(newV);

  int totalPoints = getUniqueForegroundPoints(skel.getImportance(), orgSkeleton);
  int totalSkeletonPoints = getForegroundPoints(skel.getImportance());
  int totalOrgSkeletonPoints = getForegroundPoints(orgSkeleton);

  std::cout << "Calculating saliency Metric" << std::endl;
  timer.start("Geodesic");
  Volume<float> geoImportance = saliencyEngine.ComputeGeodesicRaw(graph, estimateGoedesic);
  timer.stop("Geodesic");
  timer.start("Geo Estimate");
  Volume<float> geoEstimate= saliencyEngine.ComputeGeodesicRaw(graph, true);
  timer.stop("Geo Estimate");

  Volume<float>& importanceVol = geoImportance;
  skel.setImportance(geoImportance);


  ofstream logfile("testpipeline_error_realgeo_new2.csv", ios::out | ios::app);
  ofstream reconstructionLog("reconstruction_errors_log3.csv", ios::out | ios::app);
  ofstream performanceLog("performance_log.csv", ios::out | ios::app);


  timer.start("Derivative");
  Volume<float> saliency = saliencyEngine.ComputeDerivative(skel.getV(), graph, 4.0f);
  timer.stop("Derivative");

  timer.start("Global");
  Volume<float> globalStreamlines = saliencyEngine.ComputeGlobalMonotonic(skel.getV(), graph, 4.0f);
  timer.stop("Global");

  timer.pause("Complete");
  std::string outNoisyFilename = outPath + "Noisy_" + path.stem().string() + ".vtk";
  ExportReconstructionGen(ReconstructionSmoothingType::None, outNoisyFilename, skel, skeletonModel, saliency, 0.00001f, 0);
  noisy.exportVTKField(outNoisyFilename.c_str());

  skeletonModel->Update(skel, x, saliency, true);
  Volume<byte> processedSkeleton = toByteVolume(saliency, x);
  Volume<byte> reconstructed = Reconstruct(skeletonModel->GetSkelPoints(), skel.getEDT(),
    ReconstructionSmoothingType::None, 3, &saliency, x, &graph, &featurePoints);

  Volume<byte> reconstructedSmooth = Reconstruct(skeletonModel->GetSkelPoints(), skel.getEDT(),
    ReconstructionSmoothingType::Linear, 3, &saliency, x, &graph, &featurePoints);


  float letError = (float)volumeLeftDiff(processedSkeleton, orgSkeleton) / totalSkeletonPoints;
  float rightError = (float)volumeLeftDiff(orgSkeleton, processedSkeleton) / totalOrgSkeletonPoints;
  float allError = (float)volumeDiff(orgSkeleton, processedSkeleton) / totalPoints;

  std::cout << "saliency: " << x << ", error: " << allError << std::endl;
  logfile << "\"" << filename << "\", " << ", " << x
    << ", " << letError << ", " << rightError << ", " << allError << std::endl;


  std::string outRecFilename = outPath + path.stem().string() +  ".vtk";
  std::string outGlobalRecFilename = outPath + "Recon_Global" + path.stem().string() + ".vtk";
  std::string outSkelFilename = outPath + "Skel_" + path.stem().string() + ".vtk";
  std::string outReconFilename = outPath + "Recon_" + path.stem().string() + ".vtk";

  std::string outMeanFilename = outPath + "Recon_Mean_" + path.stem().string() + ".vtk";
  std::string outMinFilename = outPath + "Recon_Min_" + path.stem().string() + ".vtk";
  std::string outGlobalMinFilename = outPath + "Recon_Global_Min_" + path.stem().string() + ".vtk";
  std::string outOpeningFilename = outPath + "Recon_Opening_" + path.stem().string() + ".vtk";
  std::string outMedianFilename = outPath + "Recon_Median_" + path.stem().string() + ".vtk";

  std::string outConstrainedMeanFilename = outPath + "Recon_Constrained_Mean_" + path.stem().string() + ".vtk";
  std::string outMinConstrainedFilename = outPath + "Recon_Constrained_Min_" + path.stem().string() + ".vtk";
  std::string outOpeningConstrainedFilename = outPath + "Recon_Constrained_Opening_" + path.stem().string() + ".vtk";
  std::string outOpeningHalfFilename = outPath + "Recon_Half_Constrained_Opening_" + path.stem().string() + ".vtk";
  std::string outConstrainedMedianFilename = outPath + "Recon_Constrained_Median_" + path.stem().string() + ".vtk";

  std::string outLeastSquaresFilename = outPath + "Recon_LeastSquares_" + path.stem().string() + ".vtk";
  std::string outFlatProjectionFilename = outPath + "Recon_FlatProjection_" + path.stem().string() + ".vtk";

  std::string outSalFilename = outPath + "Sal_" + path.stem().string() + ".vtk";
  std::string outGlobalFilename = outPath + "Global_Float_" + path.stem().string() + ".vtk";
  

  skeletonModel->Update(skel, x, saliency, true);
  
  timer.start("Reconstruct");
  ExportReconstructionGenLog(reconstructionLog, filename, original, ReconstructionSmoothingType::None, outReconFilename, skel, skeletonModel, saliency, x, 3);
  timer.stop("Reconstruct");
  ExportReconstructionGenLog(reconstructionLog, filename, original, ReconstructionSmoothingType::Linear, outMeanFilename, skel, skeletonModel, saliency, x, 3);
  timer.start("MinAndReconstruct");
  timer.start("Complete");
  ExportReconstructionGenLog(reconstructionLog, filename, original, ReconstructionSmoothingType::Min, outMinFilename, skel, skeletonModel, saliency, x, 3);
  timer.stop("MinAndReconstruct");
  timer.stop("Complete");
  ExportReconstructionGenLog(reconstructionLog, filename, original, ReconstructionSmoothingType::Opening, outOpeningFilename, skel, skeletonModel, saliency, x, 3);
  ExportReconstructionGenLog(reconstructionLog, filename, original, ReconstructionSmoothingType::Median, outMedianFilename, skel, skeletonModel, saliency, x, 3);
  ExportReconstructionGenLog(reconstructionLog, filename, original, ReconstructionSmoothingType::LeastSquares, 
    outLeastSquaresFilename, skel, skeletonModel, saliency, x, 3, false, &graph, &featurePoints);
  ExportReconstructionGenLog(reconstructionLog, filename, original, ReconstructionSmoothingType::FlatProjectionEdt,
    outFlatProjectionFilename, skel, skeletonModel, saliency, x, 3, false, &graph, &featurePoints);

  ExportReconstructionGenLog(reconstructionLog, filename, original, ReconstructionSmoothingType::Linear, outConstrainedMeanFilename, skel, skeletonModel, saliency, x, 3, true);
  ExportReconstructionGenLog(reconstructionLog, filename, original, ReconstructionSmoothingType::Min, outMinConstrainedFilename, skel, skeletonModel, saliency, x, 3, true);
  ExportReconstructionGenLog(reconstructionLog, filename, original, ReconstructionSmoothingType::Opening, outOpeningConstrainedFilename, skel, skeletonModel, saliency, x, 3, true);
  ExportReconstructionGenLog(reconstructionLog, filename, original, ReconstructionSmoothingType::HalfConstrainedOpening, outOpeningHalfFilename, skel, skeletonModel, saliency, x, 3, true);
  ExportReconstructionGenLog(reconstructionLog, filename, original, ReconstructionSmoothingType::Median, outConstrainedMedianFilename, skel, skeletonModel, saliency, x, 3, true);

  skeletonModel->Update(skel, glob_x, globalStreamlines, true);
  ExportReconstructionGenLog(reconstructionLog, filename, original, ReconstructionSmoothingType::Min, outGlobalMinFilename, skel, skeletonModel, globalStreamlines, glob_x, 3);
  ExportReconstructionGenLog(reconstructionLog, filename, original, ReconstructionSmoothingType::None, outGlobalRecFilename, skel, skeletonModel, globalStreamlines, glob_x, 3);

  processedSkeleton.exportVTKField(outSkelFilename.c_str());
  saliency.exportVTKField(outSalFilename.c_str());
  globalStreamlines.exportVTKField(outGlobalFilename.c_str());

  performanceLog << "\"" << filename << "\""
    << ", " << original.getWidth()
    << ", " << timer.get_last_time("Adapted IMA")
    << ", " << timer.get_last_time("FeaturePoints")
    << ", " << timer.get_last_time("Geo Estimate")
    << ", " << timer.get_last_time("Geodesic")
    << ", " << timer.get_last_time("Derivative")
    << ", " << timer.get_last_time("Global")
    << ", " << timer.get_last_time("MinAndReconstruct")
    << ", " << timer.get_last_time("Reconstruct")
    << std::endl;

  return 0;
}



int main(int argc, char *argv[])
{
  std::string filename;
  bool useEstimate = false;

  if (argc == 1)
  {
    std::cout << "Filename required as first argument." << std::endl;
    return -1;
  }
  else if (argc == 3)
  {
    useEstimate = strcmp(argv[2], "true") == 0;
  }
  filename = argv[1];

  return TestCompletePipeline(filename, useEstimate, 18.0f, 16.0f, "../../src/pipeline/");

  //TestNoise(filename, "../../src/noise/", 0.03, 2, 2, 1, "inc");
  //TestNoise(filename, "../../src/noise/", 0.003, 7, 1, 1, "radius");
  //TestNoise(filename, "../../src/noise/", 0.01, 3, 6, 1, "size");
  //TestNoise(filename, "../../src/noise/", 0.003, 2, 3, 20, "length");
}
