#include <FeaturePoints.h>
#include <collapseSkel3d.h>
#include <memory>
#include <iostream>
#include <tuple>
#include <vector>
#include <boost/filesystem.hpp>

#include <iostream>
#include <fstream>
#include <sstream>

#include <SkeletonModel.h>
#include <surface/Graph.h>
#include <NoiseCreator.h>
#include <TestHelpers.h>
#include <SkeletonReconstructor.h>
#include "Stopwatch.h"

using namespace std;


int TestDerrivative(std::string filename, bool estimateGoedesic, float der_t, float clas_t, std::string outPath)
{
  Volume<byte> data;
  boost::filesystem::path path(filename);

  if (!data.readVTKField(filename.c_str(), 10))
  {
    std::cerr << "Loading " << filename << " failed" << std::endl;
    return false;
  }
  FilterLargestComponent(data, (byte)0);

  Volume<byte> original = data;
  Volume<byte> orgSkeleton = ComputeOrgSkeleton(data, false);

  collapseSkel3d skel(false, true);
  int borderVoxels;

  std::cout << "Computing Skeleton...." << std::endl;
  skel.init(data, 0, borderVoxels, false);
  std::shared_ptr<SkeletonModel> inputModel(new SkeletonModel(skel, 0.0f));
  surface::Graph graph;
  graph.construct(inputModel->GetThinPoints());

  // add noise, then reset skeleton
  NoiseCreator noiseCreator(data, graph, inputModel);
  noiseCreator.GeneratePoints(0.020f, NoisePickType::Convex, NoiseShape::Ball, 2.0f, 3, 1);

  float noiseDiff = volumeDiff(original, data);

  // reprocess input model and graph after adding noise.
  graph.reset();
  inputModel->Update(skel, 0.3e-5f);
  graph.construct(inputModel->GetThinPoints());

  skel.reset();
  skel.init(data, 0, borderVoxels, false);

  Volume<float> newImportance = ComputeEstimatedEuclidean(skel.getEDT(), 2.0 * sqrt(2));

  skel.setImportance(newImportance);

  skel.set_ready(true);
  skel.getThinImage().clearVolume();


  std::shared_ptr<SkeletonModel> skeletonModel(new SkeletonModel(skel, 0.0f));


  FeaturePoints featurePoints(inputModel->GetThinPoints());
  featurePoints.construct(skeletonModel->GetSkelPoints(), skel.getV(), skel.getImportance(), skel.getEDT());
  SaliencyMetricEngine saliencyEngine = constructSaliencyEngine(skel, featurePoints);

  Volume<Vector> newV = saliencyEngine.BuildGradientEdtVelocity(graph);
  skel.setV(newV);

  int totalPoints = getUniqueForegroundPoints(skel.getImportance(), orgSkeleton);
  int totalSkeletonPoints = getForegroundPoints(skel.getImportance());
  int totalOrgSkeletonPoints = getForegroundPoints(orgSkeleton);

  std::cout << "Calculating Geodesic Metric" << std::endl;
  Volume<float> geoImportance = getGeoDesicImportance(graph, saliencyEngine, estimateGoedesic, filename);
  Volume<float>& importanceVol = geoImportance;
  skel.setImportance(geoImportance);


  ofstream logfile("testder_error_realgeo.csv", ios::out | ios::app);

  std::cout << "Computing derivative..." << std::endl;
  Volume<float> saliency = saliencyEngine.ComputeDerivative(skel.getV(), graph, 2.0f);
  Volume<float> saliencyClassical = saliencyEngine.Compute(0.01f);

  std::string outNoisyFilename = outPath + "Noisy_" + path.stem().string() + ".vtk";
  ExportReconstructionGen(ReconstructionSmoothingType::None, outNoisyFilename, skel, skeletonModel, saliency, 0.00001f, 0);

  skeletonModel->Update(skel, der_t, saliency, true);
  Volume<byte> processedSkeletonDer = toByteVolume(saliency, der_t);

  skeletonModel->Update(skel, clas_t, saliencyClassical, true);
  Volume<byte> processedSkeletonSal = toByteVolume(saliencyClassical, clas_t);


  float letError = (float)volumeLeftDiff(processedSkeletonDer, orgSkeleton) / totalSkeletonPoints;
  float rightError = (float)volumeLeftDiff(orgSkeleton, processedSkeletonDer) / totalOrgSkeletonPoints;
  float allError = (float)volumeDiff(orgSkeleton, processedSkeletonDer) / totalPoints;

  std::cout << "saliency: " << der_t << ", error: " << allError << std::endl;
  logfile << "\"" << filename << "\", " << ", " << der_t
    << ", " << letError << ", " << rightError << ", " << allError << std::endl;


  std::string outDerSkelFilename = outPath + "Der_Skel_" + path.stem().string() + ".vtk";
  std::string outDerReconFilename = outPath + "Der_Recon_" + path.stem().string() + ".vtk";
  std::string outSalSkelFilename = outPath + "Sal_Skel_" + path.stem().string() + ".vtk";
  std::string outSalReconFilename = outPath + "Sal_Recon_" + path.stem().string() + ".vtk";

  std::string outSalFilename = outPath + "Sal_float_" + path.stem().string() + ".vtk";
  std::string outDerFilename = outPath + "Der_float_" + path.stem().string() + ".vtk";

  skeletonModel->Update(skel, der_t, saliency, true);
  ExportReconstructionGen(ReconstructionSmoothingType::None, outDerReconFilename, skel, skeletonModel, saliency, der_t, 3);

  skeletonModel->Update(skel, clas_t, saliencyClassical, true);
  ExportReconstructionGen(ReconstructionSmoothingType::None, outSalReconFilename, skel, skeletonModel, saliencyClassical, clas_t, 3);

  processedSkeletonDer.exportVTKField(outDerSkelFilename.c_str());
  processedSkeletonSal.exportVTKField(outSalSkelFilename.c_str());

  saliency.exportVTKField(outDerFilename.c_str());
  saliencyClassical.exportVTKField(outSalFilename.c_str());

  return 0;
}



//int main(int argc, char *argv[])
//{
//  std::string filename;
//  bool useEstimate = false;
//
//  if (argc == 1)
//  {
//    std::cout << "Filename required as first argument." << std::endl;
//    return -1;
//  }
//  else if (argc == 3)
//  {
//    useEstimate = strcmp(argv[2], "true") == 0;
//  }
//  filename = argv[1];
//
//  return TestDerrivative(filename, useEstimate, 18.0f, 1.40f, "../../src/derivative/");
//}
