#include <FeaturePoints.h>
#include <collapseSkel3d.h>
#include <memory>
#include <iostream>
#include <tuple>
#include <vector>
#include <boost/filesystem.hpp>

#include <iostream>
#include <fstream>
#include <sstream>

#include <SkeletonModel.h>
#include <surface/Graph.h>
#include <NoiseCreator.h>
#include <TestHelpers.h>
#include <SkeletonReconstructor.h>
#include "Stopwatch.h"

using namespace std;




void logSkeletonError(ofstream& logfile, Volume<byte>& processedSkel, Volume<byte>& orgSkel,
  std::string filename, std::string method, float threshold, int totalSkeletonPoints,
  int totalOrgSkeletonPoints, int totalPoints)
{
  float letError = (float)volumeLeftDiff(processedSkel, orgSkel) / totalSkeletonPoints;
  float rightError = (float)volumeLeftDiff(orgSkel, processedSkel) / totalOrgSkeletonPoints;
  float allError = (float)volumeDiff(orgSkel, processedSkel) / totalPoints;

  std::cout << "method: " << method << ", threshold: " << threshold << ", error: " << allError << std::endl;

  logfile << "\"" << filename << "\", " << ", " << method << ", " << threshold
    << ", " << letError << ", " << rightError << ", " << allError << std::endl;

}

int TestStreamlines(std::string filename, bool estimateGoedesic, float der_t, std::string outPath)
{
  Volume<byte> data;
  boost::filesystem::path path(filename);

  if (!data.readVTKField(filename.c_str(), 10))
  {
    std::cerr << "Loading " << filename << " failed" << std::endl;
    return false;
  }
  FilterLargestComponent(data, (byte)0);

  Volume<byte> original = data;
  Volume<byte> orgSkeleton = ComputeOrgSkeleton(data, false);

  collapseSkel3d skel(false, true);
  int borderVoxels;

  std::cout << "Computing Skeleton...." << std::endl;
  skel.init(data, 0, borderVoxels, false);
  std::shared_ptr<SkeletonModel> inputModel(new SkeletonModel(skel, 0.0f));
  surface::Graph graph;
  graph.construct(inputModel->GetThinPoints());

  // add noise, then reset skeleton
  NoiseCreator noiseCreator(data, graph, inputModel);
  //noiseCreator.GeneratePoints(0.003f, NoisePickType::Convex, NoiseShape::Ball, 2.0f, 3, 30);
  noiseCreator.GeneratePoints(0.001f, NoisePickType::Convex, NoiseShape::Ball, 3.0f, 2, 50);

  float noiseDiff = volumeDiff(original, data);

  // reprocess input model and graph after adding noise.
  graph.reset();
  inputModel->Update(skel, 0.3e-5f);
  graph.construct(inputModel->GetThinPoints());

  skel.reset();
  skel.init(data, 0, borderVoxels, false);

  Volume<float> newImportance = ComputeEstimatedEuclidean(skel.getEDT(), 2.0 * sqrt(2));

  skel.setImportance(newImportance);

  skel.set_ready(true);
  skel.getThinImage().clearVolume();


  std::shared_ptr<SkeletonModel> skeletonModel(new SkeletonModel(skel, 0.0f));


  FeaturePoints featurePoints(inputModel->GetThinPoints());
  featurePoints.construct(skeletonModel->GetSkelPoints(), skel.getV(), skel.getImportance(), skel.getEDT());
  SaliencyMetricEngine saliencyEngine = constructSaliencyEngine(skel, featurePoints);

  Volume<Vector> newV = saliencyEngine.BuildGradientEdtVelocity(graph);
  skel.setV(newV);

  int totalPoints = getUniqueForegroundPoints(skel.getImportance(), orgSkeleton);
  int totalSkeletonPoints = getForegroundPoints(skel.getImportance());
  int totalOrgSkeletonPoints = getForegroundPoints(orgSkeleton);

  std::cout << "Calculating Geodesic Metric" << std::endl;
  Volume<float> geoImportance = getGeoDesicImportance(graph, saliencyEngine, estimateGoedesic, filename);
  Volume<float>& importanceVol = geoImportance;
  skel.setImportance(geoImportance);


  ofstream logfile("teststreamlines_error_realgeo.csv", ios::out | ios::app);

  std::cout << "Computing derivative..." << std::endl;
  Volume<float> derivative = saliencyEngine.ComputeDerivative(skel.getV(), graph, 0.5f);
  Volume<float> streamlines = saliencyEngine.ComputeStreamlines(skel.getV(), graph, 0.5f);
  Volume<float> globalStreamlines = saliencyEngine.ComputeGlobalMonotonic(skel.getV(), graph, 0.5f);

  std::string outNoisyFilename = outPath + "Noisy_" + path.stem().string() + ".vtk";
  ExportReconstructionGen(ReconstructionSmoothingType::None, outNoisyFilename, skel, skeletonModel, derivative, 0.00001f, 0);

  skeletonModel->Update(skel, der_t, derivative, true);
  Volume<byte> processedSkeletonDer = toByteVolume(derivative, der_t);

  skeletonModel->Update(skel, der_t, streamlines, true);
  Volume<byte> processedSkeletonStr = toByteVolume(streamlines, der_t);

  skeletonModel->Update(skel, der_t, globalStreamlines, true);
  Volume<byte> processedSkeletonGlo = toByteVolume(globalStreamlines, der_t);

  logSkeletonError(logfile, processedSkeletonDer, orgSkeleton, filename, "derivative",
    der_t, totalSkeletonPoints, totalOrgSkeletonPoints, totalPoints);
  logSkeletonError(logfile, processedSkeletonStr, orgSkeleton, filename, "streamlines",
    der_t, totalSkeletonPoints, totalOrgSkeletonPoints, totalPoints);
  logSkeletonError(logfile, processedSkeletonGlo, orgSkeleton, filename, "global",
    der_t, totalSkeletonPoints, totalOrgSkeletonPoints, totalPoints);

  std::string outDerSkelFilename = outPath + "Der_Skel_" + path.stem().string() + ".vtk";
  std::string outDerReconFilename = outPath + "Der_Recon_" + path.stem().string() + ".vtk";

  std::string outStrSkelFilename = outPath + "Str_Skel_" + path.stem().string() + ".vtk";
  std::string outStrReconFilename = outPath + "Str_Recon_" + path.stem().string() + ".vtk";

  std::string outGloSkelFilename = outPath + "Glo_Skel_" + path.stem().string() + ".vtk";
  std::string outGloReconFilename = outPath + "Glo_Recon_" + path.stem().string() + ".vtk";

  std::string outDerFilename = outPath + "Der_float_" + path.stem().string() + ".vtk";
  std::string outStrFilename = outPath + "Str_float_" + path.stem().string() + ".vtk";
  std::string outGloFilename = outPath + "Glo_float_" + path.stem().string() + ".vtk";

  std::string outDerFiltFilename = outPath + "Der_float_filt_" + path.stem().string() + ".vtk";
  std::string outStrFiltFilename = outPath + "Str_float_filt_" + path.stem().string() + ".vtk";
  std::string outGloFiltFilename = outPath + "Glo_float_filt_" + path.stem().string() + ".vtk";

  skeletonModel->Update(skel, der_t, derivative, true);
  ExportReconstructionGen(ReconstructionSmoothingType::None, outDerReconFilename, skel, skeletonModel, derivative, der_t, 3);

  skeletonModel->Update(skel, der_t, streamlines, true);
  ExportReconstructionGen(ReconstructionSmoothingType::None, outStrReconFilename, skel, skeletonModel, streamlines, der_t, 3);

  skeletonModel->Update(skel, der_t, globalStreamlines, true);
  ExportReconstructionGen(ReconstructionSmoothingType::None, outGloReconFilename, skel, skeletonModel, globalStreamlines, der_t, 3);

  processedSkeletonDer.exportVTKField(outDerSkelFilename.c_str());
  processedSkeletonStr.exportVTKField(outStrSkelFilename.c_str());
  processedSkeletonGlo.exportVTKField(outGloSkelFilename.c_str());

  derivative.exportVTKField(outDerFilename.c_str());
  streamlines.exportVTKField(outStrFilename.c_str());
  globalStreamlines.exportVTKField(outGloFilename.c_str());

  getFilteredImportance(derivative, der_t).exportVTKField(outDerFiltFilename.c_str());
  getFilteredImportance(streamlines, der_t).exportVTKField(outStrFiltFilename.c_str());
  getFilteredImportance(globalStreamlines, der_t).exportVTKField(outGloFiltFilename.c_str());

  return 0;
}

//int main(int argc, char *argv[])
//{
//  std::string filename;
//  bool useEstimate = false;
//
//  if (argc == 1)
//  {
//    std::cout << "Filename required as first argument." << std::endl;
//    return -1;
//  }
//  else if (argc == 3)
//  {
//    useEstimate = strcmp(argv[2], "true") == 0;
//  }
//  filename = argv[1];
//
//  return TestStreamlines(filename, useEstimate, 17.0f, "../../src/streamlines/");
//}
