#include <FeaturePoints.h>
#include <collapseSkel3d.h>
#include <memory>
#include <iostream>
#include <tuple>
#include <vector>
#include <boost/filesystem.hpp>

#include <iostream>
#include <fstream>
#include <sstream>
#include <algorithm>

#include <SkeletonModel.h>
#include <surface/Graph.h>
#include <NoiseCreator.h>
#include <TestHelpers.h>
#include <SkeletonReconstructor.h>
#include "Stopwatch.h"



Volume<float> getEdt(Volume<byte>& original)
{

  Volume<float> edt(original.getWidth(), original.getHeight(), original.getDepth());
  Volume<int> sqr_edt(original.getWidth(), original.getHeight(), original.getDepth());
  edt_3d(sqr_edt, original, 0);

  for (int i = 0; i < original.getWidth(); ++i)
    for (int j = 0; j < original.getHeight(); ++j)
      for (int k = 0; k < original.getDepth(); ++k)
      {
        if (sqr_edt(k, j, i))
        {
          float dst = sqrtf((float)sqr_edt(k, j, i));
          edt(k, j, i) = dst;
        }
      }


  return edt;
}

int TestNoisySkeletonVsNoisyEdt(std::string filename, bool estimateGoedesic, float der_t, std::string outPath)
{

  Volume<byte> data;
  boost::filesystem::path path(filename);

  if (!data.readVTKField(filename.c_str(), 10))
  {
    std::cerr << "Loading " << filename << " failed" << std::endl;
    return false;
  }
  FilterLargestComponent(data, (byte)0);

  Volume<byte> original = data;
  Volume<byte> noisy = GetNoisyModel(data, 0.014, 2.1f, 3, 6);
  Volume<byte> orgSkeleton = ComputeOrgSkeleton(original, false);
  Volume<float> orgEdt = GetEdt(original);

  collapseSkel3d skel(false, true);
  int borderVoxels;

  std::cout << "Computing Skeleton...." << std::endl;
  skel.init(noisy, 0, borderVoxels, false);
  Volume<byte> oldThin = skel.getThinImage();

  std::shared_ptr<SkeletonModel> inputModel(new SkeletonModel(skel, 0.0f));
  surface::Graph graph;
  graph.construct(inputModel->GetThinPoints());

  float noiseDiff = volumeDiff(original, noisy);
  std::cout << " noise diff " << noiseDiff << std::endl;

  //export noisy model
  skel.getSurface().ExportToObj(outPath + path.stem().string() + "_Export_Noisy" + ".obj");


  Volume<float> newImportance = ComputeEstimatedEuclidean(skel.getEDT(), 2.0 * sqrt(2));

  skel.setImportance(newImportance);

  skel.set_ready(true);
  skel.getThinImage().clearVolume();


  std::shared_ptr<SkeletonModel> skeletonModel(new SkeletonModel(skel, 0.0f));


  FeaturePoints featurePoints(inputModel->GetThinPoints());
  featurePoints.construct(skeletonModel->GetSkelPoints(), skel.getV(), skel.getImportance(), skel.getEDT());
  SaliencyMetricEngine saliencyEngine = constructSaliencyEngine(skel, featurePoints);

  Volume<Vector> newV = saliencyEngine.BuildGradientEdtVelocity(graph);
  skel.setV(newV);

  std::cout << "Calculating Geodesic Metric" << std::endl;
  Volume<float> geoImportance = getGeoDesicImportance(graph, saliencyEngine, estimateGoedesic, filename);
  Volume<float>& importanceVol = geoImportance;
  skel.setImportance(geoImportance);


  std::ofstream logfile("testedtvsskel_error_realgeo.csv", std::ios::out | std::ios::app);

  std::cout << "Computing derivative..." << std::endl;

  Volume<float> global = saliencyEngine.ComputeGlobalMonotonic(skel.getV(), graph, 5.0f);
  std::string outNoisyFilename = outPath + "Noisy_" + path.stem().string() + ".vtk";
  ExportReconstructionGen(ReconstructionSmoothingType::None, outNoisyFilename, skel, skeletonModel, global, 0.00001f, 0);
  

  // Reonstruct with regularized skeleton and original edt.
  std::string outNoisySkelFilename = outPath + "reg_org_edt_float_" + path.stem().string() + ".vtk";
  std::string outRegReconFilename = outPath + "reg_org_edt_recon_" + path.stem().string() + ".vtk";
  Volume<float> edtNoisy = skel.getEDT();
  skel.setEDT(orgEdt);
  skeletonModel->Update(skel, der_t, global, true);
  ExportReconstructionGen(ReconstructionSmoothingType::None, outRegReconFilename, 
    skel, skeletonModel, global, der_t, 0);
  global.exportVTKField(outNoisySkelFilename.c_str());



  // Reonstruct with original skeleton and noisy edt.
  std::string outOrgReconFilename = outPath + "org_noisy_edt_recon_" + path.stem().string() + ".vtk";
  std::string outOrgSkelFilename = outPath + "org_noisy_edt_float_" + path.stem().string() + ".vtk";
  skel.setEDT(edtNoisy);
  Volume<float> orgImportance = ComputeEstimatedEuclidean(orgEdt, 2.0 * sqrt(2));
  skeletonModel->Update(skel, 0.1f, orgImportance, true);
  ExportReconstructionGen(ReconstructionSmoothingType::None, outOrgReconFilename,
    skel, skeletonModel, orgImportance, 0.1f, 0);
  orgImportance.exportVTKField(outOrgSkelFilename.c_str());


  return 0;
}

//int main(int argc, char *argv[])
//{
//  std::string filename;
//  bool useEstimate = false;
//
//  if (argc == 1)
//  {
//    std::cout << "Filename required as first argument." << std::endl;
//    return -1;
//  }
//  else if (argc == 3)
//  {
//    useEstimate = strcmp(argv[2], "true") == 0;
//  }
//  filename = argv[1];
//
//  return TestNoisySkeletonVsNoisyEdt(filename, useEstimate, 18.f, "../../src/skelVsEdt/");
//}
