#version 330
#extension GL_ARB_shading_language_420pack : require

#define HOLE_FILLING 0

layout(binding = 0, std140) uniform ShaderData
{
    mat4 modelViewProj;
    mat4 modelView;
    mat4 inverseModelView;

    float projectScale;
    float near;
    vec2 unprojectScale;
    vec2 unprojectOffset;
    float depthScale;
    float depthOffset;

    vec3 lightPosition;
    float shininess;
    float specularCoeff;
    float diffuseCoeff;
    float ambientCoeff;

    float splatScale;
    float holeFillingFilterRadius;

    float minRoiWeight;
    float maxRoiWeight;
    bool visualizeRoiWeights;
    bool backfaceCulling;
};

layout(binding = 0) uniform sampler2D colorWeightSampler;
layout(binding = 1) uniform sampler2D normalLambdaSampler;

in vec2 texCoords;
out vec4 fragColor;

void fillHole(out vec3 color, out vec3 normal, out float lambda)
{
    vec2 screenPos = gl_FragCoord.xy;
    vec2 screenSize = vec2(textureSize(colorWeightSampler, 0));

    vec4 colorWeight = vec4(0.0);
    vec4 normalLambda = vec4(0.0);

    float fr = holeFillingFilterRadius;
    float frSq = fr * fr;

    for (float i = -fr; i <= fr; i += 1.0)
    {
        for (float j = -fr; j <= fr; j += 1.0)
        {
            if (i*i + j*j <= frSq)
            {
                vec2 coords = (screenPos + vec2(i, j)) / screenSize;
                colorWeight += texture(colorWeightSampler, coords);
                normalLambda += texture(normalLambdaSampler, coords);
            }
        }
    }

    float weight = colorWeight.a;
    color = colorWeight.rgb / weight;
    normal = normalize(normalLambda.xyz);
    lambda = normalLambda.a / weight;
}

void sampleAttributes(out vec3 color, out vec3 normal, out float lambda)
{
    vec4 colorWeight = texture(colorWeightSampler, texCoords);
    vec4 normalLambda = texture(normalLambdaSampler, texCoords);

    float weight = colorWeight.a;
    color = colorWeight.rgb / weight;
    normal = normalize(normalLambda.xyz);
    lambda = normalLambda.a / weight;
}

vec3 applyLighting(vec3 color, vec3 normal, vec3 viewDir, vec3 lightDir)
{
    vec3 reflectDir = reflect(-lightDir, normal);

    vec3 specular = vec3(specularCoeff * pow(abs(dot(reflectDir, viewDir)), shininess));
    vec3 diffuse = diffuseCoeff * color * abs(dot(normal, lightDir));
    vec3 ambient = ambientCoeff * color;

    return specular + diffuse + ambient;
}

void main()
{
    float lambda;
    vec3 color, normal;
    sampleAttributes(color, normal, lambda);

    if (lambda < 1.0E-6)
    {
#if HOLE_FILLING
        // There should also be an isHoleInsideModel check.
        fillHole(color, normal, lambda);
#else
        discard;
#endif
    }

    vec3 qn = vec3(gl_FragCoord.xy * unprojectScale - unprojectOffset, -near);
    vec3 q = lambda * qn;

    vec3 viewDir = -normalize(qn);
    vec3 lightDir = normalize(lightPosition - q);

    fragColor = vec4(applyLighting(color, normal, viewDir, lightDir), 1.0);
    gl_FragDepth = depthScale / q.z + depthOffset;
}
