#pragma once

#include <vector>
#include <GL/glew.h>
#include <glm/gtc/quaternion.hpp>
#include <glm/glm.hpp>
#include <memory>
#include <stdint.h>

#include "Shaders.h"
#include "Surfel.h"

struct VertexSpec
{
  unsigned int
    Array = 0u,
    Buffer = 0u;

  int Count = 0;
};

struct ShaderData
{
  glm::mat4 ModelViewProj; // 0N
  glm::mat4 ModelView; // 16N
  glm::mat4 InverseModelView; // 32N

  float ProjectScale; // 48N
  float Near; // 49N
  glm::vec2 UnprojectScale; // 50N
  glm::vec2 UnprojectOffset; // 52N
  float DepthScale; // 54N
  float DepthOffset; // 55N

  glm::vec3 LightPosition; // 56N
  float Shininess; // 57N
  float SpecularCoeff; // 58N
  float DiffuseCoeff; // 59N
  float AmbientCoeff; // 60N

  float SplatScale;  // 61N
  float HoleFillingFilterRadius; // 62N

  float MinRoiWeight;  // 63N
  float MaxRoiWeight; // 64N
  int32_t VisualizeRoiWeights; // 65N (booleans are 4 bytes)
  int32_t BackfaceCulling; // 66N (booleans are 4 bytes)
};

class PointSetRenderer
{
  VertexSpec pointSet, quad;
  ShaderProgram splattingProgram, shadingProgram, positionsProgram;

  unsigned int shaderDataBuffer = 0u, roiWeightsBuffer = 0u;
  int depthEpsilonLocation = -1;

  unsigned int
    framebuffer = 0u,
    depthBuffer = 0u,
    colorWeightTexture = 0u,
    normalLambdaTexture = 0u,
    positionsFramebuffer = 0u,
    positionsTexture = 0u;

  glm::ivec2 canvasSize;
  float
    depthEpsilon = 3.0f,
    splatScale = 2.0f,
    holeFillingFilterRadius = 0.f,
    diffuse = 0.9f,
    specular = 0.5f,
    shininess = 32.0f,
    cameraDistance = 5.f,
    minRoiWeight = 0.f,
    maxRoiWeight = 1.f;
  bool visualizeRoiWeights = false;
  bool backfaceCulling = false;
  glm::vec3 lightPosition;
  glm::quat rotation;

  std::unique_ptr<glm::vec3[]> modelPositions;
  void RetrieveModelPositions();

  void QueryUniforms();

  void LoadPointSet(const std::vector<Surfel> &surfels);
  void LoadRoiWeights(const std::vector<float> &roiWeights);
  void LoadQuad();

  ShaderData GetShaderData() const;
  void ReloadShaderData();
  void ResetFramebuffer();
  void ResetPositionsFramebuffer();

  void SplattingPass() const;
  void ShadingPass() const;

public:
  PointSetRenderer(const std::vector<Surfel> &surfels,  glm::ivec2 canvasSize);
  ~PointSetRenderer();

  void ApplyRotation(glm::quat rotationToApply);
  void MoveCamera(float cameraDistanceDelta);

  void ResizeCanvas(glm::ivec2 canvasSize);
  void Draw();

  std::vector<Surfel> GetPoints() const;
  void SetPoints(const std::vector<Surfel> &surfels);
  void SetRoiWeigths(const std::vector<float> &roiWeights);
  void UpdateRoiWeights(const std::vector<std::pair<size_t, float>> &updates, float roiWeight);

  glm::vec3 GetModelPosition(int x, int y);

  void SetSplatScale(const float value)
  {
    splatScale = value;
  }

  void SetBackfaceCulling(const bool value)
  {
    backfaceCulling = value;
  }


};
