#pragma once
#include <memory>
#include "SkeletonModel.h"
#include "PointSetRenderer.h"

class SkeletonPointRenderer
{

private:
  std::shared_ptr<SkeletonModel> inputModel;
  std::shared_ptr<SkeletonModel> skeletonModel;
  std::shared_ptr<SkeletonModel> reconstructionModel;

  std::unique_ptr<PointSetRenderer> thinInputModelRenderer;
  //std::unique_ptr<PointSetRenderer> thickInputModelRenderer;

  std::unique_ptr<PointSetRenderer> thinSkeletonModelRenderer;
  std::unique_ptr<PointSetRenderer> thickSkeletonModelRenderer;

  std::unique_ptr<PointSetRenderer> thinReconstructionRenderer;

  std::vector<PointSetRenderer*> pointSetRenderers;

  Volume<float>* scalarVolume;
  glm::ivec2 canvasSize;
  Point3d defaultColor;

  float scalarMin, scalarMax;
  float splatSize = 2.0f;
  bool backfaceCulling = false;


public:
  SkeletonPointRenderer(std::shared_ptr<SkeletonModel> inputModel,
    std::shared_ptr<SkeletonModel> skeletonModel,
    Volume<float>* scalarVolume,
    float scalarMin, float scalarMax,
    glm::ivec2 canvasSize) :
    inputModel(inputModel),
    skeletonModel(skeletonModel),
    scalarVolume(scalarVolume),
    canvasSize(canvasSize),
    defaultColor(66.f / 255, 133.f / 255, 244.f / 255),
    scalarMin(scalarMin),
    scalarMax(scalarMax)
  {
    auto thinInputSurfels = CreateSurfelList(inputModel->GetThinPoints(), inputModel->GetThinNormals(), defaultColor, nullptr, 0, 0);
    auto thinSkeletonModelSurfels = CreateSurfelList(skeletonModel->GetThinPoints(),
      skeletonModel->GetThinNormals(), defaultColor, scalarVolume, scalarMin, scalarMax);
    auto thickSkeletonModelSurfels = CreateSurfelList(skeletonModel->GetSkelPoints(),
      skeletonModel->GetSkelNormals(), defaultColor, scalarVolume, scalarMin, scalarMax);

    thinInputModelRenderer.reset(new PointSetRenderer(thinInputSurfels, canvasSize));
    thinSkeletonModelRenderer.reset(new PointSetRenderer(thinSkeletonModelSurfels, canvasSize));
    thickSkeletonModelRenderer.reset(new PointSetRenderer(thickSkeletonModelSurfels, canvasSize));

    pointSetRenderers = { thinInputModelRenderer.get(), 
      thinSkeletonModelRenderer.get(), thickSkeletonModelRenderer.get() };
  
  }

  void DrawInputModel();

  void DrawReconstructionModel();

  void DrawThinSkeletonPoints();

  void DrawThickSkeletonPoints();

  void UpdateSkeletonPoints();

  void UpdateInputModelPoints();

  void SetScalarVolume(Volume<float>* scalarVolume, float scalarMin, float scalarMax);

  std::shared_ptr<SkeletonModel> GetInputModel() const
  {
    return inputModel;
  }

  void SetInputModel(const std::shared_ptr<SkeletonModel>& skeletonModel)
  {
    inputModel = skeletonModel;
  }

  std::shared_ptr<SkeletonModel> GetSkeletonModel() const
  {
    return skeletonModel;
  }

  void SetSkeletonModel(const std::shared_ptr<SkeletonModel>& skeletonModel)
  {
    this->skeletonModel = skeletonModel;
  }

  std::shared_ptr<SkeletonModel> GetReconstructionModel() const
  {
    return reconstructionModel;
  }

  void SetReconstructionModel(const std::shared_ptr<SkeletonModel>& reconstructionModel);

  float GetSplatSize() const
  {
    return splatSize;
  }

  void SetSplatSize(float value)
  {
    this->splatSize = value;

    for (auto renderer : pointSetRenderers)
      renderer->SetSplatScale(value);
  }

  void SetBackfaceCulling(bool value)
  {
    //this->backfaceCulling;

    for (auto renderer : pointSetRenderers)
      renderer->SetBackfaceCulling(value);
  }

  void ResizeCanvas(glm::ivec2 canvasSize)
  {
    for (auto renderer : pointSetRenderers)
      renderer->ResizeCanvas(canvasSize);
  }

private:
};
