#include "Application.h"


#include <VolumeHelpers.h>
#include <iostream>
#include <boost/thread/barrier.hpp>
using namespace std;


Application::Application(int argc, char *argv[]) : 
     argc(argc), argv(argv),
     qtApp(this->argc, this->argv),
     glutWindow(GlutWindow::GetInstance()),
     mainWindow(),
     volumeOpenController(glutWindow)
{
}


bool Application::ParseCmdArgs()
{
  bool retValue = false;

  shared_ptr<boost::barrier> initBarrier(new boost::barrier(2));
  // glut must be used in its own thread to make it work with QT
  glutThread = thread([this, initBarrier, &retValue]()
  {
    retValue = InitFromCmd();
    // Signal that the initialization of glut is done
    initBarrier->wait();

    if(retValue)
      glutWindow.MainLoop();
  });

  // wait until the initialization is done
  initBarrier->wait();

  return retValue;
}

void Application::InitializeGlutWindow()
{
  glutInit(&argc, argv);
  glutWindow.Initialize(skel.get());
}

bool Application::OpenVolumeExternal(std::string filename, int threshold, int pad)
{
  Volume<byte> data;

  bool ok;
  if (strstr(filename.c_str(), ".fld"))
    ok = data.readAVSField(filename.c_str(), pad);
  else if (strstr(filename.c_str(), ".vtk"))
    ok = data.readVTKField(filename.c_str(), pad);
  else if (strstr(filename.c_str(), ".dat"))
  {
    int np = 0;
    int *points(nullptr);
    float *normals(nullptr);
    ok = data.readBinaryField(filename.c_str(), pad, np, points, normals);
    delete[] points;
    delete[] normals;
  }
  else
  {
    printf("Unknown file format.\n");
    return false;
  }
  if (!ok)
  {
    cout << "Cannot open " << filename << endl;
    return false;
  }

  cout << "Extracting largest component....";
  FilterLargestComponent(data);
  cout << " done" << endl;

  int wd = data.getWidth(), ht = data.getHeight(), dt = data.getDepth();
  maxDepth = 1.5 * sqrt(wd * wd + ht * ht + dt * dt);
  glutWindow.setMaxDepth(maxDepth);

  if (!skeletonController)
  {
    foreground = skel->init(data, threshold, bordervoxels);
    skeletonController.reset(new SkeletonController(*skel, 0.000001f, *ren));
  }
  else
  {
    skeletonController->ResetSkelEngine(data);
  }

  // todo: find out what these magic constants mean
  zprInit(0, 0, -(1.7f - 0.5f)*dt);


  glutWindow.SetSkeletonController(skeletonController.get());

  ren->setModels(skeletonController->GetSkeletonModel(), skeletonController->GetInputModel());
  ren->setFeaturePoints(&skeletonController->GetFeaturePoints());
  ren->reconstructionModel = std::shared_ptr<SkeletonModel>();

  return true;
}

bool Application::OpenVolume(std::string filename, int threshold, int pad)
{
  bool ret = volumeOpenController.OpenVolumeExternal(skel.get(), ren, filename, threshold, pad, 
    foreground, maxDepth, bordervoxels);
  skeletonController = volumeOpenController.getSkeletonController();
  mainWindow.SetSkeletonController(skeletonController.get());

  return ret;
}

int Application::Execute()
{
  mainWindow.show();
  return qtApp.exec();
}

// Largely copied from main.cc; can be cleaner
bool Application::InitFromCmd()
{
  bool advect_vel = argc >= 3 && !strcmp(argv[2], "-v");
  bool force_monotonic = true;
  skel.reset(new collapseSkel3d(advect_vel, force_monotonic));

  InitializeGlutWindow();
  mainWindow.SetGlutThread(&glutThread);
  mainWindow.SetVolumeOpenController(&volumeOpenController);
  ren = glutWindow.getRen();

  if (argc < 2)
  {
    printf("\nUsage: %s  volume_data_faile[.fld | .vtk]\n\n", argv[0]);
    return false;
  }

  bool fileOpenedSuccessfully = OpenVolume(std::string(argv[1]));
  if (!fileOpenedSuccessfully) 
    return false;

  mainWindow.connectGlutWindowSignals();

  return true;
}
