#include "GlutWindow.h"
#include "glwrapper.h"

#include <GL/freeglut.h>

std::unique_ptr<GlutWindow> GlutWindow::instance;
std::once_flag GlutWindow::onceFlag;

GlutWindow& GlutWindow::GetInstance()
{
  std::call_once(onceFlag,
    [] {
        instance.reset(new GlutWindow);
  });
  return *instance.get();
}

void GlutWindow::MainLoop()
{
  glutMainLoop();
}

void GlutWindow::Initialize(collapseSkel3d *skel)
{
  this->skel = skel;

  InitGlut();
}

void GlutWindow::SetSkeletonController(SkeletonController *value)
{
  this->skeletonController = value;
}


Renderer* GlutWindow::getRen() const
{
  return ren.get();
}
float GlutWindow::getMaxDepth() const
{
  return maxDepth;
}

void GlutWindow::setMaxDepth(float value)
{
  maxDepth = value;
}
int GlutWindow::getInputDrawStyle() const
{
  return inputDrawStyle;
}

void GlutWindow::setInputDrawStyle(int value)
{
  inputDrawStyle = value;
}
int GlutWindow::getSkeletonDrawStyle() const
{
  return skeletonDrawStyle;
}

void GlutWindow::setSkeletonDrawStyle(int value)
{
  skeletonDrawStyle = value;
}
int GlutWindow::getBoundaryDrawStyle() const
{
  return boundaryDrawStyle;
}

void GlutWindow::setBoundaryDrawStyle(int value)
{
  boundaryDrawStyle = value;
}
int GlutWindow::getSelectionFeaturePoints() const
{
  return selectionFeaturePoints;
}

void GlutWindow::setSelectionFeaturePoints(int value)
{
  selectionFeaturePoints = value;
}

bool GlutWindow::getShowStatistics() const
{
  return showStatistics;
}

void GlutWindow::setShowStatistics(bool value)
{
  showStatistics = value;
}
PickType GlutWindow::getPickMode() const
{
  return pickMode;
}

void GlutWindow::setPickMode(const PickType &value)
{
  pickMode = value;
}



GlutWindow::GlutWindow()
{
  for (int i = 0; i < 256; i++) buttons[i] = 0;

  buttons['t'] = true;
  buttons['i'] = true;
}

GlutWindow::~GlutWindow()
{

}

void GlutWindow::AddTask(Task task)
{
  scheduler.AddTask(task);
}

void GlutWindow::Close()
{
  glutLeaveMainLoop();
}

void GlutWindow::preDisplay()
{
  glPushMatrix();
  glViewport ( 0, 0, win_x, win_y );
  glMatrixMode ( GL_PROJECTION );
  glLoadIdentity ();
  gluPerspective(60.0, double(win_x) / win_y, 10.0f, maxDepth);
  glMatrixMode(GL_MODELVIEW);

  glClearColor ( 1.0f, 1.0f, 1.0f, 1.0f );
  glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

void GlutWindow::postDisplay()
{
  glPopMatrix();
  glutSwapBuffers();
}

void GlutWindow::InitGlut()
{
  // Freeglut extension: Don't call Exit() after exitMainLoop is called
  // This is important so that both QT and Glut can exit cleanly
  // Currently disabled as some deconstructors cause a segmentation fault 
  // (they were never tested because GLUT called exit() without freeing resources)

  glutSetOption(GLUT_ACTION_ON_WINDOW_CLOSE, GLUT_ACTION_GLUTMAINLOOP_RETURNS);

  glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE | GLUT_DEPTH);

  glutInitWindowSize(win_x, win_y);
  win_id = glutCreateWindow("Advection skeletons");

  glClearColor(1.0f, 1.0f, 1.0f, 1.0f);
  glClear(GL_COLOR_BUFFER_BIT);
  glutSwapBuffers();
  glClear(GL_COLOR_BUFFER_BIT);
  glutSwapBuffers();

  ren.reset(new Renderer(*skel, imp_step));
  ren->initGL();

  glShadeModel(GL_SMOOTH);
  glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);

  glEnable(GL_COLOR_MATERIAL);
  glShadeModel(GL_SMOOTH);
  glEnable(GL_AUTO_NORMAL);
  glEnable(GL_NORMALIZE);

  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LEQUAL);
  preDisplay();

  glutKeyboardFunc(glutKeyboardFuncWrapper);
  glutMouseFunc(glutMouseFuncWrapper);
  glutMotionFunc(glutMotionFuncWrapper);
  glutReshapeFunc(glutReshapeFuncWrapper);
  glutDisplayFunc(glutDisplayFuncWrapper);
  glutIdleFunc(glutIdleFuncWrapper);
  zprPickFunc(zprPickFuncWrapper);

}

void GlutWindow::glutKeyboardFuncHandler(unsigned char key, int x, int y)
{
  buttons[key] = !buttons[key];
  Refresh();
}



void GlutWindow::glutMouseFuncHandler(int button, int state, int x, int y)
{
  mouse_down[button] = state == GLUT_DOWN;
  int keys = glutGetModifiers();
  if (keys & GLUT_ACTIVE_CTRL) button = GLUT_MIDDLE_BUTTON;
  if (keys & GLUT_ACTIVE_SHIFT) button = GLUT_RIGHT_BUTTON;

  zprMouse(button, state, x, y);
}

void GlutWindow::glutMotionFuncHandler(int x, int y)
{
  zprMotion(x, y);
  glutPostRedisplay();
}

void GlutWindow::glutReshapeFuncHandler(int width, int height)
{
  glutSetWindow ( win_id );
  glutReshapeWindow ( width, height );
  win_x = width;
  win_y = height;
  ren->resizeWindow(width, height);
}

void GlutWindow::glutDisplayFuncHandler()
{
  preDisplay();

  const Volume<byte> &vol = skel->getThinImage();
  const int width = vol.getWidth(), height = vol.getHeight(),
            depth = vol.getDepth();

  glTranslatef(-width / 2, -height / 2, -1.7 * depth);

  //1. Draw input shape (if desired):
  switch (inputDrawStyle)
  {
  case DRAW_ISOSURFACE:
    ren->drawSurface();
    break;
  case DRAW_CUBES:
    ren->drawInputCubes();
    break;
  case DRAW_POINTS:
    ren->drawInputSplats();
    break;
  case DRAW_GRAPH:
    //graph_ren.draw(graph,ren->init_points);
    //LUK:
    //if(dist.size())
    //graph_ren.draw_dist(graph,ren->init_points, dist, geo_front_eps);
    break;
  }

  if (buttons['t'])               //2a. Draw skeleton - thin volume active:
  {
    switch (skeletonDrawStyle)
    {
    case DRAW_POINTS:
      ren->drawThinSplats();
      break;
    case DRAW_CUBES:
      ren->drawThinCubes();
      break;
    case DRAW_FIXED_BALLS:
      ren->drawThinBalls();
      break;
    }
  }

  if (buttons['i'])               //2b. Draw skeleton - importance volume active:
  {
    switch (skeletonDrawStyle)
    {
    case DRAW_POINTS:
      ren->drawSplats();
      break;
    case DRAW_CUBES:
      ren->drawCubes();
      break;
    case DRAW_BALLS:
      ren->drawBalls();
      break;
    case DRAW_FIXED_BALLS:
      ren->drawFixedBalls();
      break;
    }
  }


  switch (boundaryDrawStyle)        //3. Draw boundary (if desired)
  {
  case DRAW_CUBES:
    ren->drawBoundaryCubes();
    break;
  case DRAW_POINTS:
    ren->drawBoundarySplats();
    break;
  case DRAW_FIXED_BALLS:
    ren->drawBoundaryFixedBalls();
    //!!ren->drawProjectedSkelBoundary();
    break;
  }


  if (buttons['d'])
  {
    ren->drawDensity();
  }

  if (buttons['v'])
  {
    const Volume<Vector> &v = skel->getV();
    if (v.getDepth() == 0)
    {
      printf("Velocity field not initialized !\n");
    }
    else
    {
      ren->drawVelocity(1, 1.2f);
      //ren->drawThinGrad(1, 1.2f);
    }
  }

  skeletonController->getShowInfoHandler()->draw(*ren);

  skeletonController->getShowCorridorGraphHandler()->draw(*ren);

  if (showStatistics)
  {
    ren->drawLog();
  }

  zprPostDraw();

  postDisplay();

}

void GlutWindow::Refresh()
{
  scheduler.AddTask([]()
  {
    glutPostRedisplay();
  });
  
}

void GlutWindow::glutIdleFuncHandler()
{
  // Consume tasks in the task scheduler in this context.
  scheduler.ConsumeCurrentTasks();

  if (skeletonController->Update())
    glutPostRedisplay();
}

void GlutWindow::showInfoHandler(double x, double y, double z, bool picked_something)
{
  //No SS computed? Nothing to do
  if (!skeletonController->IsDone())
    return;
  skeletonController->getShowInfoHandler()->onPick(x, y, z, picked_something);
}

void GlutWindow::addNoisePickHandler(double x, double y, double z, bool picked_something)
{
   skeletonController->GetNoisePickHandler()->OnPick(x, y, z, picked_something);
}

void GlutWindow::showCorridorGraphHandler(double x, double y, double z, bool picked_something)
{
  //No SS computed? Nothing to do
  if (!skeletonController->IsDone())
    return;
  skeletonController->getShowCorridorGraphHandler()->onPick(x, y, z, picked_something);
}

// Todo move code to each respective classes.
void GlutWindow::zprPickFuncHandler(double x, double y, double z, bool picked_something)
{
  switch (pickMode)
  {
  case PickType::AddNoise:
    addNoisePickHandler(x, y, z, picked_something);
    break;
  case PickType::Info:
    showInfoHandler(x, y, z, picked_something);
    break;
  case PickType::CorridorGraph:
    showCorridorGraphHandler(x, y, z, picked_something);
    break;
  }
  // refresh display since something is probably added or changed.
  glutPostRedisplay();
}

// Static GLUT wrappers which calls the handlers on this singleton
void GlutWindow::glutKeyboardFuncWrapper(unsigned char key, int x, int y)
{
  GetInstance().glutKeyboardFuncHandler(key, x, y);
}

void GlutWindow::glutMouseFuncWrapper(int button, int state, int x, int y)
{
  GetInstance().glutMouseFuncHandler(button, state, x, y);
}

void GlutWindow::glutMotionFuncWrapper(int x, int y)
{
  GetInstance().glutMotionFuncHandler(x, y);
}

void GlutWindow::glutReshapeFuncWrapper(int width, int height)
{
  GetInstance().glutReshapeFuncHandler(width, height);
}

void GlutWindow::glutDisplayFuncWrapper()
{
  GetInstance().glutDisplayFuncHandler();
}

void GlutWindow::glutIdleFuncWrapper()
{
  GetInstance().glutIdleFuncHandler();
}

void GlutWindow::zprPickFuncWrapper(double x, double y, double z, bool picked_something)
{
  GetInstance().zprPickFuncHandler(x, y, z, picked_something);
}

