#include "MainWindow.h"
#include "GlutWindow.h"
#include "moc_MainWindow.cpp"
#include "AddNoisePickHandler.h"
#include "SaliencyMetricEngine.h"

#include <boost/thread.hpp>
#include <memory>
#include <math.h>
#include <QFileDialog>
#include <QMessageBox>
#include <QString>


template <typename F>
static void PostToMainThread(F && fun) {
  QObject signalSource;
  QObject::connect(&signalSource, &QObject::destroyed, qApp, std::move(fun));
}

MainWindow::MainWindow(QWidget *parent) :
QMainWindow(parent),
ui(new Ui::MainWindow)
{
  ui->setupUi(this);
  setupComboboxes();

  //todo: disable segmentation tab until skeleton is available
  //      requires signals to be added to the skeletoncontroller...
  //ui->segmentationTab->setEnabled(false);
}

MainWindow::~MainWindow()
{
  delete ui;
}

void MainWindow::connectGlutWindowSignals()
{
  //connect signals from the glutwindow
  connect(&*skeletonController->getShowInfoHandler(),
          &ShowInfoHandler::pickChanged,
          this,
          &MainWindow::on_pickChanged,
          Qt::QueuedConnection); //needed as it's a different thread

  connect(&*skeletonController->getShowInfoHandler(),
          &ShowInfoHandler::enableCutChanged,
          ui->InfoCutCheckBox,
          &QAbstractButton::setChecked,
          Qt::QueuedConnection); //needed as it's a different thread
  connect(ui->InfoCutCheckBox,
          &QAbstractButton::toggled,
          [this](bool value) {
            skeletonController->getShowInfoHandler()->setEnableCut(value);});
  //give button value of object
  ui->InfoCutCheckBox->setChecked(
        skeletonController->getShowInfoHandler()->getEnableCut());

  connect(&*skeletonController->getShowInfoHandler(),
          &ShowInfoHandler::enableCutSetChanged,
          ui->showCutSetCheckBox,
          &QAbstractButton::setChecked,
          Qt::QueuedConnection); //needed as it's a different thread
  connect(ui->showCutSetCheckBox,
          &QAbstractButton::toggled,
          [this](bool value) {
            skeletonController->getShowInfoHandler()->setEnableCutSet(value);});
  //give button value of object
  ui->showCutSetCheckBox->setChecked(
        skeletonController->getShowInfoHandler()->getEnableCutSet());


  connect(&*skeletonController->getShowInfoHandler(),
          &ShowInfoHandler::enableFeaturePointsChanged,
          ui->InfoFeaturePointsCheckBox,
          &QAbstractButton::setChecked,
          Qt::QueuedConnection); //needed as it's a different thread
  connect(ui->InfoFeaturePointsCheckBox,
          &QAbstractButton::toggled,
          [this](bool value) {
            skeletonController->getShowInfoHandler()->setEnableFeaturePoints(value);});
  //give button value of object
  ui->InfoFeaturePointsCheckBox->setChecked(
        skeletonController->getShowInfoHandler()->getEnableFeaturePoints());
}

void MainWindow::setupComboboxes()
{
  // Render combo boxes
  ui->inputRenderComboBox->addItem("Voxels", DRAW_CUBES);
  ui->inputRenderComboBox->addItem("2D splats", DRAW_POINTS);
  ui->inputRenderComboBox->addItem("Isosurface", DRAW_ISOSURFACE);
  ui->inputRenderComboBox->addItem("Connectivity", DRAW_GRAPH);
  ui->inputRenderComboBox->addItem("Nothing", DRAW_NOTHING);
  ui->inputRenderComboBox->setCurrentIndex(2); // DRAW_ISOSURFACE

  ui->skeletonRenderComboBox->addItem("Voxels", DRAW_CUBES);
  ui->skeletonRenderComboBox->addItem("2D splats", DRAW_POINTS);
  ui->skeletonRenderComboBox->addItem("Reconstruction (balls)", DRAW_BALLS);
  ui->skeletonRenderComboBox->addItem("Balls (fixed)", DRAW_FIXED_BALLS);
  ui->skeletonRenderComboBox->addItem("Nothing", DRAW_NOTHING);
  ui->skeletonRenderComboBox->setCurrentIndex(1); // DRAW_POINTS

  ui->boundaryRenderComboBox->addItem("Voxels", DRAW_CUBES);
  ui->boundaryRenderComboBox->addItem("Balls (fixed)", DRAW_FIXED_BALLS);
  ui->boundaryRenderComboBox->addItem("2D splats", DRAW_POINTS);
  ui->boundaryRenderComboBox->addItem("Nothing", DRAW_NOTHING);
  ui->boundaryRenderComboBox->setCurrentIndex(3); // DRAW_NOTHING

  ui->pickModeCombobox->addItem("Add Noise", static_cast<int>(PickType::AddNoise));
  ui->pickModeCombobox->addItem("Show Info", static_cast<int>(PickType::Info));
  ui->pickModeCombobox->addItem("Show Corridor Graph", static_cast<int>(PickType::CorridorGraph));
  ui->pickModeCombobox->setCurrentIndex(static_cast<int>(PickType::Info));

  ui->noiseShapeComboBox->addItem("Cube", static_cast<int>(NoiseShape::Cube));
  ui->noiseShapeComboBox->addItem("Sphere", static_cast<int>(NoiseShape::Ball));
  ui->noiseShapeComboBox->setCurrentIndex(
    ui->noiseShapeComboBox->findData(static_cast<int>(NoiseShape::Ball)));

  ui->noiseTypeComboBox->addItem("Convex", static_cast<int>(NoisePickType::Convex));
  ui->noiseTypeComboBox->addItem("Concave", static_cast<int>(NoisePickType::Concave));
  ui->noiseTypeComboBox->setCurrentIndex(
    ui->noiseTypeComboBox->findData(static_cast<int> (NoisePickType::Convex)));

  ui->cutMethodComboBox->addItem("3 shortest paths", static_cast<int>(cuts::CutController::Method::SHORTESTPATH));
  ui->cutMethodComboBox->addItem("plane intersection", static_cast<int>(cuts::CutController::Method::PLANETRACE));
  ui->cutMethodComboBox->addItem("perp to skel plane intersection", static_cast<int>(cuts::CutController::Method::PERPTOSKEL));
  ui->cutMethodComboBox->addItem("EDT triangle fit plane intersection", static_cast<int>(cuts::CutController::Method::EDTFIT));
  ui->cutMethodComboBox->setCurrentIndex(-1);

  ui->cutTransformMethodComboBox->addItem("Laplacian filtering", static_cast<int>(cuts::CutController::TransformMethod::LAPLACIANSMOOTHING));
  ui->cutTransformMethodComboBox->setCurrentIndex(-1);

  ui->saliencyMethodComboBox->addItem("Classical", static_cast<int>(SaliencyMethod::Classical));
  ui->saliencyMethodComboBox->addItem("Derrivative", static_cast<int>(SaliencyMethod::Derrivative));
  ui->saliencyMethodComboBox->addItem("Inverse Feature", static_cast<int>(SaliencyMethod::InvFeature));
  ui->saliencyMethodComboBox->addItem("Global Importance (streamlines)", static_cast<int>(SaliencyMethod::Streamline));
  ui->saliencyMethodComboBox->addItem("Global Importance (order)", static_cast<int>(SaliencyMethod::GlobalImportance));
  ui->saliencyMethodComboBox->setCurrentIndex(
    ui->saliencyMethodComboBox->findData(static_cast<int> (SaliencyMethod::GlobalImportance)));

  ui->scalarFieldComboBox->addItem("Importance", static_cast<int>(ScalarFieldType::Importance));
  ui->scalarFieldComboBox->addItem("Saliency", static_cast<int>(ScalarFieldType::Saliency));
  ui->scalarFieldComboBox->addItem("EDT", static_cast<int>(ScalarFieldType::EDT));
  ui->scalarFieldComboBox->setCurrentIndex(
    ui->scalarFieldComboBox->findData(static_cast<int> (ScalarFieldType::Importance)));

  ui->importanceMetricComboBox->addItem("USM", static_cast<int>(SkeletonImportanceType::TPAMI));
  ui->importanceMetricComboBox->addItem("Geodesic", static_cast<int>(SkeletonImportanceType::Geodesic));
  ui->importanceMetricComboBox->addItem("Geodesic (estimated)", static_cast<int>(SkeletonImportanceType::GeodesicEstimate));
  ui->importanceMetricComboBox->addItem("Implicit Euclidean", static_cast<int>(SkeletonImportanceType::ImplicitEuclidean));

  ui->extractionMethodComboBox->addItem("USM", static_cast<int>(SkeletonExtractionMethod::TPAMI));
  ui->extractionMethodComboBox->addItem("IMA", static_cast<int>(SkeletonExtractionMethod::IMA));
  ui->extractionMethodComboBox->addItem("Implicit Euclidean", static_cast<int>(SkeletonExtractionMethod::ImplicitEuclidean));

  ui->reconstructionSmoothingTypeComboBox->addItem("None", static_cast<int>(ReconstructionSmoothingType::None));
  ui->reconstructionSmoothingTypeComboBox->addItem("Mean", static_cast<int>(ReconstructionSmoothingType::Linear));
  ui->reconstructionSmoothingTypeComboBox->addItem("Median", static_cast<int>(ReconstructionSmoothingType::Median));
  ui->reconstructionSmoothingTypeComboBox->addItem("Min", static_cast<int>(ReconstructionSmoothingType::Min));
  ui->reconstructionSmoothingTypeComboBox->addItem("Least Squares", static_cast<int>(ReconstructionSmoothingType::LeastSquares));
  ui->reconstructionSmoothingTypeComboBox->addItem("Flat Projection (EDT)", static_cast<int>(ReconstructionSmoothingType::FlatProjectionEdt));
  ui->reconstructionSmoothingTypeComboBox->addItem("Flat Projection (Position)", static_cast<int>(ReconstructionSmoothingType::FlatProjectionPosition));
}


SkeletonCreationOptions MainWindow::getSkeletonCreationOptions()
{
  SkeletonCreationFlags flags = SkeletonCreationFlags::None;

  if (ui->smoothEdtCheckBox->isChecked()) flags |= SkeletonCreationFlags::SmoothEdt;
  if (ui->visualiseSkeletonCheckBox->isChecked()) flags |= SkeletonCreationFlags::VisualizeCreation;
  if (ui->importanceBoosting->isChecked()) flags |= SkeletonCreationFlags::ImportanceBoosting;
  //if (ui->replaceImportanceCheckBox->isChecked()) flags |= SkeletonCreationFlags::GeodesicImportance;
  if (ui->enableFpReflectionCheckBox->isChecked()) flags |= SkeletonCreationFlags::EnableFpReflection;

  SkeletonImportanceType importanceType = static_cast<SkeletonImportanceType>(ui->importanceMetricComboBox->currentIndex());
  SkeletonExtractionMethod extractionMethod = static_cast<SkeletonExtractionMethod>(ui->extractionMethodComboBox->currentIndex());

  return SkeletonCreationOptions(flags, importanceType, extractionMethod);
}

void MainWindow::on_skeletonizeButton_clicked()
{
  if (skeletonController)
  {

    SkeletonCreationOptions options = getSkeletonCreationOptions();

    GlutWindow::GetInstance().AddTask(
      [this, options]()
    {
      skeletonController->ResetSkelEngine(options);

      if ((bool)(options.flags & SkeletonCreationFlags::VisualizeCreation))
        skeletonController->StartCollapse(true);
      else
        skeletonController->ComputeSkeleton();
    });

    ui->pickModeCombobox->setCurrentIndex(static_cast<int>(PickType::Info));
  }


}

void MainWindow::on_setDefaultButton_clicked()
{
  ui->maxImportanceSlider->setValue(1000);
  on_maxImportanceSlider_valueChanged(1000);
  ui->importanceThresholdSlider->setValue(0);
  on_importanceThresholdSlider_sliderReleased();
}

void MainWindow::CloseWindow()
{
  std::cout << "Closing window and cleaning resources..." << std::endl;

  // We first close GLUT
  GlutWindow::GetInstance().AddTask(
    []() {
    GlutWindow::GetInstance().Close();
  });

  // Wait until GLUT is closed
  // Can throw exception if the thread is already closed
  try 
  {
    if (glutThread)
      glutThread->join();
  }
  catch (std::system_error &e) {}

}

void MainWindow::on_actionExit_triggered()
{
  this->close();
}

void MainWindow::closeEvent(QCloseEvent* event)
{
  CloseWindow();
}

void MainWindow::on_inputRenderComboBox_currentIndexChanged(int index)
{
  int drawStyle = ui->inputRenderComboBox->itemData(index).toInt();
  GlutWindow::GetInstance().setInputDrawStyle(drawStyle);
  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_skeletonRenderComboBox_currentIndexChanged(int index)
{
  int drawStyle = ui->skeletonRenderComboBox->itemData(index).toInt();
  GlutWindow::GetInstance().setSkeletonDrawStyle(drawStyle);
  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_boundaryRenderComboBox_currentIndexChanged(int index)
{
  int drawStyle = ui->boundaryRenderComboBox->itemData(index).toInt();
  GlutWindow::GetInstance().setBoundaryDrawStyle(drawStyle);
  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_splatSizeOffsetSpinBox_valueChanged(double value)
{
  GlutWindow::GetInstance().getRen()->splat_dt_offset = value;
  GlutWindow::GetInstance().getRen()->pointRenderer->SetSplatSize(value);
  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_ballSizeOffsetSpinBox_valueChanged(double value)
{
  GlutWindow::GetInstance().getRen()->ball_dt_offset = value;
  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_featurePointsCheckBox_stateChanged(int value)
{
  GlutWindow::GetInstance().setSelectionFeaturePoints(value);
  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_importanceThresholdSlider_sliderReleased()
{
  if (!skeletonController->GetSkel().IsReady())
    return;

  int position = ui->importanceThresholdSlider->value();
  // Update value on the thread of the renderer
  GlutWindow::GetInstance().AddTask(
    [position, this]() {
    float posNorm = (float)position / ui->importanceThresholdSlider->maximum();
    float impThr = skeletonController->GetEqualizedImportance(posNorm);

    Renderer* ren = GlutWindow::GetInstance().getRen();
    ren->imp_thr = impThr;

    printf("Importance threshold: %.7e\n", ren->imp_thr);
    skeletonController->UpdateImportance(ren->imp_thr);

  });

  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_minAlphaSlider_valueChanged(int position)
{
  // Update value on the thread of the renderer
  GlutWindow::GetInstance().AddTask(
    [position, this]() {
      Renderer* ren = GlutWindow::GetInstance().getRen();
      ren->imp_alpha_min = (float) position / 1000;
  });

  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_alphaSlopeSlider_valueChanged(int position)
{
  // Update value on the thread of the renderer
  GlutWindow::GetInstance().AddTask(
    [position, this]() {
      Renderer* ren = GlutWindow::GetInstance().getRen();
      ren->imp_alpha_slope = (float) position / 1000;
  });

  GlutWindow::GetInstance().Refresh();
}


void MainWindow::on_maxImportanceSlider_valueChanged(int position)
{
  if (!skeletonController->GetSkel().IsReady())
    return;


  // Update value on the thread of the renderer
  GlutWindow::GetInstance().AddTask(
    [position, this]() {
      Renderer* ren = GlutWindow::GetInstance().getRen();
      ren->vis_imp_max = (float)position / 1000;
      ren->setScalar(ren->scalars, ren->scalarMin, ren->vis_imp_max * ren->imp_max);
  });

  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_statsCheckBox_toggled(bool checked)
{
    GlutWindow::GetInstance().setShowStatistics(checked);
    GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_pickModeCombobox_currentIndexChanged(int index)
{
    GlutWindow::GetInstance().setPickMode((PickType) index);
}



void MainWindow::on_noiseTypeComboBox_currentIndexChanged(int index)
{
  if (!skeletonController) return;

  NoisePickType value = static_cast<NoisePickType> (ui->noiseTypeComboBox->itemData(index).toInt());
  skeletonController->GetNoisePickHandler()->setPickType(value);
}

void MainWindow::on_noiseShapeComboBox_currentIndexChanged(int index)
{
  if (!skeletonController) return;

  NoiseShape value = static_cast<NoiseShape> (ui->noiseShapeComboBox->itemData(index).toInt());
  skeletonController->GetNoisePickHandler()->setShape(value);
}

void MainWindow::on_noiseRadiusSpinbox_valueChanged(int value)
{
  if (!skeletonController) return;

  skeletonController->GetNoisePickHandler()->setRadius(value);
}

void MainWindow::on_showInputRadioButton_toggled(bool checked)
{
  Renderer* ren = GlutWindow::GetInstance().getRen();
  ren->drawReconstruction = false;

  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_showReconstructionRadioButton_toggled(bool checked)
{
  Renderer* ren = GlutWindow::GetInstance().getRen();
  ren->drawReconstruction = true;

  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_reconstructButton_clicked()
{
  if (!skeletonController->GetSkel().IsReady())
    return;

  ReconstructionSmoothingType smoothing = static_cast<ReconstructionSmoothingType> 
    (ui->reconstructionSmoothingTypeComboBox->currentIndex());
  int radius = ui->reconstructionSmoothingRadiusSpinBox->value();

  // Update value on the thread of the renderer
  GlutWindow::GetInstance().AddTask(
    [this, smoothing, radius]() {
    Renderer* ren = GlutWindow::GetInstance().getRen();
    skeletonController->Reconstruct(smoothing, radius);
    ren->setModels(skeletonController->GetSkeletonModel(), skeletonController->GetInputModel());

  });

  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_generatePointsButton_clicked()
{
  double probability = ui->probabilitySpinBox->value();
  GlutWindow::GetInstance().AddTask(
    [this, probability]() {
    skeletonController->GetNoisePickHandler()->generatePoints(probability);
  });

  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_computeSaliencyButton_clicked()
{
  GlutWindow::GetInstance().AddTask(
    [this]() {
    skeletonController->ComputeSaliency();
  });

}


void MainWindow::on_saliencyImportanceSpinbox_valueChanged(double threshold)
{
  ImportanceVolume* saliencyVolume = skeletonController->GetSaliencyVolume();
  if (!saliencyVolume)
    return;

  float step = (saliencyVolume->GetMaxImportance() - saliencyVolume->GetMinImportance()) / 1000;
  ui->saliencyImportanceSpinbox->setSingleStep(step);

  bool keepLargest = ui->keepLargestComponentCheckBox->checkState();
  GlutWindow::GetInstance().AddTask(
    [this, threshold, keepLargest]() {
    skeletonController->UpdateSaliency((float)threshold, keepLargest);
  });

  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_keepLargestComponentCheckBox_clicked(bool checked)
{
  float threshold = ui->saliencyImportanceSpinbox->value();
  on_saliencyImportanceSpinbox_valueChanged(threshold);
}

void MainWindow::on_actionSave_triggered()
{
  
  QString filename = QFileDialog::getSaveFileName(this, "Save Skeleton Volume as",
    QDir::currentPath(), "Volumes (*.vtk)");
  

  if (!filename.isNull() && !filename.isEmpty())
  {
    Volume<byte>& volume = skeletonController->GetVisibleVolume();

    std::string filenameConverted = filename.toLocal8Bit().constData();

    // Disabled as AVS export is completely broken
    // if (filename.endsWith(".fld"))
    //   volume.exportAVSField(filenameConverted.c_str());

    if (filename.endsWith("vtk"))
      volume.exportVTKField(filenameConverted.c_str());
  }

}

void MainWindow::on_pickChanged()
{
  ui->infoPlainTextEdit->setPlainText(QString::fromStdString(
        skeletonController->getShowInfoHandler()->getInfoString()));
}

void MainWindow::on_cutMethodComboBox_activated(int index)
{
  GlutWindow::GetInstance().AddTask(
    [this, index]() {
      skeletonController->getSegmentationHandler().createCutCreator(
            static_cast<cuts::CutController::Method>(index));
  });
}

void MainWindow::on_computeCutsButton_clicked()
{
  GlutWindow::GetInstance().AddTask(
    [this]() {
    skeletonController->getSegmentationHandler().computeCuts();
  });

}

void MainWindow::on_noiseHeightSpinBox_valueChanged(int height)
{
  if (!skeletonController) return;

  skeletonController->GetNoisePickHandler()->setNoiseHeight(height);
}



void MainWindow::on_saliencyThresholdSlider_sliderReleased()
{
  ImportanceVolume* saliencyVolume = skeletonController->GetSaliencyVolume();
  if(!saliencyVolume)
    return;

  bool keepLargest = ui->keepLargestComponentCheckBox->checkState();

  int position = ui->saliencyThresholdSlider->value();
  // Update value on the thread of the renderer
  GlutWindow::GetInstance().AddTask(
    [this, position, saliencyVolume, keepLargest]() {
    float posNorm = (float)position / ui->saliencyThresholdSlider->maximum();
    float impThr = saliencyVolume->GetEqualizedImportance(posNorm);
    ui->saliencyImportanceSpinbox->setValue(impThr);
    skeletonController->UpdateSaliency((float)impThr, keepLargest);
  });
}


void MainWindow::on_saliencyMethodComboBox_activated(int index)
{
  if (!skeletonController) return;

  SaliencyMethod value = static_cast<SaliencyMethod> (ui->saliencyMethodComboBox->itemData(index).toInt());
  skeletonController->SetSaliencyMethod(value);
}

void MainWindow::on_cutTransformMethodComboBox_activated(int index)
{
  GlutWindow::GetInstance().AddTask(
    [this, index]() {
      skeletonController->getSegmentationHandler().createCutTransform(
            static_cast<cuts::CutController::TransformMethod>(index));
  });
}

void MainWindow::on_applyCutTransformPushButton_clicked()
{
  GlutWindow::GetInstance().AddTask(
    [this]() {
    skeletonController->getSegmentationHandler().applyTransform();
  });
}

void MainWindow::on_splatBackfaceCullingCheckBox_toggled(bool checked)
{
  GlutWindow::GetInstance().getRen()->pointRenderer->SetBackfaceCulling(checked);
  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_scalarFieldComboBox_currentIndexChanged(int index)
{
  if (!skeletonController || !skeletonController->GetSkel().IsReady())
    return;

  ScalarFieldType value = static_cast<ScalarFieldType> (ui->scalarFieldComboBox->itemData(index).toInt());
  const int defaultIndex = 0;

  GlutWindow::GetInstance().AddTask(
    [this, value, defaultIndex]() {

    if (skeletonController->SetScalarField(value))
      return;

    // If not succeeded set to previous value:
    PostToMainThread([this, defaultIndex] {
      ui->scalarFieldComboBox->setCurrentIndex(defaultIndex);
    });
  });

  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_drawStreamlinesCheckBox_toggled(bool checked)
{
  skeletonController->getShowInfoHandler()->setEnableStreamlines(checked);
  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_noiseLengthSpinBox_valueChanged(int value)
{
    if (!skeletonController) return;

    skeletonController->GetNoisePickHandler()->setLength(value);
}

void MainWindow::on_actionImport_triggered()
{
  QString filename = QFileDialog::getOpenFileName(this, tr("Open File"),
    QString(),
    tr("Importance Volumes (*.vtk)"));

  Volume<float> newImportance;

  if (!newImportance.readVTKField(filename.toStdString().c_str()))
  {
    QMessageBox msgBox;
    msgBox.setText("Could not read importance volume.");
    msgBox.exec();
    return;
  }

  if (!skeletonController->GetSkel().IsReady())
  {
    QMessageBox msgBox;
    msgBox.setText("Skeleton must be extracted first, before importing the importance volume");
    msgBox.exec();
    return;
  }

  skeletonController->ImportImportanceVolume(newImportance);

  GlutWindow::GetInstance().AddTask( [this](){
    Renderer* renderer = GlutWindow::GetInstance().getRen();
    renderer->setScalar( (Volume<float>*) &skeletonController->GetSkel().getImportance(), renderer->imp_min, renderer->vis_imp_max);
  });
  GlutWindow::GetInstance().Refresh();
}

void MainWindow::on_actionOpen_triggered()
{
  QString filename = QFileDialog::getOpenFileName(this, tr("Open Volume"),
    QString(),
    tr("Volumes (*.vtk *.fld)"));



  GlutWindow::GetInstance().AddTask(
    [this, filename]() 
  {
    collapseSkel3d& skel = skeletonController->GetSkel();
    GlutWindow& glutWindow = GlutWindow::GetInstance();
    Renderer* renderer = GlutWindow::GetInstance().getRen();

    int foreground, bordervoxels;
    float maxDepth;

    bool ret = volumeOpenController->OpenVolumeExternal(&skel, renderer, filename.toStdString(), 0, 0,
      foreground, maxDepth, bordervoxels);
  });

  GlutWindow::GetInstance().Refresh();

}

void MainWindow::on_actionExport_Volume_triggered()
{

  QString filename = QFileDialog::getSaveFileName(this, "Save Importance as",
    QDir::currentPath(), "Voxel Field (*.vtk)");

  if (!filename.isNull() && !filename.isEmpty())
  {
    Volume<float>* volume = skeletonController->GetVisibleImportance();

    std::string filenameConverted = filename.toLocal8Bit().constData();

    if (volume)
    {
      Volume<float> copy(volume->getWidth(), volume->getHeight(), volume->getDepth());
      // Only export visible parts of the volume
      for (auto& p : skeletonController->GetSkeletonModel()->GetSkelPoints())
        copy(p.x, p.y, p.z) = (*volume)(p.x, p.y, p.z);

      copy.exportVTKField(filenameConverted.c_str());
    }
    else
    {
      QMessageBox msgBox;
      msgBox.setText("The importance volume is not ready yet");
      msgBox.exec();
    }

  }
}


void MainWindow::on_actionExport_Surface_triggered()
{
  QString filename = QFileDialog::getSaveFileName(this, "Save Surface as",
    QDir::currentPath(), "Meshes (*.obj)");


  if (!filename.isNull() && !filename.isEmpty())
  {
    Volume<byte>& volume = skeletonController->GetVisibleVolume();

    std::string filenameConverted = filename.toLocal8Bit().constData();

    if (filename.endsWith("obj"))
    {
      ExportVolumeToObj(filenameConverted, volume);
    }

  }
}
