#include "SegmentationHandler.h"

#include "SkeletonController.h"
#include "cuts/ShortestPathCutCreator.h"
#include "cuts/PlaneTraceCutCreator.h"
#include "cuts/TransformLaplacianSmooth.h"
#include "cuts/PerpToSkelCutCreator.h"
#include "cuts/EdtFitCutCreator.h"

SegmentationHandler::SegmentationHandler(SkeletonController &SkeletonController)
:
  r_skeletonController(SkeletonController)
{

}

void SegmentationHandler::createCutCreator(cuts::CutController::Method method)
{
  //we need a skeleton!!
  if (! r_skeletonController.IsDone()) return;

  r_skeletonController.getCutController().setCutSet(
        std::unique_ptr<cuts::CutSet>(new cuts::CutSet(
                                        r_skeletonController.GetFeaturePoints(),
                                        *r_skeletonController.GetSkeletonModel())));

  switch (method)
  {
    case cuts::CutController::Method::SHORTESTPATH:
      createShortestPathCutCreator();
      break;
    case cuts::CutController::Method::PLANETRACE:
      createPlaneTraceCutCreator();
      break;
    case cuts::CutController::Method::PERPTOSKEL:
      createPerpToSkelCutCreator();
      break;
    case cuts::CutController::Method::EDTFIT:
      createEdtFitCutCreator();
      break;
    }
}

void SegmentationHandler::computeCuts()
{
  if (! r_skeletonController.getCutController().hasCutCreator()) return;
  if (! r_skeletonController.getCutController().hasCutSet()) return;
  r_skeletonController.getCutController().compute();
}

void SegmentationHandler::createShortestPathCutCreator()
{
  //no further configuration options
  std::unique_ptr<cuts::CutCreator> tmp_cutCreator(
        new cuts::ShortestPathCutCreator(
          r_skeletonController.GetFeaturePoints(),
          r_skeletonController.GetGraph(),
          r_skeletonController.GetSkel()));
  r_skeletonController.getCutController().setCutCreator(std::move(tmp_cutCreator));
}

void SegmentationHandler::createPlaneTraceCutCreator()
{
  //pop up config dialog

  //create new cutcreator
  std::unique_ptr<cuts::CutCreator> tmp_cutCreator(
        new cuts::PlaneTraceCutCreator(
          r_skeletonController.GetFeaturePoints(),
          r_skeletonController.GetGraph(),
          r_skeletonController.GetSkel(),
          //skelmodel has uninitialized normals,
          //this works als long as we're working on the input model
          r_skeletonController.GetInputModel()->GetThinNormals()));
  r_skeletonController.getCutController().setCutCreator(std::move(tmp_cutCreator));
}

void SegmentationHandler::createPerpToSkelCutCreator()
{
  //pop up config dialog
  size_t bandSize = 4;
  //create new cutcreator
  std::unique_ptr<cuts::CutCreator> tmp_cutCreator(
        new cuts::PerpToSkelCutCreator(
          r_skeletonController.GetFeaturePoints(),
          r_skeletonController.GetGraph(),
          r_skeletonController.GetSkel(),
          //skelmodel has uninitialized normals,
          //this works als long as we're working on the input model
          r_skeletonController.GetInputModel()->GetThinNormals(),
          bandSize));
  r_skeletonController.getCutController().setCutCreator(std::move(tmp_cutCreator));
}

void SegmentationHandler::createEdtFitCutCreator()
{
  //pop up config dialog
  size_t bandSize = 4;
  //create new cutcreator
  std::unique_ptr<cuts::CutCreator> tmp_cutCreator(
        new cuts::EdtFitCutCreator(
          r_skeletonController.GetFeaturePoints(),
          r_skeletonController.GetGraph(),
          r_skeletonController.GetSkel(),
          //skelmodel has uninitialized normals,
          //this works als long as we're working on the input model
          r_skeletonController.GetInputModel()->GetThinNormals(),
          bandSize));
  r_skeletonController.getCutController().setCutCreator(std::move(tmp_cutCreator));
}

void SegmentationHandler::applyTransform()
{
  if (! r_skeletonController.getCutController().hasTransform()) return;
  if (! r_skeletonController.getCutController().hasCutSet()) return;
  r_skeletonController.getCutController().transform();
}

void SegmentationHandler::createCutTransform(cuts::CutController::TransformMethod method)
{
  //we need a cutset!!
  if (! r_skeletonController.getCutController().hasCutSet()) return;

  switch (method)
  {
    case cuts::CutController::TransformMethod::LAPLACIANSMOOTHING:
      createLaplacianSmoothingCutTransform();
      break;
    }
}

void SegmentationHandler::createLaplacianSmoothingCutTransform()
{
  std::cout << "Creating LaplacianSmoothingCutTransform\n";
  //pop up config dialog

  //create new cutFilter
  std::unique_ptr<cuts::Transform> tmp_cutTransform(
        new cuts::TransformLaplacianSmooth(
          r_skeletonController.GetFeaturePoints(),
          r_skeletonController.GetGraph()));
  r_skeletonController.getCutController().setTransform(std::move(tmp_cutTransform));
}
